(* Erstellen von Installationsdateien mit Versionskennung
   - Lese Versionsinfo aus einer Exe-Datei und ndere den Dateinamen einer
     Installationsdatei entsprechend.
   - Kopiere die Installationsdatei in beliebige Verzeichnisse, lsche dort
     Vorversionen ab EraseLevel (s.u.)
   - Modifiziere eine Textdatei <InstallFile>.ver mit Versionsbezeichnung und
     Download URL
   Aufruf:
     CopyVersionFile <ExeFile> <InstallFile> [Directory|@DirListFile] [/l:<n>] [/e:<n>] [</md5>]
     ExeFile : Exe-Datei mit Versionsangabe
     InstallFile : zu ndernder Dateiname, z.B.
         pb-setup.exe ==> pb-setup-4.1.0201.exe
     DirListFile : Zielverzeichnis oder Datei mit zeilenweiser Liste von Verzeichnissen,
         in die InstallFile kopiert werden soll
         Optional auch FTP-Kopie mit FtpCopy (im gleichen Verz. wie CopyVersionFile):
           #<FTP-Conf>
     Optionen:
     /l:n  n = 1,2,3,4 (Standard: 4)
         Zu benutzende Versionslevel
     /e:n  n = 0,1,2,3,4 (Standard: 1)
         lsche Vorversionen im Ziel ab Versionslevel (0 = alle, 4 = keine)
     /md5  MD5-Datei erzeugen und kopieren

    2015 Dr. J. Rathlev, D-24222 Schwentinental

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   Vers. 1 - Okt. 2007
         2 - Jul. 2008
         3 - Aug. 2011  : <Level> = 0 -> keine Version
         4 - Mrz. 2017  : MD5
   *)

program CopyVersionFile;

{$APPTYPE CONSOLE}

{$R *.res}

uses
  Winapi.Windows, System.SysUtils, ExtSysUtils, Hashes;

const
  Space = ' ';
  Tab = #9;

  VerExt = '.ver';
//  FtpCopy = 'FtpCopy.exe';
  FtpCopy = 'FtpRefresh.exe';
  Md5Ext  = 'md5';

  InfoNum = 12;
  InfoStr : array[1..InfoNum] of string = ('CompanyName','FileDescription','FileVersion',
          'InternalName','LegalCopyright','Comments','LegalTradeMarks','OriginalFileName',
          'ProductName','ProductVersion','PrivateBuild','SpecialBuild');

type
  TSystemID = (osWin32,osWin95,osWinNT);
  TSystemInfo = record
    SystemID          : TSystemID;
    MajorVersion,
    MinorVersion,
    Build             : integer;
    Info              : string[127];
    end;

{------------------------------------------------------------------}
(* fhrende Leerstellen und Tabs lschen *)
function RemSp (const S : string) : string;
var
  i : integer;
begin
  Result:=s; i:=1;
  while (i<=length(Result)) and ((Result[i]=Space) or (Result[i]=Tab)) do inc(i);
  delete(Result,1,pred(i));
  end;

{ --------------------------------------------------------------- }
(* fhrende Zeichen ergnzen *)
function AddChar (const S : string;
                  c       : char;
                  len     : integer) : string;
var
  i  : integer;
begin
  Result:=s;
  for i:=succ(length(Result)) to len do Result:=c+Result;
  end;

{ ---------------------------------------------------------------- }
// Routinen zur Auswertung einer Befehlszeile
// prfe, ob die ersten Zeichen einer Option mit dem Parameter bereinstimmen
function CompareOption (const Param,Option : string) : boolean;
begin
  Result:=AnsiLowercase(Param)=copy(Option,1,length(Param));
  end;

// Option vom Typ option:value einlesen
function ReadOptionValue (var Param : string; const Option : string) : boolean;
var
  i : integer;
begin
  Result:=false;
  i:=AnsiPos(':',Param);
  if i=0 then exit;
  if not CompareOption(copy(Param,1,i-1),Option) then exit;
  Delete(Param,1,i); Result:=true;
  end;

{ ------------------------------------------------------------------- }
function ReadNxtStr (var s   : String;
                     Del     : char) : string;
var
  i : integer;
begin
  if length(s)>0 then begin
    i:=pos (Del,s);
    if i=0 then i:=succ(length(s));
    ReadNxtStr:=copy(s,1,pred(i));
    delete(s,1,i);
    end
  else ReadNxtStr:='';
  end;

{ ------------------------------------------------------------------- }
(* Integer, Double oder String aus einem String s bis zum nchsten
   Komma lesen
   s wird um den verarbeiteten Teil gekrzt *)
function ReadNxtInt (var s   : String;
                     Del     : char;
                     Default : integer) : integer;
var
  n    : integer;
  i,ic : integer;
begin
  s:=RemSp(s); i:=pos (Del,s);
  if i=0 then i:=succ(length(s));
  val(copy(s,1,pred(i)),n,ic);
  if ic=0 then ReadNxtInt:=n
  else ReadNxtInt:=Default;
  delete(s,1,i);
  end;

{ ---------------------------------------------------------------- }
(* Integer-Zahl in String mit fhrenden Nullen umsetzen *)
function ZStrInt (x : int64;
                  n : integer) : string;
var
  s : string;
begin
  s:=IntToStr(abs(x));
  if x<0 then begin
    s:=AddChar(s,'0',n-1);
    s:='-'+s;
    end
  else s:=AddChar(s,'0',n);
  ZStrInt:=s;
  end;

{ --------------------------------------------------------------- }
(* Ausmaskieren der Erweiterung (ohne Punkt) *)
function GetExt (const name : string) : string;
var
  i,j : integer;
begin
  j:=length(Name);
  if j>0 then begin
    i:=j;
    while (i>0) and (Name[i]<>'.') do dec(i);
    if i=0 then Result:=''
    else Result:=copy(name,i+1,j-i);
    end
  else Result:='';
  end;

{ --------------------------------------------------------------- }
(* Dateinamenerweiterung entfernen (sucht letzten Punkt) *)
function DelExt (Name : string) : string;
var
  i,j : integer;
begin
  j:=length(Name);
  if j>0 then begin
    i:=j;
    while (i>0) and (Name[i]<>'.') do dec(i);
    if i>0 then delete (Name,i,j-i+1);
    end;
  Result:=Name;
  end;

{ --------------------------------------------------------------- }
(* ndern einer Dateinamenerweiterung (sucht letzten Punkt - JR 9.8.05) *)
function NewExt (Name,Ext  : string) : string;
begin
  Name:=DelExt(Name);
  if (length(Ext)>0) and (Ext[1]='.') then delete(Ext,1,1);
  if length(Ext)>0 then Result:=Name+'.'+Ext
  else Result:=Name;
  end;

{ ---------------------------------------------------------------- }
(* Suffix und Erweiterung zu Dateinamen hinzufgen *)
function AddNameSuffix (FName,Suffix,Ext : string) : string;
begin
  if (length(Ext)>0) and (Ext[1]<>'.') then Ext:='.'+Ext;
  Result:=DelExt(FName)+Suffix+Ext;
  end;

(* Suffix zu Dateinamen mit Extension hinzufgen *)
function InsertNameSuffix (FName,Suffix : string) : string;
begin
  Result:=AddNameSuffix(FName,Suffix,GetExt(FName));
  end;

{ ---------------------------------------------------------------- }
// execute command
procedure ExecuteCommand(const Command,Param : string);
var
  s : string;
  si : TStartupInfo;
  pi : TProcessInformation;
  dwExitCode : DWord;
begin
  WriteLn('-----------------------------------------------------');
  // Create process to start Program
  FillChar(si,sizeof(TStartupInfo),0);
  with si do begin
    cb:=sizeof(TStartupInfo);
    dwFlags:=STARTF_USESHOWWINDOW;
    wShowWindow:=SW_SHOWDEFAULT;
    end;
  s:='"'+Command+'"'+' '+Param;
  try
    if CreateProcess(nil,   // command line for application to exedute
      PChar(s),nil,               // Security
      nil,                        // Security
      false,                      // use InheritHandles
      NORMAL_PRIORITY_CLASS,      // Priority
      nil,                        // Environment
      nil,                        // directory
      si, pi) then begin
      // Wait until child process exits.
      WaitForSingleObject(pi.hProcess,INFINITE);
      // Checking the exit code
      GetExitCodeProcess(pi.hProcess,dwExitCode);
      if dwExitCode <> 0 then begin
        WriteLn(Format('"%s" returned error code: %d',[Command,dwExitCode]));
        end
      else WriteLn(Format('Calling of "%s" successful!',[Command]));
      end
    else begin
      WriteLn(Format('Calling of "%s" failed!',[Command]));
      WriteLn(' - ',SystemErrorMessage(GetLastError));
      end;
  finally
    CloseHandle(pi.hProcess);
    CloseHandle(pi.hThread);
    WriteLn('-----------------------------------------------------');
    end;
  end;

{ ---------------------------------------------------------------- }
(*  Get Version Info from File *)
function GetVersion (Filename : string; var Version : string) : boolean;
var
  t       : string;
  n,Len,i : DWORD;
  Buf     : PByte;
  Value   : PChar;
  vvar    : PWordArray;
begin
  Result:=false;
  if (Win32Platform=VER_PLATFORM_WIN32_NT) and (Win32MajorVersion>=5) then begin   // use for Windows 2000 and newer
    n := GetFileVersionInfoSize(PChar(Filename),n);
    if n > 0 then begin
      Buf:=AllocMem(n);
      GetFileVersionInfo(PChar(Filename),0,n,Buf);     // get buffer
      // get translation code
      VerQueryValue(Buf,PChar('\VarFileInfo\Translation'),pointer(vvar),Len);
      t:=IntToHex(vvar^[0],4)+IntToHex(vvar^[1],4);
      for i := 1 to InfoNum do begin
        if VerQueryValue(Buf, PChar('\StringFileInfo\'+t+'\'+InfoStr[i]), Pointer(Value), Len) then begin
          if i=3 then Version:=value;
          end;
        end;
      FreeMem(Buf,n);
      Result:=true;
      end;
    end;
  end;

function ReadDestList (FName : string) : string;
var
  f : TextFile;
  s : string;
begin
  Result:='';
  if FileExists(FName) then begin
    AssignFile(f,FName); Reset(f);
    while not Eof(f) do begin
      readln(f,s);
      s:=Trim(ReadNxtStr(s,';'));  // bis Kommentar ";" lesen
      if length(s)>0 then Result:=Result+'|'+s
      end;
    CloseFile(f);
    if length(Result)>0 then Delete(Result,1,1);
    end
  end;

{ ------------------------------------------------------------------- }
(* Datei kopieren mit Attribut und Datum - verwendet CopyFile aus WindowsAPI *)
function FileCopy (srcfilename,destfilename : String) : boolean;
begin
  Result:=false;
  if FileExists (srcfilename) and (length(destfilename)>0) then begin
    if FileExists(destfilename) then DeleteFile(destfilename);
    Result:=CopyFile(pchar(srcfilename),pchar(destfilename),false);
    end;
  end;

procedure DeleteMatchingFiles(APath,AMask : string);
var
  DirInfo    : TSearchRec;
  Findresult : integer;
begin
  FindResult:=FindFirst (APath+AMask,faAnyFile,DirInfo);
  while (FindResult=0) do begin
    DeleteFile(APath+DirInfo.Name);
    FindResult:=FindNext(DirInfo)
    end;
  FindClose(DirInfo);
  end;

  function StringToTextFile (const OutName,AString : string) : boolean;
  var
    oText : TextFile;
  begin
    Result:=false;
    try
      AssignFile(oText,OutName); rewrite(oText);
      writeln (oText,AString);
      Result:=true;
    finally
      CloseFile(oText);
      end;
    end;

{ ---------------------------------------------------------------- }
var
  s,sn,sf,t,
  sm,sc,vl,v  : string;
  n,m,i       : integer;
  mmd5        : boolean;
begin
  if ParamCount>1 then begin
    n:=4; m:=1; vl:=''; mmd5:=false;
    if ParamCount>2 then for i:=3 to ParamCount do begin
      s:=ParamStr(i);
      if s[1]='/' then begin
        Delete(s,1,1);
        if ReadOptionValue(s,'level') then n:=ReadNxtInt(s,'|',4)
        else if ReadOptionValue(s,'erase') then m:=ReadNxtInt(s,'|',1)
        else if CompareOption(s,'md5') then mmd5:=true;
        end
      else begin
        if s[1]='@' then begin
          delete(s,1,1);
          vl:=ReadDestList(s);
          if ParamCount>3 then s:=ParamStr(4) else s:='';
          end
        else vl:=s;
        end;
      end;
    if FileExists(ParamStr(2)) then begin
      if FileExists(ParamStr(1)) and GetVersion(ParamStr(1),v) then begin
        t:=v;
        sn:=IntToStr(ReadNxtInt(t,'.',0));
        if n>1 then sn:=sn+'.'+IntToStr(ReadNxtInt(t,'.',0));
        if n>2 then sn:=sn+'.'+ZStrInt(ReadNxtInt(t,'.',0),2);
        if n>3 then sn:=sn+ZStrInt(ReadNxtInt(t,'.',0),2);
        t:=v;
        if m=0 then sf:=''
        else begin
          sf:=IntToStr(ReadNxtInt(t,'.',0));
          if m>1 then sf:=sf+'.'+IntToStr(ReadNxtInt(t,'.',0));
          if m>2 then sf:=sf+'.'+ZStrInt(ReadNxtInt(t,'.',0),2);
          if m>3 then sf:=sf+ZStrInt(ReadNxtInt(t,'.',0),2);
          end;
        if n>0 then begin
          sn:=InsertNameSuffix(ParamStr(2),'-'+sn);
          if FileExists(ParamStr(2)) then begin
            DeleteFile(sn);
            RenameFile(ParamStr(2),sn);
            end;
          end
        else sn:=ParamStr(2);
        if (length(vl)>0) then begin  // Datei kopieren
          t:=InsertNameSuffix(ParamStr(2),'-'+sf+'*');
          if mmd5 then begin
            sm:=NewExt(sn,Md5Ext);
            s:=HashFromFile(htMD5,sn);
            StringToTextFile(sm,s);
            end;
          sc:=ExtractFilePath(ParamStr(0))+FtpCopy;
          repeat
            s:=ReadNxtStr(vl,'|');
            if (length(s)>0) and (s[1]='#') then begin   // FTP
              if FileExists(sc) then begin
                delete(s,1,1);
                ExecuteCommand(sc,'/file '+s+' '+sn+' '+sm);
                end;
              end
            else begin  // copy
              s:=IncludeTrailingPathDelimiter(s);
              if m<4 then DeleteMatchingFiles(s,t);
              if FileCopy(sn,s+ExtractFileName(sn)) then writeln('Copied: ',sn,' to ',s)
              else writeln('Error copying: ',sn,' to ',s);
              if mmd5 then begin
                if FileCopy(sm,s+ExtractFileName(sm)) then writeln('Copied: ',sm,' to ',s)
                else writeln('Error copying: ',sm,' to ',s);
                end;
              end;
            until length(vl)=0;
          end;
        end
      else writeln('File not found or no version info: ',ParamStr(1));
      end
    else writeln('File not found: ',ParamStr(2));
    end
  else begin
    writeln ('Calling:');
    writeln ('  CopyVersionFile <Exe> <Install> [Dir|@DirList] [/l:<Level>] [/e:<EraseLevel>] [/md5]');
    end;
{$IFDEF Debug}
  write ('Strike enter key to continue ...');
  readln;
{$EndIf}
end.
