(* Common definitions and routines to copy files to FTP server
   (FtpCopy and FtpRefresh)

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   last modified: April 2025
   *)

unit FtpCopyCommon;


interface

uses System.Classes, System.IniFiles, FtpUtils;

const
  MaxUserPwdLength = 64;

  // SSL-Zertifikate
  CertConfig = 'certfiles.cfg';
//  RootCertFile = 'xyzroot.pem';
//  CertFile     = 'xyz.crt';
//  KeyFile      = 'xyz.key';
//  SSLPassword  = 'abc';

  iniTName  = 'TaskName';

type
  TCopySettings = class(TObject)
    TaskName,
    Source,
    Prefix,
    Exclude     : string;
    FtpPar      : TFtpParams;
    SFiles      : TStringList;
    OvWrMode    : integer;
    Rename      : boolean;
    NewName     : string;
    constructor Create(AName : string);
    constructor CreateFrom(AName : string; ACs : TCopySettings);
    destructor Destroy; override;
    end;

procedure ReadCopySettings (IniFile : TCustomIniFile; const Section : string;
                            CopySettings : TCopySettings);
procedure WriteCopySettings (IniFile : TCustomIniFile; const Section : string;
                            CopySettings : TCopySettings);

implementation

uses IdFtp, NumberUtils, StringUtils, Crypt;

{ ------------------------------------------------------------------- }
constructor TCopySettings.Create(AName : string);
begin
  inherited Create;
  SFiles:=TStringList.Create;
  TaskName:=AName;
  Source:=''; Exclude:=''; Prefix:=''; NewName:=''; Rename:=false; OvWrMode:=1;
  with FtpPar do begin
    Host:='';
    Port:=defFtpPort;
    Username:='';
    Password:='';
    Directory:='';
    SecureMode:=0;
    ConnTimeout:=defTimeOut;
    ConnAttempts:=defConnect;
    Passive:=true;
    UseHost:=false;
    WriteLog:=false;
    ForceUtf8:=false;
    CaseMode:=tcNone;
    Quotes:=false;
    InvalidChars:='';
    with Proxy do begin
      Server:='';
      Username:='';
      Password:='';
      Port:=defProxyPort;
      Mode:=fpcmNone;
      end;
    end;
  end;

constructor TCopySettings.CreateFrom(AName : string; ACs : TCopySettings);
begin
  Create(AName);
  Source:=ACs.Source; Prefix:=ACs.Prefix; Exclude:=ACs.Exclude;
  NewName:=ACs.NewName; FtpPar:=ACs.FtpPar;
  SFiles.Assign(ACs.SFiles);
  Rename:=ACs.Rename; OvWrMode:=ACs.OvWrMode;
  end;

destructor TCopySettings.Destroy;
begin
  SFiles.Free;
  inherited Destroy;
  end;

{ ------------------------------------------------------------------- }
const
// key to encode passwords in the ini file
{$I ..\..\Common\CopyPwdKey.inc }
// The include file contains the following declaration:
// PwdKey = '<keystring>';
// <keystring> can be created using the routine "EncodeString" in unit "Crypt"
// from any user defined string
// Use the enclosed program "EncPwd" to create your own key.
// Example: SampleKey => AA83CEA34295F249B384F15046A5D42D6625E80FCE189B4FEF


  iniSDir   = 'SourceDir';
  iniPref   = 'Prefix';
  iniExcl   = 'Exclude';
  iniFCount = 'FileCount';
  iniSFile  = 'File';
  iniPort   = 'Port';
  iniHost   = 'Host';
  iniDir    = 'Directory';
  iniSecure = 'UseFTPS';
  iniUseHost = 'UseHost';
  iniUtf8    = 'ForceUtf8';
  iniUser    = 'Username';
  iniPwd     = 'Password';
  iniPassive = 'Passive';
  iniCMode   = 'ChangeNames';
  IniPrxServ = 'ProxyServer';
  IniPrxPort = 'ProxyPort';
  IniPrxUser = 'ProxyUsername';
  IniPrxPwd  = 'ProxyPassword';
  IniPrxType = 'ProxyType';
  IniTimeSt  = 'CheckTimeStamp';
  IniOvrMd   = 'Overwrite';
  IniRename  = 'Rename';
  iniDest    = 'DestFile';

procedure ReadCopySettings (IniFile : TCustomIniFile; const Section : string;
                            CopySettings : TCopySettings);
var
  j,m : integer;
  s   : string;
begin
  with IniFile,CopySettings do begin
    Source:=ReadString(Section,iniSDir,'');
    Prefix:=ReadString(Section,iniPref,'');
    Exclude:=ReadString(Section,iniExcl,'');
    m:=ReadInteger(Section,iniFCount,0);
    for j:=0 to m-1 do begin
      s:=ReadString(Section,iniSFile+ZStrint(j,3),'');
      if length(s)>0 then SFiles.Add(s);
      end;
    with FtpPar do begin
      Host:=ReadString(Section,iniHost,'');
      Port:=ReadInteger(Section,iniPort,defFtpPort);
      Username:=ReadString(Section,iniUser,'');
      Password:=DecryptPwdString(DecodeString(PwdKey),ReadString(Section,iniPwd,''),teAscii85);
      Directory:=ReadString(Section,iniDir,'');
      SecureMode:=ReadInteger(Section,iniSecure,0);
      Passive:=ReadBool(Section,iniPassive,false);
      UseHost:=ReadBool(Section,iniUseHost,false);
      ForceUtf8:=ReadBool(Section,iniUtf8,false);
      CaseMode:=TTextChange(ReadInteger (Section,iniCMode,integer(tcNone)));
      if (length(Host)>0) and (length(Username)>0) then
      with Proxy do begin
        Server:=ReadString(Section,iniPrxServ,'');
        Port:=ReadInteger(Section,iniPrxPort,defProxyPort);
        Username:=ReadString(Section,iniPrxUser,'');
        Password:=DecryptPwdString(DecodeString(PwdKey),ReadString(Section,IniPrxPwd,''),teAscii85);
        Mode:=TIdFtpProxyType(ReadInteger(Section,IniPrxType,0));
        end;
      end;
    if ValueExists(Section,iniTimeSt) then begin
      if ReadBool(Section,iniTimeSt,true) then OvWrMode:=1
      else OvWrMode:=0;
      DeleteKey(Section,iniTimeSt);
      end
    else OvWrMode:=ReadInteger(Section,IniOvrMd,1);
    Rename:=ReadBool(Section,iniRename,false);
    NewName:=ReadString(Section,IniDest,'');
    end;
  end;

procedure WriteCopySettings (IniFile : TCustomIniFile; const Section : string;
                            CopySettings : TCopySettings);
var
  j : integer;
begin
  with IniFile,CopySettings do begin
    WriteString(Section,iniTName,TaskName);
    WriteString(Section,iniSDir,Source);
    WriteString(Section,iniPref,Prefix);
    WriteString(Section,iniExcl,Exclude);
    WriteInteger(Section,iniFCount,SFiles.Count);
    with SFiles do for j:=0 to Count-1 do begin
      WriteString(Section,iniSFile+ZStrint(j,3),Strings[j]);
      end;
    with FtpPar do begin
      WriteString(Section,iniHost,Host);
      WriteInteger(Section,iniPort,Port);
      WriteString(Section,iniUser,Username);
      WriteString(Section,iniPwd,EncryptPwdString(DecodeString(PwdKey),Password,MaxUserPwdLength,teAscii85));
      WriteString(Section,iniDir,Directory);
      WriteInteger(Section,iniSecure,SecureMode);
      WriteBool(Section,iniPassive,Passive);
      WriteBool(Section,iniUseHost,UseHost);
      WriteBool(Section,iniUtf8,ForceUtf8);
      WriteInteger(Section,iniCMode,integer(CaseMode));
      with Proxy do begin
        WriteString(Section,iniPrxServ,Server);
        WriteInteger(Section,iniPrxPort,Port);
        WriteString(Section,iniPrxUser,Username);
        WriteString(Section,iniPrxPwd,EncryptPwdString(DecodeString(PwdKey),Password,MaxUserPwdLength,teAscii85));
        WriteInteger(Section,IniPrxType,integer(Mode));
        end;
      end;
    WriteInteger(Section,IniOvrMd,OvWrMode);
    WriteBool(Section,iniRename,Rename);
    WriteString(Section,IniDest,NewName);
    Free;
    end;
  end;

end.
