(* Copy files to FTP server , Unicode-Version (Delphi 10)

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

    Befehlszeilenoptionen:
    ----------------------
    Fr den Aufruf aus einer Batch-Datei (oder auch ber den Befehle "Ausfhren" im
    Windows-Startmen) stehen einige Befehlszeilenoptionen zur Verfgung.
    Programmaufruf: FtpCopy <Auftrag> [Optionen]
      <Auftrag>   - Name des Kopierauftrags (siehe oben)
                    Enthlt der Name Leerzeichen, muss er durch Anfhrungszeichen
                    umschlossen sein.
      Optionen    - Werden keine Optionen angegeben, ffnet sich das Programmfenster
                    mit dem in dem Aufruf enthaltenen Auftrag
        /force    - Der in dem Aufruf enthaltene Auftrag wird sofort ausgefhrt
        /continue - Das Fenster wird nach dem Transfer automatisch geschlossen
        /ini:<Name> - Abweichung von der o.g. Ini-Datei kann eine beliebige andere
                    Datei fr die Einstellungen ausgewhlt werden.

   Mar. 2015
   Aug. 2020 : new arrangement of FTP units
   last modified: April 2025
   *)

unit FtpCopyMain;

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes,
  Vcl.Graphics, Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ExtCtrls,
  Vcl.ComCtrls, Vcl.Buttons, FtpUtils, FtpDlg, NumberEd, FtpCopyCommon,
  TransferStat, FtpCopyUtils, WinApiUtils, CBFunctions;


const
  ProgName = 'FTP-Copy';
  Vers = ' - Vers. 3.0';
  CopRgt = ' 2008 - 2024 Dr. J. Rathlev, D-24222 Schwentinental';
  EmailAdr = 'kontakt(a)rathlev-home.de';
  FilterSep = ',';

type
  TMainForm = class(TForm)
    Label1: TLabel;
    btnAddFiles: TSpeedButton;
    btnCopy: TBitBtn;
    btnQuit: TBitBtn;
    btnInfo: TBitBtn;
    btnFtp: TBitBtn;
    Label3: TLabel;
    edDest: TEdit;
    btnCopyName: TSpeedButton;
    cbRename: TCheckBox;
    Label2: TLabel;
    lbFiles: TListBox;
    btnNewtask: TSpeedButton;
    cbxSource: TComboBox;
    edNewName: TEdit;
    btnRemFiles: TSpeedButton;
    btnNewFiles: TSpeedButton;
    laStatus: TLabel;
    cbxTask: TComboBox;
    Label4: TLabel;
    btnSourceDir: TSpeedButton;
    btnDuplTask: TSpeedButton;
    btnShortcut: TBitBtn;
    edExcludeFilter: TLabeledEdit;
    rgOverwrite: TRadioGroup;
    btnDelTask: TSpeedButton;
    paTop: TPanel;
    paFileList: TPanel;
    sbStatus: TStatusBar;
    paBottom: TPanel;
    edPrefix: TLabeledEdit;
    procedure FormCreate(Sender: TObject);
    procedure btnInfoClick(Sender: TObject);
    procedure btnQuitClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnAddFilesClick(Sender: TObject);
    procedure btnFtpClick(Sender: TObject);
    procedure btnCopyClick(Sender: TObject);
    procedure btnCopyNameClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure SourceSelectBtnClick(Sender: TObject);
    procedure cbxTaskCloseUp(Sender: TObject);
    procedure btnDelTaskClick(Sender: TObject);
    procedure btnRemFilesClick(Sender: TObject);
    procedure btnNewFilesClick(Sender: TObject);
    procedure cbRenameClick(Sender: TObject);
    procedure edNewNameExit(Sender: TObject);
    procedure btnNewtaskClick(Sender: TObject);
    procedure btnDuplTaskClick(Sender: TObject);
    procedure cbxSourceCloseUp(Sender: TObject);
    procedure btnShortcutClick(Sender: TObject);
    procedure edExcludeFilterExit(Sender: TObject);
    procedure rgOverwriteClick(Sender: TObject);
    procedure edPrefixExit(Sender: TObject);
  private
    { Private-Deklarationen }
    ProgVersName,
    ProgVersDate,
    UserPath,AltIni,
    AppPath,IniName,
    FileList,
    RootCertFile,
    CertFile,
    KeyFile          : string;
    SSLPassword      : AnsiString;
    FTP              : TExtFtp;
    CopyThread       : TCopyThread;
    WaitPrompt,
    Force,Connecting : boolean;
    FSize            : int64;
    DefCopySettings,
    CopySettings     : TCopySettings;
    procedure ShowData(ACopySettings : TCopySettings);
    procedure CancelCopy(var Msg: TMessage); message wmCancelAction;
    procedure ShowProgress (AAction : TFileAction; ACount: int64; ASpeedLimit : boolean);
    function StartCopy (IsRemote : boolean; ACopySettings : TCopySettings) : integer;
    { Public-Deklarationen }
  public
    function CheckRemote : boolean;
  end;

var
  MainForm: TMainForm;

implementation

{$R *.dfm}

uses Winapi.ShlObj, System.StrUtils, Web.HttpApp, System.IniFiles, IniFileUtils,
  WinUtils, MsgDialogs, WinShell, ExtSysUtils, FileErrors, UnitConsts, FtpConsts,
  Crypt, PwdDlg, IdFTPList, IdFtp, ShellDirDlg, GnuGetText, InitProg, LangUtils,
  ListUtils, StringUtils, NumberUtils, OpenFilesDlg, TextDlg, PathUtils, FileUtils;

{ ------------------------------------------------------------------- }
const
  IniExt = 'ini';

  CfGSekt    = 'Config';
  TaskSekt   = 'Task';
  SourceSekt = 'Source';
  TransSekt  = 'Status';

  iniLast   = 'LastTask';
  iniSCount = 'TaskCount';

  CertSekt = 'certificate';
  iniRootCert = 'RootCertFile';
  iniCertFile = 'CertFile';
  iniKeyFile  = 'KeyFile';
  iniPassword  = 'SSLPassword';

procedure TMainForm.FormCreate(Sender: TObject);
var
  IniFile : TUnicodeIniFile;
  ndx,
  i,n : integer;
  sn,
  sekt,s  : string;
  ts      : TCopySettings;
begin
  TranslateComponent(self);
  InitPaths(AppPath,UserPath);
  InitVersion(ProgName,Vers,CopRgt,3,3,ProgVersName,ProgVersDate);
  Caption:=_('Copy file(s) via FTP')+' ('+VersInfo.Comments+')';
  IniName:=Erweiter(AppPath,PrgName,IniExt);
  sn:=''; FileList:=''; Force:=false; AltIni:=''; WaitPrompt:=true;
  if ParamCount>0 then for i:=1 to ParamCount do begin
    s:=ParamStr(i);
    if s[1]='/' then begin
      Delete(s,1,1);
      if CompareOption(s,'force') then Force:=true;
      if CompareOption(s,'continue') then WaitPrompt:=false;
      if ReadOptionValue(s,'ini') then AltIni:=s;
      end
    else if length(sn)=0 then sn:=s else FileList:=FileList+s+',';
    end;
  if length(FileList)>0 then FileList:=CutChar(FileList,',');
  if (length(AltIni)>0) then begin
    if ContainsFullPath(AltIni) then IniName:=Erweiter('',AltIni,IniExt)
    else IniName:=Erweiter(PrgPath,AltIni,IniExt);
    end;
  s:=SetDirName(PrgPath)+CertConfig;
  if FileExists(s) then with TMemIniFile.Create(s) do begin
    RootCertFile:=ReadString(CertSekt,iniRootCert,'');
    CertFile:=ReadString(CertSekt,iniCertFile,'');
    KeyFile:=ReadString(CertSekt,iniKeyFile,'');
    SSLPassword:=ReadString(CertSekt,iniPassword,'');
    end
  else begin
    RootCertFile:=''; CertFile:='';
    KeyFile:=''; SSLPassword:='';
    end;
  IniFile:=TUnicodeIniFile.CreateForRead(IniName);
  with IniFile do begin
    n:=ReadInteger(CfgSekt,iniSCount,0);
    for i:=0 to n-1 do begin
      sekt:=TaskSekt+ZStrint(i,2);
      s:=ReadString(sekt,iniTName,'');
      if length(s)>0 then begin
        ts:=TCopySettings.Create(s);
        ReadCopySettings (IniFile,sekt,ts);
        cbxTask.AddItem(ts.TaskName,ts);
        end;
      end;
    ndx:=ReadInteger(CfgSekt,iniLast,0);
    LoadHistory(IniFile,SourceSekt,cbxSource.Items);
    Free;
    end;
  if length(sn)>0 then ndx:=cbxTask.Items.IndexOf(sn);
  if ndx>=0 then cbxTask.ItemIndex:=ndx else Force:=false;
  DefCopySettings:=TCopySettings.Create('');
  CopySettings:=DefCopySettings;
  Connecting:=false;
  FTP:=TExtFtp.Create(self,PrgPath,RootCertFile,CertFile,KeyFile,SSLPassword);
  end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  FreeAndNil(Ftp);
  DefCopySettings.Free;
  end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  IniFile : TUnicodeIniFile;
  i,j     : integer;
  sekt    : string;
begin
  IniFile:=TUnicodeIniFile.CreateForWrite(IniName,true);
  with IniFile do begin
    try
      with cbxTask,Items do begin
        WriteInteger(CfgSekt,iniSCount,Count);
        WriteInteger(CfgSekt,iniLast,ItemIndex);
        for i:=0 to Count-1 do with objects[i] as TCopySettings do begin
          sekt:=TaskSekt+ZStrint(i,2);
          WriteCopySettings(iniFile,sekt,objects[i] as TCopySettings);
          end;
        SaveHistory(IniFile,SourceSekt,'',true,cbxSource.Items);
        end;
    finally
      Free;
      end;
    end;
  end;

function TMainForm.CheckRemote : boolean;
begin
  with cbxTask do if Force and (Items.Count>0) and (ItemIndex>=0) then begin
//    InfoDialog('',IniName+sLineBreak+ts.taskname);
    if StartCopy(true,Items.Objects[ItemIndex] as TCopySettings)>0 then ExitCode:=1;
    Result:=true;
    end
  else Result:=false;
  end;

procedure TMainForm.FormShow(Sender: TObject);
begin
  FtpDialog.Init(IniName,'');
  TransferStatus.LoadFromIni(IniName,TransSekt);
  with cbxTask do if (Items.Count>0) and (ItemIndex>=0) then
      CopySettings:=Items.objects[ItemIndex] as TCopySettings;
  ShowData(CopySettings);
  end;

procedure TMainForm.ShowData(ACopySettings : TCopySettings);
var
  i : integer;
  filt,sub : boolean;
begin
  with ACopySettings do begin
    AddToHistory(cbxSource.Items,Source);
    cbxSource.ItemIndex:=0;
    lbFiles.Clear; filt:=false; sub:=false;
    with SFiles do begin
      for i:=0 to Count-1 do begin
        lbFiles.Items.Add(Strings[i]);
        Filt:=Filt or ContainsCharacters(Strings[i],['*','?']);
        Sub:=Sub or ContainsCharacters(Strings[i],['\']);
        end;
      cbRename.Enabled:=(Count=1) and not (Filt or Sub);
      end;
    btnRemFiles.Enabled:=SFiles.Count>0;
    btnNewFiles.Enabled:=SFiles.Count>0;
    with edExcludeFilter do begin
      Enabled:=Filt;
      Text:=Exclude;
      end;
    edPrefix.Text:=Prefix;
    with FtpPar do if length(Host)>0 then edDest.Text:=GetFtpHint(FtpPar)
    else edDest.Text:='';
    btnCopy.Enabled:=(length(edDest.Text)>0) and (lbFiles.Items.Count>0);
    rgOverwrite.ItemIndex:=OvWrMode;
    if cbRename.Enabled then begin
      cbRename.Checked:=Rename;
      edNewName.Text:=NewName;
      end
    else begin
      cbRename.Checked:=false;
      edNewName.Text:='';
      end;
    edNewName.Enabled:=cbRename.Checked;
    btnCopyName.Enabled:=cbRename.Checked;
    end;
  end;

procedure TMainForm.rgOverwriteClick(Sender: TObject);
begin
  CopySettings.OvWrMode:=rgOverwrite.ItemIndex;
  end;

procedure TMainForm.cbRenameClick(Sender: TObject);
begin
  CopySettings.Rename:=cbRename.Checked;
  edNewName.Enabled:=cbRename.Checked;
  btnCopyName.Enabled:=cbRename.Checked;
  end;

procedure TMainForm.edExcludeFilterExit(Sender: TObject);
begin
  CopySettings.Exclude:=edExcludeFilter.Text;
  end;

procedure TMainForm.edNewNameExit(Sender: TObject);
begin
  CopySettings.NewName:=edNewName.Text;
  end;

procedure TMainForm.edPrefixExit(Sender: TObject);
begin
  CopySettings.Prefix:=edPrefix.Text;
  end;

procedure TMainForm.btnInfoClick(Sender: TObject);
begin
  InfoDialog(ProgName,ProgVersName+' - '+ProgVersDate+#13+
           VersInfo.CopyRight+#13'D-24222 Schwentinental');
  end;

procedure TMainForm.btnQuitClick(Sender: TObject);
begin
  Close;
  end;

procedure TMainForm.btnNewtaskClick(Sender: TObject);
var
   s : string;
begin
  s:='';
  if InputQuery(_('New copy task'),_('Taskname:'),s) then begin
    CopySettings:=TCopySettings.Create(s);
    with cbxTask do ItemIndex:=Items.AddObject(s,CopySettings);
    SourceSelectBtnClick(Sender);
    end;
  end;

procedure TMainForm.btnDelTaskClick(Sender: TObject);
var
  n : integer;
begin
  with cbxTask do if (Items.Count>0) and
      ConfirmDialog('',Format(_('Remove task: '+sLineBreak+'%s?'),[Text])) then begin
    n:=ItemIndex;
    CopySettings.Free;
    Items.Delete(ItemIndex);
    if Items.Count=0 then CopySettings:=DefCopySettings
    else begin
      if n<Items.Count then ItemIndex:=n else ItemIndex:=Items.Count-1;
      CopySettings:=(Items.Objects[ItemIndex] as TCopySettings);
      end;
    ShowData(CopySettings);
    end;
  end;

procedure TMainForm.btnDuplTaskClick(Sender: TObject);
var
   s : string;
begin
  s:=cbxTask.Text;
  if InputQuery(_('Duplicate copy task'),_('Taskname:'),s) then begin
    CopySettings:=TCopySettings.CreateFrom(s,CopySettings);
    with cbxTask do ItemIndex:=Items.AddObject(s,CopySettings);
    SourceSelectBtnClick(Sender);
    end;
  end;

procedure TMainForm.SourceSelectBtnClick(Sender: TObject);
var
  s : string;
begin
  if length(cbxSource.Text)>0 then s:=cbxSource.Text
  else s:=UserPath;
  if ShellDirDialog.Execute(_('Select source directory'),true,true,true,UserPath,s) then begin
    CopySettings.Source:=IncludeTrailingPathDelimiter(s);
    ShowData(CopySettings);
    end;
  end;

procedure TMainForm.cbxSourceCloseUp(Sender: TObject);
begin
  CopySettings.Source:=IncludeTrailingPathDelimiter(cbxSource.Text);
  ShowData(CopySettings);
  end;

procedure TMainForm.cbxTaskCloseUp(Sender: TObject);
begin
  with cbxTask do if Items.Count>0 then begin
    CopySettings:=Items.Objects[ItemIndex] as TCopySettings;
    ShowData(CopySettings);
    end;
  end;

procedure TMainForm.btnNewFilesClick(Sender: TObject);
begin
  with CopySettings do if (SFiles.Count>0) and ConfirmDialog('',_('Delete all items?')) then begin
    SFiles.Clear;
    ShowData(CopySettings);
    end;
  end;

procedure TMainForm.btnAddFilesClick(Sender: TObject);
var
  i   : integer;
  fl  : TStringList;
begin
  if (length(cbxSource.Text)>0) then with CopySettings do begin
    fl:=TStringList.Create;
    if OpenFilesDialog.Execute(_('File(s) to copy'),cbxSource.Text,
                 _('all|*.*|Pdf|*.pdf|Html|*.html|Zip|*.zip|Exe|*.exe'),fl,true) then begin
      with fl do for i:=0 to Count-1 do SFiles.Add(Strings[i]);
      ShowData(CopySettings);
      end;
    fl.Free;
    end
  else ErrorDialog(_('Select directory first!'));
  end;

procedure TMainForm.btnRemFilesClick(Sender: TObject);
var
  i,n : integer;
  s   : string;
begin
  if lbFiles.SelCount=0 then begin
    ErrorDialog(_('Please select at least one entry!'));
    Exit;
    end;
  if lbFiles.SelCount=1 then s:=_('Remove %u entry?')
  else s:=_('Remove %u entries?');
  if ConfirmDialog('',Format(s,[lbFiles.SelCount])) then with CopySettings do begin
    with lbFiles do for i:=Items.Count-1 downto 0 do if Selected[i] then begin
      n:=SFiles.IndexOf(Items[i]);
      if n>=0 then SFiles.Delete(n);
      end;
    end;
  ShowData(CopySettings);
  end;

procedure TMainForm.btnShortcutClick(Sender: TObject);
var
  s,t : string;
begin
  if ConfirmDialog(Format(_('Create desktop shortcut to start copy task  "%s"'),[CopySettings.TaskName])) then begin
    t:=Erweiter(GetDesktopFolder(CSIDL_DESKTOPDIRECTORY),CopySettings.TaskName,'lnk');
    if FileExists(t) and
      not ConfirmDialog('',Format(_('Replace existing desktop icon for "%s"?'),[CopySettings.TaskName])) then Exit;
    s:=AnsiQuotedStr(CopySettings.TaskName,Quote)+' /f';
    if length(AltIni)>0 then s:=s+' "/i:'+AltIni+'"';
    MakeLink (t,Application.ExeName,s,PrgPath,Format(_('Copy files for %s'),[CopySettings.TaskName]));
    RefreshDesktop;
    end;
  end;

procedure TMainForm.btnCopyNameClick(Sender: TObject);
begin
  edNewName.Text:=CopySettings.SFiles[0];
  cbRename.Checked:=true;
  end;

procedure TMainForm.btnFtpClick(Sender: TObject);
begin
  with CopySettings do if FtpDialog.Execute (FtpPar,false,Ftp.CanUseTls) then
    ShowData(CopySettings);
  end;

{ ------------------------------------------------------------------- }
(* Kopierfortschritt aktualisieren *)
procedure TMainForm.ShowProgress (AAction : TFileAction; ACount : int64; ASpeedLimit : boolean);
begin
  if ACount<0 then begin
    FSize:=abs(ACount);
    TransferStatus.UpdateProgress(0,FSize);
    end
  else TransferStatus.UpdateProgress(ACount,FSize);
  end;

(* Nachricht von Fortschrittsanzeige bei Klick auf Abbrechen-Button *)
procedure TMainForm.CancelCopy(var Msg: TMessage);
begin
  if assigned(CopyThread) then with CopyThread do if not Done then CancelThread
  else if Connecting then FTP.Disconnect;
  end;

procedure TMainForm.btnCopyClick(Sender: TObject);
begin
  with CopySettings do begin
    NewName:=edNewName.Text;
    Rename:=cbRename.Checked;
    OvWrMode:=rgOverwrite.ItemIndex;
    with FtpPar do begin
      if (length(Host)=0) or (length(Username)=0) then btnFtpClick(self);
      end;
    end;
  StartCopy(false,CopySettings);
  end;


function TMainForm.StartCopy (IsRemote : boolean; ACopySettings : TCopySettings) : integer;
var
  ok         : boolean;
  s,t,ss,sd,fd  : string;
  i,n        : integer;
  DirInfo    : TSearchRec;
  Findresult : integer;
begin
  Result:=0;
  with ACopySettings do if (SFiles.Count>0) or (IsRemote and (length(FileList)>0)) then begin
    sbStatus.SimpleText:='';
    with FtpPar do if (length(Host)=0) or (length(Username)=0) then begin
      ErrorDialog(_('No FTP host or username'));
      Exit;
      end;
//    FTP:=TExtFtp.Create(self,PrgPath,RootCertFile,CertFile,KeyFile,SSLPassword);
    with FtpPar do begin
      FTP.SecureTransfer:=SecureMode;
      Ftp.Host:=Host;
      Ftp.Port:=Port;
      Ftp.Username:=Username;
      Ftp.Passive:=Passive;
      Ftp.ForceUtf8:=ForceUtf8;
      Ftp.WriteLogFile:=WriteLog;
      Ftp.UseHOST:=UseHost;
      Ftp.CaseMode:=CaseMode;
      Ftp.TransferTimeout:=defTimeout;
      if length(Password)=0 then begin
        ok:=PasswordDialog.Execute (rsFtpConn,Format(_('The connection to server:'+
              sLineBreak+'  %s:%u'+sLineBreak+'for user: "%s" requires a password!'),
              [Host,Port,Username]),Password)=mrOK;
        end
      else ok:=true;
      if ok and (Proxy.Mode<>fpcmNone) then begin // proxy settings
        Ftp.ProxySettings.ProxyType:=Proxy.Mode;
        Ftp.ProxySettings.Host:=Proxy.Server;
        Ftp.ProxySettings.Port:=Proxy.Port;
        Ftp.ProxySettings.Username:=Proxy.Username;
        with Proxy do if length(Password)=0 then
          ok:=PasswordDialog.Execute (rsFtpConn,
              Format(rsProxyHint,[Server,Port,Username]),Password)=mrOK;
        end;
      if ok then begin
        TransferStatus.ShowWindow(Handle,_('Transfer files'),
                     GetFtpHint(FtpPar,false),_('Connecting to FTP server'));
        Application.ProcessMessages;
        with Ftp.ProxySettings do if ProxyType<>fpcmNone then Password:=Proxy.Password;
        Ftp.Password:=Password;
        try
          Connecting:=true;
          Ftp.Connect;
        except
          ok:=false;
          end;
        Application.ProcessMessages;
        Connecting:=false;
        if ok then ok:=not TransferStatus.Stopped and Ftp.ForceDir(Directory);
        if ok then Ftp.GetFileList;
        end;
      end;
    if ok then begin
      Application.ProcessMessages;
      n:=0; ss:=Source;
      with TransferStatus do begin
        Directory:='/'+FtpPar.Directory;
        StartCopy;
        Action:=_('Copy from ')+Source;
        with FtpPar do Target:=_('to ')+GetFtpHint(FtpPar,false);
        end;
      fd:=Ftp.CurrentDir;
// copy all files from list
      with SFiles do for i:=0 to Count-1 do begin
        if TransferStatus.Stopped then Break;
        if Ftp.ChangeToDir(fd) then Ftp.GetFileList;
        if (i=0) and Rename and (length(NewName)>0) then begin
          sd:=DosPathToUnixPath(ExtractFileDir(NewName));
          ok:=Ftp.ForceDir(sd);
          if ok then s:=ExtractFileName(NewName)
          end
        else begin
          s:=Strings[i]; sd:=ExtractFileDir(s);
          // goto subdirectory
          if length(sd)>0 then begin
            ss:=AddPath(Source,sd);
            sd:=DosPathToUnixPath(Prefix+sd);
            ok:=Ftp.ForceDir(sd);
            if ok then Ftp.GetFileList;
            end
          else ok:=true;
          if ok then s:=ExtractFileName(s);
          end;
        if not ok then begin
          TransferStatus.Close;
          ErrorDialog(TryFormat(_('Cannot create remote directory:'+sLineBreak+'%s on %s!'),
                      [sd,FtpPar.Host]));
          exit;
          end;
        FindResult:=FindFirst (AddPath(ss,ExtractFileName(Strings[i])),faArchive+faReadOnly+faHidden+faSysfile,DirInfo);
        if FindResult<>0 then TransferStatus.Status:='==> '+SysErrorMessage(FindResult);
        while FindResult=0 do with DirInfo do begin
          if not MatchesFilter(Name,Exclude,FilterSep) then begin
            if not Rename then s:=Name;
            t:=AddPath(ss,Name);
            FSize:=LongFileSize(t);
            case OvWrMode of
            0 : ok:=true;
            1 : ok:=Ftp.CheckTimeStamp(s,GetFileDateTime(t),2)<2;
            else ok:=not Ftp.FtpFileExists(s);
              end;
            if ok then begin
              with TransferStatus do begin
                Position:=0;
                if Rename then Status:=ExtractFilename(t)+_(' to ')+s
                else Status:=DosPathToUnixPath(AddPath(sd,s));
                end;
              Application.ProcessMessages;
              CopyThread:=TWriteFtpThread.Create(Ftp,t,'',s,false,false,defFtpBufferSize,tpNormal);
              with CopyThread do begin
                OnProgress:=ShowProgress;
                repeat
                  Sleep(10);
                  Application.ProcessMessages;
                  until Done;
                if ErrorCode<>errOK then TransferStatus.Status:='==> '+GetCopyErrMsg(ErrorCode);
                inc(n);
                Free;
                end;
              Ftp.SetTimeStampFromFile(t,s);
              end;
            end;
          FindResult:=FindNext (DirInfo);
          end;
        FindClose(DirInfo);
        sbStatus.SimpleText:=Format(_('%u File(s) copied'),[n]);
        end;
      if IsRemote and (length(FileList)>0) then begin  // files from command line
        repeat
          t:=ReadNxtStr(FileList,',');
          if not ContainsFullPath(t) then t:=Source+t;
          if FileExists(t) then begin
            s:=ExtractFileName(t);
            FSize:=LongFileSize(t);
            case OvWrMode of
            0 : ok:=true;
            1 : ok:=Ftp.CheckTimeStamp(s,GetFileDateTime(t),2)<2;
            else ok:=not Ftp.FtpFileExists(s);;
              end;
            if ok then begin
              with TransferStatus do begin
                Position:=0;
                Status:=s;
                end;
              Application.ProcessMessages;
              CopyThread:=TWriteFtpThread.Create(Ftp,t,'',s,false,false,defFtpBufferSize,tpNormal);
              with CopyThread do begin
                OnProgress:=ShowProgress;
                repeat
                  Sleep(10);
                  Application.ProcessMessages;
                  until Done;
                if ErrorCode<>errOK then TransferStatus.Status:='==> '+GetCopyErrMsg(ErrorCode);
                inc(n);
                Free;
                end;
              Ftp.SetTimeStampFromFile(t,s,false);
              end;
            end
          else TransferStatus.Error:='==> '+_('File not found: ')+t;
          until TransferStatus.Stopped or (length(FileList)=0);
        end;
      sbStatus.SimpleText:=Format(_('%u File(s) copied'),[n]);
      with TransferStatus do begin
        Status:=_('Done');
        if WaitPrompt then CloseWindow(0,n) else Close;
        end;
      FTP.Disconnect;
      Result:=n;
      end
    else begin
      TransferStatus.Close;
      ErrorDialog(TryFormat(_('Connection to FTP server:'+sLineBreak+'%s failed!'),[FtpPar.Host]));
      end;
    end
  else ErrorDialog('',_('No files selected!'));;
  end;

end.
