(* Update files on FTP server (Delphi 10)
   Console version for use in batch scripts
   Using settings from FtpCopy

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   The program reads the tasks created by FtpCopy (see FtpCopy.ini) and will
   update all older files on the FTP server

   Command line:
   -------------
   Calling: FtpRefresh <task1> <task2> ... [Options]
            FtpRefresh /file <task1> <file1> <file2> ... [Options]
      <task#>  - Name of task as defined in FtpCopy
                 If the task name contains spaces enclosing quotes are required
      <file#>  - Use specified file with <task1>
      /ini:<Name> - Use other than default ini file
      /log     - Write FTP communication log
      /file    - File mode

   Exit code: 0 - some files were copied
              1 - no files were copied

   Jan. 2019
   last modified: December 2024
   *)

program FtpRefresh;

{$APPTYPE CONSOLE}

{$R *.res}

uses
  System.SysUtils, System.IniFiles, System.Classes, IdFtp, PathUtils,
  FileErrors, FtpUtils, FtpCopyUtils, FtpCopyCommon, StringUtils, FileUtils,
  IniFileUtils, ListUtils, NumberUtils, Crypt, ConsolePwd;

{ ------------------------------------------------------------------- }
const
  PrgDesc = 'Update files on FTP server';
  CopRgt = ' 2019-2025 Dr. J. Rathlev, D-24222 Schwentinental';

  IniExt = 'ini';

  CfGSekt    = 'Config';
  TaskSekt   = 'Task';

  CertSekt = 'certificate';
  iniRootCert = 'RootCertFile';
  iniCertFile = 'CertFile';
  iniKeyFile  = 'KeyFile';
  iniPassword  = 'SSLPassword';
  iniSCount = 'TaskCount';

  defIniName = 'FtpCopy.ini';

  FilterSep = ',';

{ ------------------------------------------------------------------- }
var
  tn,s,sekt,
  IniName,
  PrgPath,
  AppPath,
  RootCertFile,
  CertFile,
  KeyFile          : string;
  SSLPassword      : AnsiString;
  i,j,n,m,k        : integer;
  FileMode,
  WLog             : boolean;
  ts               : TCopySettings;
  FTP              : TExtFTP;
  TaskList         : TStringList;
  IniFile          : TUnicodeIniFile;


{ ------------------------------------------------------------------- }
function StartCopy (ACopySettings : TCopySettings; FileList : string = '') : integer;
var
  ok     : boolean;
  s,t,ss,sd,fd  : string;
  i,n    : integer;
  DirInfo    : TSearchRec;
  Findresult : integer;
  CopyThread : TWriteFtpThread;

  function DosPathToUnixPath(const Path: string): string;
  begin
    Result := Path.Replace('\', '/');
  end;

  function GetFtpHint(FtpPar : TFtpParams; ShowDir : boolean) : string;
  begin
    with FtpPar do begin
      if SecureMode>0 then Result:='ftps' else Result:='ftp';
      Result:=Result+'://'+Host+Colon+IntToStr(Port);
      if ShowDir then Result:=Result+'/'+Directory;
      end;
    end;

begin
  Result:=0;
  Writeln('Task: ',ACopySettings.TaskName);
  with ACopySettings do if (SFiles.Count>0) or not IsEmptyStr(FileList) then begin
    with FtpPar do if (length(Host)=0) or (length(Username)=0) then begin
      Writeln('No FTP host or username');
      Exit;
      end;
    with FtpPar do begin
      FTP.SecureTransfer:=SecureMode;
      Ftp.Host:=Host;
      Ftp.Port:=Port;
      Ftp.Username:=Username;
      Ftp.Passive:=Passive;
      Ftp.ForceUtf8:=ForceUtf8;
      Ftp.WriteLogFile:=WriteLog;
      Ftp.UseHOST:=UseHost;
      Ftp.CaseMode:=CaseMode;
      Ftp.TransferTimeout:=defTimeout;
      Ftp.LogName:=AddPath(GetEnvironmentVariable('TEMP'),'FtpRefresh.log');
      Ftp.WriteLogFile:=WLog; // overwrite ini setting
      if length(Password)=0 then begin
        Writeln(Format('The connection to server: %s:%u requires a password!',[Host,Port]));
        Password:=ConsoleGetPassword (Format('Enter password for user "%s": ',[Username]));
        ok:=length(Password)>0;
        end
      else ok:=true;
      if ok and (Proxy.Mode<>fpcmNone) then begin // proxy settings
        Ftp.ProxySettings.ProxyType:=Proxy.Mode;
        Ftp.ProxySettings.Host:=Proxy.Server;
        Ftp.ProxySettings.Port:=Proxy.Port;
        Ftp.ProxySettings.Username:=Proxy.Username;
        with Proxy do if length(Password)=0 then  begin
          Writeln(Format('The connection to FTP proxy server: %s:%u requires a password!',[Server,Port]));
          Password:=ConsoleGetPassword (Format('Enter proxy password for user "%s": ',[Username]));
          end
        end;
      if ok then begin
        Writeln('Connecting to server ',GetFtpHint(FtpPar,false));
        with Ftp.ProxySettings do if ProxyType<>fpcmNone then Password:=Proxy.Password;
        Ftp.Password:=Password;
        try
          Ftp.Connect;
        except
          writeln(Ftp.IndyError);
          ok:=false;
          end;
        if ok then begin
          ok:=Ftp.ForceDir(Directory);
          if ok then begin
            if not Ftp.GetFileList then Writeln('Could not read FTP file list!');
            end
          else Writeln(Format('=> Creation of remote directory: %s failed!',[Directory]));
          end
        else Writeln(Format('=> Connection to FTP server: %s failed!',[Host]));
        end;
      end;
    if ok then begin
      n:=0; ss:=Source;
      Writeln('Remote directory: /'+FtpPar.Directory);
      Writeln(Format('Copy files from %s',[Source]));
      fd:=Ftp.CurrentDir;
// copy all files from list
      with SFiles do for i:=0 to Count-1 do begin
        if Ftp.ChangeToDir(fd) then Ftp.GetFileList;
        if (i=0) and Rename and (length(NewName)>0) then begin
          sd:=DosPathToUnixPath(ExtractFileDir(NewName));
          ok:=Ftp.ForceDir(sd);
          if ok then s:=ExtractFileName(NewName)
          end
        else begin
          s:=Strings[i]; sd:=ExtractFileDir(s);
          // goto subdirectory
          if length(sd)>0 then begin
            ss:=AddPath(Source,sd);
            sd:=DosPathToUnixPath(Prefix+sd);
            ok:=Ftp.ForceDir(sd);
            if ok then Ftp.GetFileList;
            end
          else ok:=true;
          if ok then s:=ExtractFileName(s);
          end;
        if not ok then begin
          Writeln('',Format('Cannot create remote directory: %s on %s!',[FtpPar.Host,s]));
          exit;
          end;
        FindResult:=FindFirst (AddPath(ss,ExtractFileName(Strings[i])),faArchive+faReadOnly+faHidden+faSysfile,DirInfo);
        if FindResult<>0 then Writeln('==> '+SysErrorMessage(FindResult));
        while FindResult=0 do with DirInfo do begin
          if not MatchesFilter(Name,Exclude,FilterSep) then begin
            if not Rename then s:=Name;
            t:=AddPath(ss,Name);
            case OvWrMode of
            0 : ok:=true;
            1 : ok:=Ftp.CheckTimeStamp(s,GetFileDateTime(t),2)<2;
            else ok:=not Ftp.FtpFileExists(s);
              end;
            if ok then begin
              if Rename then Writeln('  -> '+ExtractFilename(t)+' to '+s)
              else Writeln('  -> '+DosPathToUnixPath(AddPath(sd,s)));
              CopyThread:=TWriteFtpThread.Create(Ftp,t,'',s,false,false,defFtpBufferSize,tpNormal);
              with CopyThread do begin
                repeat
                  Sleep(10);
//                  Application.ProcessMessages;
                  until Done;
                if ErrorCode<>errOK then Writeln('==> '+GetCopyErrMsg(ErrorCode));
                inc(n);
                Free;
                end;
              Ftp.SetTimeStampFromFile(t,s);
              end;
            end;
          FindResult:=FindNext (DirInfo);
          end;
        FindClose(DirInfo);
        end;
      if (length(FileList)>0) then begin  // files from command line
        repeat
          t:=ReadNxtStr(FileList,VertBar);
          if not ContainsFullPath(t) then t:=Source+t;
          if FileExists(t) then begin
            s:=ExtractFileName(t);
            case OvWrMode of
            0 : ok:=true;
            1 : ok:=Ftp.CheckTimeStamp(s,GetFileDateTime(t),2)<2;
            else ok:=not Ftp.FtpFileExists(s);;
              end;
            if ok then begin
              Writeln('  -> '+s);
              CopyThread:=TWriteFtpThread.Create(Ftp,t,'',s,false,false,defFtpBufferSize,tpNormal);
              with CopyThread do begin
                repeat
                  Sleep(10);
  //                Application.ProcessMessages;
                  until Done;
                if ErrorCode<>errOK then writeln('==> '+GetCopyErrMsg(ErrorCode));
                inc(n);
                Free;
                end;
              Ftp.SetTimeStampFromFile(t,s);
              end;
            end
          else writeln('==> File not found: ',t);
          until (length(FileList)=0);
        end;
      Writeln(Format('=> %u File(s) copied',[n]));
      Result:=n;
      end;
    with FTP do if Connected then Disconnect;
    Writeln('Disconnected from FTP server');
    end
  else Writeln('=> No files specified!');
  end;

{ ------------------------------------------------------------------- }
begin
  try
    Writeln(PrgDesc); Writeln(CopRgt);
    AppPath:=GetEnvironmentVariable('APPDATA'); //DesktopFolder(CSIDL_APPDATA);
    PrgPath:=ExtractFilePath(ParamStr(0));
    tn:=''; IniName:=''; WLog:=false; FileMode:=false;
    if ParamCount>0 then for i:=1 to ParamCount do begin
      s:=ParamStr(i);
      if s[1]='/' then begin
        Delete(s,1,1);
        if ReadOptionValue(s,'ini') then IniName:=s
        else if CompareOption(s,'log') then WLog:=true
        else if CompareOption(s,'file') then FileMode:=true;
        end
      else if length(tn)=0 then tn:=s else tn:=tn+VertBar+s;
      end;
    if length(tn)>0 then begin
      if (length(IniName)>0) then begin
        if ContainsFullPath(IniName) then IniName:=Erweiter('',IniName,IniExt)
        else IniName:=Erweiter(AppPath,IniName,IniExt);
        end
      else IniName:=AddPath(AppPath,defIniName);
      s:=AddPath(PrgPath,CertConfig);
      if FileExists(s) then with TMemIniFile.Create(s) do begin
        RootCertFile:=ReadString(CertSekt,iniRootCert,'');
        CertFile:=ReadString(CertSekt,iniCertFile,'');
        KeyFile:=ReadString(CertSekt,iniKeyFile,'');
        SSLPassword:=ReadString(CertSekt,iniPassword,'');
        end
      else begin
        RootCertFile:=''; CertFile:='';
        KeyFile:=''; SSLPassword:='';
        end;
      TaskList:=TStringList.Create;
      TaskList.Sorted:=true;
      IniFile:=TUnicodeIniFile.CreateForRead(IniName);
      with IniFile do begin
        n:=ReadInteger(CfgSekt,iniSCount,0);
        for i:=0 to n-1 do begin
          sekt:=TaskSekt+ZStrint(i,2);
          s:=ReadString(sekt,iniTName,'');
          if (length(s)>0) then begin
            ts:=TCopySettings.Create(s);
            ReadCopySettings (IniFile,sekt,ts);
            TaskList.AddObject(ts.TaskName,ts);
            end;
          end;
        Free;
        end;
      if TaskList.Count>0 then begin
        k:=0;
        FTP:=TExtFtp.Create(nil,PrgPath,RootCertFile,CertFile,KeyFile,SSLPassword);
        if FileMode then begin
          Writeln;
          s:=ReadNxtStr(tn,VertBar);  // Task
          n:=TaskList.IndexOf(s);
          if n>=0 then k:=StartCopy(TaskList.Objects[n] as TCopySettings,tn)
          else Writeln(Format('Task "%s" not found!',[s]));
          end
        else repeat
          Writeln;
          s:=ReadNxtStr(tn,VertBar);
          n:=TaskList.IndexOf(s);
          if n>=0 then k:=k+StartCopy(TaskList.Objects[n] as TCopySettings)
          else Writeln(Format('Task "%s" not found!',[s]));
          until IsEmptyStr(tn);
        FTP.Free;
        if k=0 then ExitCode:=1;
        end
      else Writeln(Format('No tasks found in "%s"!',[IniName]));
      FreeListObjects(TaskList);
      TaskList.Free;
      end
    else begin
      Writeln('No task specified!');
      Writeln;
      Writeln('Calling: FtpRefresh <task1> <task2> ... [Options]');
      Writeln('  <task#>     - Name of task as defined in FtpCopy or files');
      Writeln('                Enclose in quotes if the task name contains spaces!');
      Writeln('  /file       - Use file mode');
      Writeln('  /ini:<Name> - Use other than default ini file');
      Writeln('  /log        - Write FTP communication log');
      end;
  except
    on E: Exception do
      Writeln(E.ClassName, ': ', E.Message);
  end;
{$IFDEF Debug}
  write ('Strike enter key to continue ...');
  readln;
{$EndIf}
end.
