(* Delphi Dialog
   Dialog: eine oder mehrere Dateien oder Filterausdruck auswhlen
   ===============================================================

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   Oct. 2008
   last modified: December 2024
   *)

unit OpenFilesDlg;

interface

uses Winapi.Windows, System.SysUtils, System.Classes, Vcl.Graphics, Vcl.Forms,
  Vcl.Controls, Vcl.StdCtrls, Vcl.Buttons, Vcl.ExtCtrls, Vcl.FileCtrl;

const
  IgnoreExt : array of string = ['~','bak','backup'];

type
  TOpenFilesDialog = class(TForm)
    OKBtn: TBitBtn;
    CancelBtn: TBitBtn;
    lePath: TLabeledEdit;
    FileListBox: TFileListBox;
    FilterComboBox: TFilterComboBox;
    Label1: TLabel;
    btnAll: TBitBtn;
    btnNone: TBitBtn;
    paTop: TPanel;
    paFiles: TPanel;
    paSubDir: TPanel;
    leSubdir: TLabeledEdit;
    bbSubDir: TBitBtn;
    bbFilter: TBitBtn;
    procedure btnAllClick(Sender: TObject);
    procedure btnNoneClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure bbSubDirClick(Sender: TObject);
  private
    { Private declarations }
    procedure GetSelectedDir (n : integer);
    procedure BuildDirList (const ADir : string; AllowUp : boolean);
  public
    { Public declarations }
    function Execute (const ATitle,APath,AFilter : string; FList : TStringList;
                      ASubDir : boolean = false) : boolean;
  end;

var
  OpenFilesDialog: TOpenFilesDialog;

implementation

{$R *.dfm}

uses System.StrUtils, GnuGetText, WinUtils, PathUtils, ListSelectDlg;

procedure TOpenFilesDialog.FormCreate(Sender: TObject);
begin
  TranslateComponent (self);
  end;

procedure TOpenFilesDialog.bbSubDirClick(Sender: TObject);
begin
  ListSelectDialog.ShowList(BottomLeftPos(bbSubDir,0,2));
  end;

procedure TOpenFilesDialog.btnAllClick(Sender: TObject);
begin
  FileListBox.SelectAll;
  end;

procedure TOpenFilesDialog.btnNoneClick(Sender: TObject);
begin
  FileListBox.ClearSelection;
  end;

procedure TOpenFilesDialog.GetSelectedDir (n : integer);
var
  sd : string;
begin
  sd:=ListSelectDialog.GetItem(n);
  with leSubdir do begin
    if sd='..' then begin
      Text:=ExtractFilePath(Text);
      end
    else Text:=AddPath(Text,sd);
    BuildDirList(sd,length(Text)>0);
    end;
  end;

procedure TOpenFilesDialog.BuildDirList (const ADir : string; AllowUp : boolean);
var
  sr  : TSearchRec;
  n,i : integer;
  fl  : TStringList;
  se  : string;

  function IsValidExt (const Ext : string) : boolean;
  var
    i : integer;
  begin
    Result:=true;
    for i:= low(IgnoreExt) to high(IgnoreExt) do if AnsiContainsText(Ext,IgnoreExt[i]) then begin
      Result:=false; Break;
      end;
    end;

begin
  fl:=TStringList.Create;
  fl.Sorted:=true;
  with ListSelectDialog do begin
    Clear; Sorted:=false;
    end;
//  DirList.Clear;
  n:=FindFirst(AddPath(ADir,'*.*'),faAnyFile,sr);
  while n=0 do with sr do begin
    if ((Attr and faDirectory)<>0) then begin
      if ((Name<>'..') or AllowUp) and (Name<>'.') then ListSelectDialog.Add(Name);
      end
    else begin // files
      se:=GetExt(Name);
      n:=fl.IndexOf(se);
      if (n<0) and IsValidExt(se) then fl.Add(se);
      end;
    n:=FindNext(sr);
    end;
  FindClose(sr);
//  with FileListBox do begin
//    for i:=0 to Items.Count-1 do begin
//      se:=GetExt(Items[i]);
//      n:=fl.IndexOf(se);
//      if (n<0) and IsValidExt(se) then fl.Add(se);
//      end;
//    end;
  with fl do begin
    se:=_('all')+'|*.*';
    for i:=0 to Count-1 do se:=se+'|'+Strings[i]+_(' files')+'|*.'+Strings[i];
    FilterComboBox.Filter:=se;
    Free;
    end;
  FileListBox.Directory:=ADir;
  end;

function TOpenFilesDialog.Execute (const ATitle,APath,AFilter : string;
                                   FList : TStringList; ASubDir : boolean) : boolean;
var
  i  : integer;
  mr : TModalResult;
begin
  ListSelectDialog:=TListSelectDialog.Create(self);
  ListSelectDialog.OnSelect:=GetSelectedDir;
  if length(ATitle)>0 then Caption:=ATitle;
  lePath.Text:=APath;
  leSubdir.Text:='';
  paSubDir.Visible:=ASubDir;
  BuildDirList(APath,false);
//  FileListBox.Directory:=APath;
//  FilterComboBox.Filter:=AFilter;
  mr:=ShowModal;
  if mr=mrOK then with FileListBox do begin
    FList.Clear;
    for i:=0 to Items.Count-1 do if Selected[i] then FList.Add(AddPath(leSubdir.Text,Items[i]));
    Result:=true;
    end
  else if mr=mrYes then begin
    FList.Text:=AddPath(leSubdir.Text,FilterComboBox.Mask);
    Result:=true;
    end
  else Result:=false;
  ListSelectDialog.Free;
  end;

end.
