(* Add version info to a program setup and
   copy the setup files to local directories and/or via FTP

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.


   Vers. 1.0 (Oct. 2008): Delphi7
   Vers. 3.0 (Jan. 2010): Delphi 2009 (Unicode)
       Syntax of HTML templates
       <-- #var
       date-format = 1               // 0 = ISO, 1 = German, 2 = English
       decimal-sep = ,               // instead of "."

       xx-vers = a.b.c.d             // Version info, e.g. 4.3.1.2
       xx-date = dd.mm.yyyy          // Version date
       xx-setup = setupfile.exe      // Name of setup file
       xx-size = nn MB               // Size of setup file
                                     // xx = package dependent prefix
       -->
       ...  HTML with placeholders [%xx-yyyy]
   Vers. 3.2 (Aug. 2011): Delphi XE2
                    Optional format settings in header,
                    Creation and distribution of file with MD5 checksumm
   Vers. 3.7 (Jul. 2013):
                    File with version info (*.ver) splitted into
                      *.ver: only version number
                      *.loc: download location
                    All project files are relative to task file
   Vers. 3.7.3 (Dec. 2013):
                    New command line options:
                    /continue - close status window automatically
                    /append   - append to existing log file
   Vers. 4.0 (June 2016): Delphi 10 Seattle
                    MD5 is copied to destinations (if selected)
   Vers. 4.1 (July 2016): generate OpenPGP signature files
   Vers. 4.2 (July 2017): Error fixes
   Vers. 4.3 (Aug. 2017): FTPS support
   Vers. 4.4 (July 2018): added: SHA256 checksum
             (Feb. 2019): compatibilty to gpg4win 3.1.5
   Vers. 4.5 (Aug. 2020): rearranged FTP units, template processing changed
   Vers. 4.6 (Sep. 2022): new button "replace file"
   Vers. 4.7 (Apr. 2024): ini and vcj files changed to Unicode

   last modifed: July 2023
   *)

unit VersionCopyMain;

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes,
  Vcl.Graphics, Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ExtCtrls,
  Vcl.ComCtrls, Vcl.Buttons, CBFunctions, TransferStat, FileConsts,
  FtpUtils, FtpCopyUtils, CheckLst, Indicators, Vcl.Menus;

const
  ProgName = 'VersionCopy';
  Vers = ' - Vers. 4.7.0';
  CopRgt = ' 2008-2023 - Dr. J. Rathlev - D-24222 Schwentinental';
  EmailAdr = 'kontakt(a)rathlev-home.de';

  GpgKey = 'Software\GnuPG';
  GpgCommand = 'gpg -sb -u %s --output %s %s';

  VerExt  = 'ver';
  LocExt  = 'loc';
  HtmlExt = 'html';
  TxtExt  = 'txt';
  Md5Ext  = 'md5';
  Sha256Ext = 'sha256';
  LogExt  = 'log';
  SigExt  = 'sig';
  phBegin = '[%';
  phEnd   = ']';

  MaxSect = 100;
  MaxUserPwdLength = 64;

  // SSL-Zertifikate
  CertConfig = 'certfiles.cfg';
// is a text file in ini format:
//  [certificate]
//  RootCertFile = root.pem
//  CertFile     = cert.crt
//  KeyFile      = key.key
//  SSLPassword  = password

// Schlssel zum Kodieren der Passwrter in der Ini-Datei
{$I ..\..\Common\CopyPwdKey.inc }
// Die Include-Datei enthlt folgende Definition:
// PwdKey = '<keystring>';
// <keystring> kann mit der Routine "EncodeString" aus der Units "Crypt"
// aus einem beliebigen String erzeugt werden

type
  TVarField = class (TObject)
    FID  : integer;
    FString : string;
    constructor Create (AID : integer; AString : string);
    end;

  TMainForm = class(TForm)
    PageControl: TPageControl;
    tsSettings: TTabSheet;
    tsTargets: TTabSheet;
    edExeFile: TLabeledEdit;
    edSetupFile: TLabeledEdit;
    rgLevel: TRadioGroup;
    rgKeep: TRadioGroup;
    btnLoad: TBitBtn;
    btnSaveAs: TBitBtn;
    paTop: TPanel;
    btnExeFile: TBitBtn;
    btnSetup: TBitBtn;
    Label1: TLabel;
    laFtpServer: TLabel;
    btnAddDirs: TSpeedButton;
    btnRemDirs: TSpeedButton;
    btnAddFtpTarget: TSpeedButton;
    btnRemFtpTarget: TSpeedButton;
    btnNew: TBitBtn;
    btnCopy: TBitBtn;
    laTargetHint: TLabel;
    lbFtp: TCheckListBox;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    paBottom: TPanel;
    btnQuit: TBitBtn;
    btnInfo: TBitBtn;
    lbDirs: TCheckListBox;
    tsHtml: TTabSheet;
    lvHtml: TListView;
    btnAddHtml: TSpeedButton;
    btnRemHtml: TSpeedButton;
    tsFiles: TTabSheet;
    Label5: TLabel;
    btnAddFiles: TSpeedButton;
    btnRemFiles: TSpeedButton;
    Label6: TLabel;
    Label7: TLabel;
    edProjPrefix: TEdit;
    lbFiles: TCheckListBox;
    btnTest: TBitBtn;
    Label8: TLabel;
    cbTaskName: TComboBox;
    edVersionSetup: TLabeledEdit;
    edVersionInfoFile: TLabeledEdit;
    btnVersionInfoFile: TBitBtn;
    tsFTP: TTabSheet;
    Label4: TLabel;
    edServer: TLabeledEdit;
    lbFtpServer: TListBox;
    edPort: TLabeledEdit;
    edDir: TLabeledEdit;
    edUser: TLabeledEdit;
    btnAddFtpServer: TSpeedButton;
    btnRemFtpServer: TSpeedButton;
    btnEditFtpServer: TSpeedButton;
    laPass: TLabel;
    btnViewLog: TBitBtn;
    cbHash: TCheckBox;
    Label10: TLabel;
    cbGpgSig: TCheckBox;
    btnSave: TBitBtn;
    laStatus: TLabel;
    pmTasks: TPopupMenu;
    pmiEdit: TMenuItem;
    pmiClear: TMenuItem;
    btnReplFile: TSpeedButton;
    btnReplHtml: TSpeedButton;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormShow(Sender: TObject);
    procedure btnQuitClick(Sender: TObject);
    procedure btnInfoClick(Sender: TObject);
    procedure btnExeFileClick(Sender: TObject);
    procedure btnNewClick(Sender: TObject);
    procedure btnLoadClick(Sender: TObject);
    procedure btnSaveAsClick(Sender: TObject);
    procedure btnSetupClick(Sender: TObject);
    procedure rgLevelClick(Sender: TObject);
    procedure btnAddDirsClick(Sender: TObject);
    procedure btnRemDirsClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnAddFtpServerClick(Sender: TObject);
    procedure btnRemFtpServerClick(Sender: TObject);
    procedure btnCopyClick(Sender: TObject);
    procedure btnAddHtmlClick(Sender: TObject);
    procedure btnRemHtmlClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure lbDirsDblClick(Sender: TObject);
    procedure lbFtpDblClick(Sender: TObject);
    procedure lvHtmlDblClick(Sender: TObject);
    procedure PageControlChange(Sender: TObject);
    procedure btnAddFilesClick(Sender: TObject);
    procedure btnRemFilesClick(Sender: TObject);
    procedure btnTestClick(Sender: TObject);
    procedure cbTaskNameCloseUp(Sender: TObject);
    procedure btnVersionInfoFileClick(Sender: TObject);
    procedure lbFtpServerDblClick(Sender: TObject);
    procedure lbFtpServerClick(Sender: TObject);
    procedure btnAddFtpTargetClick(Sender: TObject);
    procedure btnRemFtpTargetClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure btnViewLogClick(Sender: TObject);
    procedure cbGpgSigClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure pmiEditClick(Sender: TObject);
    procedure pmiClearClick(Sender: TObject);
    procedure btnReplFileClick(Sender: TObject);
  private
    { Private-Deklarationen }
    ProgVersName,
    ProgVersDate,
    UserPath,AppPath,
    IniName,LogName,
    Version,NewTask,
    CmdTaskName,PgpID,
    TaskName,TaskPath,
    LastFile,LastExe,
    LastSetup,LastTempl,
    LastTarget,
    RootCertFile,
    CertFile,
    KeyFile          : string;
    SSLPassword      : AnsiString;
    FTP              : TExtFTP;
    CopyThread       : TCopyThread;
    CopyFiles,LogOn,
    Modified,Force,
    WaitPrompt,Append,
    GpgInstalled,
    Connecting       : boolean;
    FSize            : int64;
    LogFile          : TextFile;
    DlgPos           : TPoint;
    function GetDlgPos : TPoint;
    function GetSSLPwd : boolean;
    procedure SetTaskName (const FName : string);
    procedure ClearTask;
    function LoadTask : boolean;
    procedure SaveTask (const Filename : string);
    procedure UpdateFtpServer;
    function NewID : integer;
    function MakeFtpHint (FtpID : integer; const Dir : string) : string;
    function CheckModified : boolean;
    function SaveTaskAs : boolean;
    function AskForSave : boolean;
    function OpenLog (Append : boolean) : boolean;
    procedure CloseLog;
    procedure InitLog(AAction : string);
    procedure WriteLog (Indent : integer; AText : string);
    procedure UpdateActionInfo(AAction : string);
    procedure UpdateStatus(AStatus : string);
    procedure CancelCopy(var Msg: TMessage); message wmCancelAction;
    procedure ShowProgress (AAction : TFileAction; ACount: int64; ASpeedLimit : boolean);
    function ProcessHtmlTemplate(const TemplName,Version,SuName,Md5,Sha : string) : string;
    function GetVersionSuffix(t : string; n : integer) : string;
    function GetVersionName(t : string; n : integer) : string;
    function ProcessFiles : boolean;
    function StartFtpCopy (AFtpPar: TFtpParams; const Dir,Mask,SName,VName,LName : string;
                            Version,AddFiles : boolean; var fc : integer) : boolean;
  public
    { Public-Deklarationen }
    function CheckRemote : boolean;
  end;

var
  MainForm: TMainForm;

implementation

{$R *.dfm}

uses System.IniFiles, System.Win.Registry, System.StrUtils, FileErrors,
  StringUtils, Crypt, WinApiUtils, IdFtp, GnuGetText, LangUtils, WinExecute,
  NumberUtils, WinUtils, MsgDialogs, PathUtils, FileUtils, WinDevUtils,
  FtpDlg, FtpConsts, ExtSysUtils, ShellDirDlg, PwdDlg, EditHistListDlg, SelectFtpDlg,
  ShowMemo, InitProg, UnitConsts, Hashes, IniFileUtils;

const
  ColSpace = ': ';
  FSep = '|';

{ ------------------------------------------------------------------- }
constructor TVarField.Create (AID : integer; AString : string);
begin
  inherited Create;
  FID:=AID;
  FString:=AString;
  end;

{ ------------------------------------------------------------------- }
function ReconnectPath (const Path : string) : boolean;
var
  nl       : dword;
  NetRes   : TNetResource;
  nn       : pchar;
begin
  nl:=1024; nn:=StrAlloc(nl);
  with NetRes do begin
    dwScope:=0; dwDisplayType:=0;
    lpProvider:=''; lpLocalName:='';
    dwUsage:=RESOURCEUSAGE_CONNECTABLE;
    dwType:=RESOURCETYPE_DISK;
    if (copy(Path,1,2)='\\') then begin  // Netzwerkumgebung
      lpRemoteName:=pchar(ExcludeTrailingPathDelimiter(Path));
      end
    else begin
      lpLocalName:=pchar(ExtractFileDrive(Path));
      WNetGetConnection(lpLocalName,nn,nl);   // Netzwerkname des Pfads
      lpRemoteName:=nn;
      end;
    end;
  // als angemeldeter Benutzer versuchen
  Result:=WNetAddConnection2(NetRes,nil,nil,0)=NO_ERROR;
  end;

{ ------------------------------------------------------------------- }
const
  IniExt = 'ini';

  CfgSekt    = 'Config';
  StwSekt    = 'StatusWindow';

  iniTask   = 'Task';
  iniTCount = 'TaskCount';
  iniLast   = 'LastTask';
  iniDir    = 'LastDir';
  iniExe    = 'LastExe';
  iniSetup  = 'LastSetup';
  iniTarget = 'LastTarget';
  iniTempl  = 'LastTemplate';

  CertSekt = 'certificate';
  iniRootCert = 'RootCertFile';
  iniCertFile = 'CertFile';
  iniKeyFile  = 'KeyFile';
  iniPassword  = 'SSLPassword';

procedure TMainForm.FormCreate(Sender: TObject);
var
  s,sn    : string;
  i,n     : integer;
begin
  TranslateComponent(self);
  InitPaths(AppPath,UserPath);
  InitVersion(ProgName,Vers,CopRgt,3,3,ProgVersName,ProgVersDate);
  Caption:=_('Copy setups with version info')+' ('+VersInfo.Comments+')';
  IniName:=Erweiter(AppPath,PrgName,IniExt);
  CmdTaskName:=''; Force:=false; Modified:=false;
  CopyFiles:=true; WaitPrompt:=true; Append:=false;
  if ParamCount>0 then for i:=1 to ParamCount do begin
    s:=ParamStr(i);
    if s[1]='/' then begin
      Delete(s,1,1);
      if CompareOption(s,'force') then Force:=true;
      if CompareOption(s,'continue') then WaitPrompt:=false;
      if CompareOption(s,'append') then Append:=true;
      if CompareOption(s,'test') then CopyFiles:=false;
      end
    else CmdTaskName:=s;
    end;
  sn:=SetDirName(PrgPath)+CertConfig;
  if FileExists(sn) then with TMemIniFile.Create(sn) do begin
    RootCertFile:=ReadString(CertSekt,iniRootCert,'');
    CertFile:=ReadString(CertSekt,iniCertFile,'');
    KeyFile:=ReadString(CertSekt,iniKeyFile,'');
    SSLPassword:=ReadString(CertSekt,iniPassword,'');
    end
  else begin
    RootCertFile:=''; CertFile:='';
    KeyFile:=''; SSLPassword:='';
    end;
  with TUnicodeIniFile.CreateForRead(IniName) do begin
    TaskName:=ReadString(CfgSekt,iniLast,'');
    TaskPath:=ReadString(CfgSekt,iniDir,UserPath);
    LastExe:=ReadString(CfgSekt,iniExe,TaskPath);
    LastSetup:=ReadString(CfgSekt,iniSetup,TaskPath);
    LastTarget:=ReadString(CfgSekt,iniTarget,'');
    LastTempl:=ReadString(CfgSekt,iniTempl,'');
    n:=ReadInteger(CfgSekt,iniTCount,0);
    for i:=0 to n-1 do begin
      s:=ReadString(CfgSekt,iniTask+ZStrint(i,2),'');
      if length(s)>0 then cbTaskName.Items.Add(s);
      end;
    Free;
    end;
  // Cgheck if GnuPG is installed
  with TRegistry.Create(KEY_READ) do begin
    RootKey:=HKEY_LOCAL_MACHINE;
    GpgInstalled:=KeyExists(GpgKey);
//    CloseKey;
    Free;
    end;
  lbDirs.MultiSelect:=true;
  Version:=''; LastFile:=''; LogOn:=true; LogName:='';
  NewTask:=_('[New Task]');
  lbFiles.MultiSelect:=true;
  ExitCode:=0;
  end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  i       : integer;
begin
  with cbTaskName.Items do for i:=Count-1 downto 0 do
    if Strings[i]=NewTask then Delete(i);
  with TUnicodeIniFile.CreateForWrite(IniName,true) do begin
    WriteString(CfgSekt,iniLast,TaskName);
    WriteString(CfgSekt,iniDir,TaskPath);
    WriteString(CfgSekt,iniExe,LastExe);
    WriteString(CfgSekt,iniSetup,LastSetup);
    WriteString(CfgSekt,iniTarget,LastTarget);
    WriteString(CfgSekt,iniTempl,LastTempl);
    with cbTaskName.Items do begin
      WriteInteger(CfgSekt,iniTCount,Count);
      for i:=0 to Count-1 do WriteString(CfgSekt,iniTask+ZStrint(i,2),Strings[i]);
      end;
    Free;
    end;
  end;

function  TMainForm.GetSSLPwd : boolean;
begin
  if (length(RootCertFile)>0) and (length(CertFile)>0) and (length(KeyFile)>0) then begin
    if (length(SSLPassword)>0) then Result:=true
    else Result:=PasswordDialog.Execute(_('Secure FTP connection'),
      _('Enter the password to read the private key'+sLineBreak+'required to establish the SSL connection!'),SSLPassword)=mrOk;
    end
  else Result:=true;
  end;

procedure TMainForm.SetTaskName (const FName : string);
begin
  if ContainsFullPath(FName) then begin     // Befehlszeile
    TaskPath:=ExtractFilePath(FName);
    TaskName:=FName;
    SetCurrentDir(TaskPath);
    end
  else begin
    TaskPath:=SetDirName(GetCurrentDir);
    if length(FName)>0 then TaskName:=MakeAbsolutePath(TaskPath,FName)
    else TaskName:='';
    end;
  end;

function TMainForm.CheckRemote : boolean;
begin
  if GetSSLPwd then begin
    FTP:=TExtFTP.Create(self,PrgPath,RootCertFile,CertFile,KeyFile,SSLPassword);
    Result:=Force and (length(CmdTaskName)>0);
    if Force then begin
      SetTaskName(CmdTaskName);
      if LoadTask then begin
        with TransferStatus do begin
          LoadFromIni(IniName,StwSekt);
          DlgPos:=Point(Left+Width,Top+Height-150);
          end;
        if not ProcessFiles then ExitCode:=2;
        end
      else begin
        ErrorDialog(GetDlgPos,_('File not found:')+Space+CmdTaskName);
        ExitCode:=1;
        end;
      end;
    end
  else Result:=true;
  end;

procedure TMainForm.FormShow(Sender: TObject);
begin
  with TransferStatus do begin
    LoadFromIni(IniName,StwSekt);
    DlgPos:=Point(Left+Width,Top+Height-150);
    end;
  FtpDialog.Init(IniName,'');
  SetTaskName(CmdTaskName);
  if not FileExists(TaskName) then with cbTaskName do if (Items.Count>0) then begin
    SetTaskName(Items[0]);
    end;
  if not LoadTask then ClearTask;
  cbTaskName.ItemIndex:=0;
  PageControl.ActivePageIndex:=0;
  end;

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  AskForSave;
  end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  FreeAndNil(Ftp);
  ClearTask;
  end;

procedure TMainForm.FormResize(Sender: TObject);
var
  n : integer;
begin
  n:=edSetupFile.Width div 2 -5;;
  rgLevel.Width:=n;
  with rgKeep do begin
    Width:=n; Left:=rgLevel.Left+n+9;
    end;
  with lvHtml do begin
    n:=(Width-22) div 2;
    Columns[0].Width:=n;
    Columns[1].Width:=n;
    end;
  n:=(tsTargets.Height-75) div 2;
  lbDirs.Height:=n;
  with lbFtp do begin
    Height:=n; Top:=LbDirs.Top+n+24;
    end;
  laFtpServer.Top:=lbFtp.Top-15;
  with lbDirs do n:=Top+Height;
  with btnRemDirs do Top:=n-Height;
  with btnAddDirs do Top:=btnRemDirs.Top-5-Height;
  end;

procedure TMainForm.btnQuitClick(Sender: TObject);
begin
  Close;
  end;

procedure TMainForm.btnInfoClick(Sender: TObject);
begin
  InfoDialog(BottomRightPos(btnInfo),ProgName,ProgVersName+' - '+ProgVersDate+#13+
           VersInfo.CopyRight+#13'D-24222 Schwentinental');
  end;

procedure TMainForm.btnViewLogClick(Sender: TObject);
begin
  if FileExists(LogName) then begin
    ShowTextDialog.Execute (Caption,LogName,'','','','','','',
       Format(_('Log files|*.%s|all|*.*'),[LogExt]),
       TopRightPos(btnSave,Point(20,0)),0,stModal,
       [sbOpen,sbPrint,sbErase,sbSearch]);
    end
  else ErrorDialog(GetDlgPos,_('Log file not found!'));
  end;

function TMainForm.GetDlgPos : TPoint;
begin
  if TransferStatus.Visible then Result:=DlgPos
  else Result:=CursorPos;
  end;

{ ------------------------------------------------------------------- }
function TMainForm.OpenLog (Append : boolean) : boolean;
begin
  if length(LogName)=0 then LogName:=NewExt(IniName,LogExt);
  AssignFile(LogFile,LogName);
  if Append and FileExists(LogName) then {$I-} system.append(LogFile) {$I+}
  else {$I-} rewrite(LogFile) {$I+};
  Result:=IOResult<>0;
  LogOn:=not Result;
  end;

procedure TMainForm.CloseLog;
begin
  if LogOn then begin
    writeln (LogFile);
    writeln(LogFile,_('Finished:')+Space+DateTimeToStr(Now));
    writeln (LogFile);
    CloseFile (LogFile);
    end;
  end;

procedure TMainForm.InitLog(AAction : string);
begin
  WriteLog(0,'====================================');
  with VersInfo do WriteLog(0,InternalName+' (Vers. '+Version+')');
  WriteLog(0,AAction+ColSpace+DateTimeToStr(Now));
  end;

procedure TMainForm.WriteLog (Indent : integer; AText : string);
begin
  if LogOn then begin
    writeln (LogFile,FillSpace(Indent),AText);
    Flush(LogFile);
    end;
  end;

{ ------------------------------------------------------------------- }
const
  TaskExt = 'vcj';

  GlobSekt    = 'Global';
  FtpSekt     = 'Ftp';
  FileSekt    = 'Files';
  DirSekt     = 'Directory';
  FtpTSekt    = 'FtpTarget';
  WwwSekt     = 'WWWTarget';

  vcjVersFile = 'VersionFile';
  vcjSuFile   = 'SetupFile';
  vcjVersLev  = 'VersionLevel';
  vcjVersKeep = 'KeepVersion';
  vcjPrefix   = 'Prefix';
  vcjVersInfo = 'VersionInfoFile';
  vcjFtpCount = 'FtpCount';
  vcjFilCount = 'FileCount';
  vcjFtpID    = 'ID';
  vcjFTgCount = 'FtpTargetCount';
  vcjDirCount = 'DirCount';
  vcjWWWCount = 'WWWTargetCount';
  vcjDName    = 'Name';
  vcjLevInfo  = 'Levelinfo';
  vcjPort     = 'Port';
  vcjHost     = 'Host';
  vcjDir      = 'Directory';
  vcjSecure   = 'UseFTPS';
  vcjUser     = 'Username';
  vcjPwd      = 'Password';
  vcjPassive  = 'Passive';
  vcjUseHost  = 'UseHost';
  vcjUtf8     = 'ForceUtf8';
  vcjFtpRef   = 'FtpConnection';
  vcjCMode    = 'ChangeNames';
  vcjPrxServ  = 'ProxyServer';
  vcjPrxPort  = 'ProxyPort';
  vcjPrxUser  = 'ProxyUsername';
  vcjPrxPwd   = 'ProxyPassword';
  vcjPrxType  = 'ProxyType';
  vcjTemplate = 'Template';
  vcjMd5      = 'CreateMd5';
  vcjUsePgp   = 'CreatePgp';
  vcjPgpID    = 'PgpID';

function TMainForm.LoadTask : boolean;
var
  i,n,k,m   : integer;
  s,ss,st   : string;
  fp        : TFtpSettings;
  IniFile   : TUnicodeIniFile;

  function ReadFtpPar (const Sekt : string; var AFtpPar : TFtpParams) : string;
  begin
    with IniFile,AFtpPar do begin
      Host:=ReadString(Sekt,vcjHost,'');
      Port:=ReadInteger(Sekt,vcjPort,defFtpPort);
      Username:=ReadString(Sekt,vcjUser,'');
      Password:=DecryptPwdString(DecodeString(PwdKey),ReadString(Sekt,vcjPwd,''),teAscii85);
      Directory:=ReadString(Sekt,vcjDir,'');
      SecureMode:=ReadInteger(Sekt,vcjSecure,0);
      Passive:=ReadBool(Sekt,vcjPassive,false);
      UseHost:=ReadBool(Sekt,vcjUseHost,false);
      ForceUtf8:=ReadBool(Sekt,vcjUtf8,false);
      CaseMode:=TTextChange(ReadInteger (Sekt,vcjCMode,integer(tcNone)));
      with Proxy do begin
        Server:=ReadString(Sekt,vcjPrxServ,'');
        Port:=ReadInteger(Sekt,vcjPrxPort,defProxyPort);
        Username:=ReadString(Sekt,vcjPrxUser,'');
        Password:=DecryptPwdString(DecodeString(PwdKey),ReadString(Sekt,vcjPrxPwd,''),teAscii85);
        Mode:=TIdFtpProxyType(ReadInteger(Sekt,vcjPrxType,0));
        end;
      if (length(Host)>0) and (length(Username)>0) then Result:=GetFtpHint(AFtpPar)
      else Result:='';
      end;
    end;

begin
  if FileExists(TaskName) then begin
    cbGpgSig.Enabled:=false;
    LogName:=NewExt(TaskName,LogExt);
    AddToHistory(cbTaskName.Items,TaskName);
    cbTaskName.ItemIndex:=0;
    IniFile:=TUnicodeIniFile.CreateForRead(TaskName);
    with IniFile do begin
      edExeFile.Text:=ExpandPath(ReadString(GlobSekt,vcjVersFile,''));
      edSetupFile.Text:=ExpandPath(ReadString(GlobSekt,vcjSuFile,''));
      edProjPrefix.Text:=ReadString(GlobSekt,vcjPrefix,'');
      edVersionInfoFile.Text:=ExpandPath(DelExt(ReadString(GlobSekt,vcjVersInfo,'')));
      rgLevel.ItemIndex:=ReadInteger(GlobSekt,vcjVersLev,4)-1;
      rgKeep.ItemIndex:=ReadInteger(GlobSekt,vcjVersKeep,1);
      cbHash.Checked:=ReadBool(GlobSekt,vcjMd5,false);
      cbGpgSig.Checked:=ReadBool(GlobSekt,vcjUsePgp,false);
      PgpID:=ReadString(GlobSekt,vcjPgpID,'');
      n:=ReadInteger(GlobSekt,vcjFtpCount,0);
      with lbFtpServer do begin
        FreeListObjects(Items);
        Clear;
        end;
      for i:=0 to n-1 do begin
        ss:=FtpSekt+ZStrint(i,3);
        n:=ReadInteger(ss,vcjFtpID,i);
        fp:=TFtpSettings.Create(n);
        with fp do begin
          s:=ReadFtpPar(ss,FtpPar);
          if length(s)>0 then lbFtpServer.AddItem(s,fp)
          else Free;
          end;
        end;
      n:=ReadInteger(GlobSekt,vcjFilCount,0);  // additional files
      lbFiles.Clear;
      for i:=0 to n-1 do begin
        ss:=FileSekt+ZStrint(i,3);
        s:=ExpandPath(ReadString(ss,vcjDName,''));
        if length(s)>0 then with lbFiles do
          Checked[Items.Add(s)]:=ReadBool(ss,vcjPrefix,false);
        end;
      n:=ReadInteger(GlobSekt,vcjDirCount,0);   // local destination directories
      lbDirs.Clear;
      for i:=0 to n-1 do begin
        ss:=DirSekt+ZStrint(i,3);
        s:=ReadString(ss,vcjDName,'');
        if length(s)>0 then with lbDirs do
          Checked[Items.Add(s)]:=ReadBool(ss,vcjLevInfo,false);
        end;
      with lbFtp do begin
        FreeListObjects(Items);
        Clear;
        end;
      n:=ReadInteger(GlobSekt,vcjFTgCount,0);   // FTP destinations
      for i:=0 to n-1 do begin
        ss:=FtpTSekt+ZStrint(i,3);
        s:=ReadString(ss,vcjDName,'');
        k:=ReadInteger(ss,vcjFtpRef,-1);  // FTP ID
        if (k>=0) then with lbFtp do begin
          m:=Items.AddObject('',TVarField.Create(k,s));
          Items[m]:=MakeFtpHint(k,s);
          Checked[m]:=ReadBool(ss,vcjLevInfo,false);
          end;
        end;
      with lvHtml do begin
        FreeListViewData(Items);
        Clear;
        end;
      n:=ReadInteger(GlobSekt,vcjWWWCount,0);   // WWW destinations
      for i:=0 to n-1 do begin
        ss:=WWWSekt+ZStrint(i,3);
        st:=ReadString(ss,vcjTemplate,'');
        s:=ReadString(ss,vcjDName,'');
        k:=ReadInteger(ss,vcjFtpRef,-1);     // FTP ID
        if (k>=0) then with lvHtml.Items.Add do begin
          Caption:=st;
          SubItems.Add(MakeFtpHint(k,s));
          Data:=TVarField.Create(k,s);
          end;
        end;
      Free;
      end;
    if not GetFileVersionString(edExeFile.Text,Version) then
      ErrorDialog(GetDlgPos,Format(_('No version info found for:')+sLineBreak+'"%s"',
                         [edExeFile.Text]));
    Modified:=false;
    Result:=true;
    cbGpgSig.Enabled:=GpgInstalled;
    end
  else Result:=false;
  end;

procedure TMainForm.SaveTask (const Filename : string);
var
  i  : integer;
  ss : string;
  IniFile : TUnicodeIniFile;

  procedure WriteFtpPar (const Sekt : string; AFtpPar : TFtpParams);
  begin
    with IniFile,AFtpPar do begin
      if (length(Host)>0) and (length(Username)>0) then begin
        WriteString(Sekt,vcjHost,Host);
        WriteInteger(Sekt,vcjPort,Port);
        WriteString(Sekt,vcjUser,Username);
        WriteString(Sekt,vcjPwd,EncryptPwdString(DecodeString(PwdKey),Password,MaxUserPwdLength,teAscii85));
        WriteString(Sekt,vcjDir,Directory);
        WriteInteger(Sekt,vcjSecure,SecureMode);
        WriteBool(Sekt,vcjPassive,Passive);
        WriteBool(Sekt,vcjUseHost,UseHost);
        WriteBool(Sekt,vcjUtf8,ForceUtf8);
        WriteInteger(Sekt,vcjCMode,integer(CaseMode));
        with Proxy do begin
          WriteString(Sekt,vcjPrxServ,Server);
          WriteInteger(Sekt,vcjPrxPort,Port);
          WriteString(Sekt,vcjPrxUser,Username);
          WriteString(Sekt,vcjPrxPwd,EncryptPwdString(DecodeString(PwdKey),Password,MaxUserPwdLength,teAscii85));
          WriteInteger(Sekt,vcjPrxType,integer(Mode));
          end;
        end;
      end;
    end;

begin
  TaskName:=Filename;
  TaskPath:=ExtractFilePath(TaskName);
  IniFile:=TUnicodeIniFile.CreateForWrite(TaskName,true);
  with IniFile do begin
    WriteString(GlobSekt,vcjVersFile,ExtractRelativePath(TaskPath,edExeFile.Text));
    WriteString(GlobSekt,vcjSuFile,ExtractRelativePath(TaskPath,edSetupFile.Text));
    WriteString(GlobSekt,vcjPrefix,edProjPrefix.Text);
    WriteString(GlobSekt,vcjVersInfo,ExtractRelativePath(TaskPath,edVersionInfoFile.Text));
    WriteInteger(GlobSekt,vcjVersLev,rgLevel.ItemIndex+1);
    WriteInteger(GlobSekt,vcjVersKeep,rgKeep.ItemIndex);
    WriteBool(GlobSekt,vcjMd5,cbHash.Checked);
    WriteBool(GlobSekt,vcjUsePgp,cbGpgSig.Checked);
    WriteString(GlobSekt,vcjPgpID,PgpID);
    with lbFtpServer do begin
      WriteInteger(GlobSekt,vcjFtpCount,Items.Count);
      for i:=0 to Items.Count-1 do begin
        ss:=FtpSekt+ZStrint(i,3);
        with (Items.Objects[i] as TFtpSettings) do begin
          WriteInteger(ss,vcjFtpID,ID);
          WriteFtpPar(ss,FtpPar);
          end;
        end;
      end;
    with lbFiles do begin
      WriteInteger(GlobSekt,vcjFilCount,Count);
      for i:=0 to Count-1 do begin
        ss:=FileSekt+ZStrint(i,3);
        WriteString(ss,vcjDName,ExtractRelativePath(TaskPath,Items[i]));
        WriteBool(ss,vcjPrefix,Checked[i]);
        end;
      end;
    with lbDirs do begin
      WriteInteger(GlobSekt,vcjDirCount,Items.Count);
      for i:=0 to Items.Count-1 do begin
        ss:=DirSekt+ZStrint(i,3);
        WriteString(ss,vcjDName,Items[i]);
        WriteBool(ss,vcjLevInfo,Checked[i]);
        end;
      end;
    with lbFtp do begin
      WriteInteger(GlobSekt,vcjFTgCount,Items.Count);
      for i:=0 to Items.Count-1 do begin
        ss:=FtpTSekt+ZStrint(i,3);
        with Items.Objects[i] as TVarField do begin
          WriteInteger(ss,vcjFtpRef,FID);
          WriteString(ss,vcjDName,FString);
          end;
        WriteBool(ss,vcjLevInfo,Checked[i]);
        end;
      end;
    with lvHtml do begin
      WriteInteger(GlobSekt,vcjWWWCount,Items.Count);
      for i:=0 to Items.Count-1 do with Items[i] do begin
        ss:=WWWSekt+ZStrint(i,3);
        EraseSection(ss);
        WriteString(ss,vcjTemplate,Caption);
        with TVarField((Data)) do begin
          WriteInteger(ss,vcjFtpRef,FID);
          WriteString(ss,vcjDName,FString);
          end;
        end;
      end;
    Free;
    end;
  Modified:=false;
  edExeFile.Modified:=false;
  edSetupFile.Modified:=false;
  edProjPrefix.Modified:=false;
  end;

function TMainForm.MakeFtpHint (FtpID : integer; const Dir : string) : string;
var
  n : integer;
begin
  n:=SearchForID(lbFtpServer.Items, FtpID);
  with lbFtpServer.Items do if (n>=0) and (n<Count) then begin
    Result:=GetFtpHint((Objects[n] as TFtpSettings).FtpPar);
    if length(Dir)>0 then Result:=IncludeTrailingSlash(Result)+Dir;
    end
  else Result:='';
  end;

procedure TMainForm.ClearTask;
var
  i : integer;
begin
  AddToHistory(cbTaskName.Items,NewTask);
  cbTaskName.ItemIndex:=0;
  TaskName:='';
  edExeFile.Text:='';
  edSetupFile.Text:='';
  rgLevel.ItemIndex:=3;
  rgKeep.ItemIndex:=1;
  edProjPrefix.Text:='';
  lbFiles.Clear;
  lbDirs.Clear;
  with lbFtpServer do begin
    FreeListObjects(Items);
    Clear;
    end;
  with lbFtp do begin
    FreeListObjects(Items);
    Clear;
    end;
  with lvHtml do begin
    for i:=0 to Items.Count-1 do with Items[i] do TObject(Data).Free;
    Clear;
    end;
  Modified:=false;
  edExeFile.Modified:=false;
  edSetupFile.Modified:=false;
  edProjPrefix.Modified:=false;
  end;

function TMainForm.CheckModified : boolean;
begin
  Result:=Modified or edExeFile.Modified or edSetupFile.Modified or edProjPrefix.Modified;
  end;

function TMainForm.SaveTaskAs : boolean;
begin
  with SaveDialog do begin
    InitialDir:=TaskPath;
    Filename:=ExtractFilename(TaskName);
    Filter:=_('Copy tasks')+'|*.'+TaskExt;
    DefaultExt:=TaskExt;
    Title:=_('Save version copy task');
    if Execute then begin
      SaveTask(FileName);
      AddToHistory(cbTaskName.Items,TaskName);
      cbTaskName.ItemIndex:=0;
      Result:=true;
      end
    else Result:=false;
    end;
  end;

function TMainForm.AskForSave : boolean;
begin
  Result:=true;
  if CheckModified and
    ConfirmDialog(BottomLeftPos(btnQuit),_('Save modified task?')) then SaveTask(TaskName);
  end;

procedure TMainForm.btnNewClick(Sender: TObject);
begin
  if AskForSave then begin
    ClearTask;
    end;
  end;

procedure TMainForm.btnLoadClick(Sender: TObject);
begin
  with OpenDialog do begin
    if length(Taskname)>0 then InitialDir:=ExtractFilePath(Taskname)
    else InitialDir:=TaskPath;
    Filename:='';
    Filter:=_('Copy tasks')+'|*.'+TaskExt;
    Title:=_('Select version copy task');
    Options:=Options-[ofAllowMultiSelect,ofFileMustExist];
    if Execute then begin
      SetTaskName(Filename);
      LoadTask;
      cbTaskName.ItemIndex:=0;
      PageControl.ActivePageIndex:=0;
      end;
    end;
  end;

procedure TMainForm.cbGpgSigClick(Sender: TObject);
begin
  with cbGpgSig do if Enabled and Checked then begin
    InputQuery(cbGpgSig.Caption,_('Key ID:'),PgpID);
    end;
  end;

procedure TMainForm.cbTaskNameCloseUp(Sender: TObject);
begin
  SetTaskName(cbTaskName.Text);
  LoadTask;
  PageControl.ActivePage:=tsSettings;
  end;

procedure TMainForm.btnSaveAsClick(Sender: TObject);
begin
  SaveTaskAs;
  end;

procedure TMainForm.btnSaveClick(Sender: TObject);
begin
  SaveTask(TaskName);
  end;

procedure TMainForm.btnExeFileClick(Sender: TObject);
var
  v : string;
begin
  with OpenDialog do begin
    if length(edExeFile.Text)>0 then InitialDir:=ExtractFilePath(edExeFile.Text)
    else InitialDir:=LastExe;
    Filename:='';
    Filter:=_('Exe files')+'|*.exe'+FSep+_('All files')+'|*.*';
    Title:=_('Select exe file with version info');
    Options:=Options-[ofAllowMultiSelect]+[ofFileMustExist];
    if Execute then begin
      if GetFileVersionString(Filename,v) then begin
        edExeFile.Text:=Filename;
        Version:=v;
        Modified:=true;
        end
      else begin
        ErrorDialog(GetDlgPos,Format(_('No version info found for:')+sLineBreak+'"%s"',
                           [Filename]));
        end;
      end;
    end;
  end;

procedure TMainForm.btnSetupClick(Sender: TObject);
begin
  with OpenDialog do begin
    if length(edSetupFile.Text)>0 then InitialDir:=ExtractFilePath(edSetupFile.Text)
    else InitialDir:=LastSetup;
    Filename:='';
    Filter:=_('Exe files')+'|*.exe'+FSep+_('All files')+'|*.*';
    Title:=_('Select setup file');
    Options:=Options-[ofAllowMultiSelect,ofFileMustExist];
    if Execute then begin
      edSetupFile.Text:=Filename;
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.btnVersionInfoFileClick(Sender: TObject);
var
  s : string;
begin
  with OpenDialog do begin
    if length(edVersionInfoFile.Text)=0 then s:=NewExt(edSetupFile.Text,VerExt)
    else s:=NewExt(edVersionInfoFile.Text,VerExt);
    InitialDir:=ExtractFilePath(s);
    Filename:=ExtractFileName(s);
    Filter:=_('Version infos')+'|*.'+VerExt+FSep+_('All files')+'|*.*';
    Title:=_('Select version info file');
    Options:=Options-[ofAllowMultiSelect,ofFileMustExist];
    if Execute then begin
      edVersionInfoFile.Text:=DelExt(Filename);
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.rgLevelClick(Sender: TObject);
begin
  Modified:=true;
  end;

procedure TMainForm.PageControlChange(Sender: TObject);
begin
  if PageControl.ActivePage=tsFtp then begin
    with lbFtpServer do if Items.Count>0 then begin
      if ItemIndex<0 then ItemIndex:=0;
      UpdateFtpServer;
      end;
    end
  else if PageControl.ActivePage=tsFiles then if length(edSetupFile.Text)>0 then begin
    edVersionSetup.Text:=InsertNameSuffix(ExtractFileName(edSetupFile.Text),
      '-'+GetVersionSuffix(Version,rgLevel.ItemIndex+1));
    end;
  end;

procedure TMainForm.pmiClearClick(Sender: TObject);
begin
  if ConfirmDialog(BottomLeftPos(cbTaskName),_('Remove all items from list?')) then
    cbTaskName.Clear;
  end;

procedure TMainForm.pmiEditClick(Sender: TObject);
var
  s : string;
  n : integer;
begin
  with cbTaskName do begin
    s:=Text;
    EditHistList(BottomLeftPos(cbTaskName),'',Hint,Items);
    n:=Items.IndexOf(s);
    if n>=0 then ItemIndex:=n
    else if Items.Count>0 then ItemIndex:=0
    end;
  end;

procedure TMainForm.btnAddFilesClick(Sender: TObject);
var
  i : integer;
begin
  with OpenDialog do begin
    if length(LastFile)>0 then InitialDir:=ExtractFilePath(LastFile)
    else InitialDir:=TaskPath;
    Filename:='';
    Filter:=_('All files')+'|*.*';
    Title:=_('Select files');
    Options:=Options+[ofAllowMultiSelect,ofFileMustExist];
    if Execute then begin
      with Files do begin
        for i:=0 to Count-1 do lbFiles.Items.Add(Strings[i]);
        if Count>0 then LastFile:=Strings[0];
        end;
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.btnReplFileClick(Sender: TObject);
var
  n : integer;
  s : string;
begin
  with lbFiles do begin
    n:=ItemIndex; s:=Items[n];
    end;
  with OpenDialog do begin
    InitialDir:=ExtractFilePath(s);
    Filename:=ExtractFilename(s);
    Filter:=_('Selected type')+'|*'+ExtractFileExt(s)+FSep+_('All files')+'|*.*';
    Title:=Format(_('Replace "%s" by'),[Filename]);
    Options:=Options-[ofAllowMultiSelect]+[ofFileMustExist];
    if Execute then begin
      lbFiles.Items[n]:=Filename;
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.btnRemFilesClick(Sender: TObject);
var
  i : integer;
begin
  if ConfirmDialog(BottomLeftPos(btnRemFiles),_('Remove selected file?')) then begin
    with lbFiles do for i:=Items.Count-1 downto 0 do if Selected[i] then Items.Delete(i);
    Modified:=true;
    end;
  end;

procedure TMainForm.btnAddDirsClick(Sender: TObject);
var
  s : string;
begin
  if length(LastTarget)=0 then s:=TaskPath
  else s:=LastTarget;
  if ShellDirDialog.Execute(_('Select target directory'),true,true,true,UserPath,s) then begin
    lbDirs.Items.Add(s);
    LastTarget:=s;
    Modified:=true;
    end;
  end;

procedure TMainForm.lbDirsDblClick(Sender: TObject);
var
  s : string;
begin
  with lbDirs do if ItemIndex>=0 then begin
    s:=Items[ItemIndex];
    if ShellDirDialog.Execute(_('Replace selected target directory'),true,true,true,UserPath,s) then begin
      Items[ItemIndex]:=s;
      LastTarget:=s;
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.btnRemDirsClick(Sender: TObject);
var
  i : integer;
begin
  if ConfirmDialog(BottomLeftPos(btnRemDirs),_('Remove selected target directory from list?')) then begin
    with lbDirs do for i:=Items.Count-1 downto 0 do if Selected[i] then Items.Delete(i);
    Modified:=true;
    end;
  end;

{ ------------------------------------------------------------------- }
procedure TMainForm.UpdateFtpServer;
begin
  with lbFtpServer do if (ItemIndex>=0) then with (Items.Objects[ItemIndex] as TFtpSettings).FtpPar do begin
    edServer.Text:=Host;
    edPort.Text:=IntToStr(Port);
    edUser.Text:=Username;
    edDir.Text:=Directory;
    laPass.Visible:=Passive;
    laStatus.Caption:=GetFtpSecMode(SecureMode);
    end;
  end;

procedure TMainForm.btnAddFtpServerClick(Sender: TObject);
var
  fp : TFtpSettings;
begin
  fp:=TFtpSettings.Create(NewID);
  with fp do if FtpDialog.Execute (FtpPar,false,false) then begin
    with lbFtpServer do ItemIndex:=Items.AddObject(GetFtpHint(FtpPar),fp);
    UpdateFtpServer;
    Modified:=true;
    end
  else Free;
  end;

procedure TMainForm.lbFtpServerClick(Sender: TObject);
begin
  UpdateFtpServer;
  end;

procedure TMainForm.lbFtpServerDblClick(Sender: TObject);
begin
  with lbFtpServer do if (ItemIndex>=0) then with (Items.Objects[ItemIndex] as TFtpSettings) do begin
    if FtpDialog.Execute (FtpPar,false,Ftp.CanUseTls) then begin
      Items[ItemIndex]:=GetFtpHint(FtpPar);
      UpdateFtpServer;
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.btnRemFtpServerClick(Sender: TObject);
var
  n : integer;
begin
  if ConfirmDialog(BottomLeftPos(btnRemFtpServer),_('Remove selected FTP server from list?')) then begin
    with lbFtpServer,Items do if (Count>0) and (ItemIndex>=0) then begin
      n:=ItemIndex;
      Objects[ItemIndex].Free;
      Delete(Itemindex);
      if n<Count then ItemIndex:=n else ItemIndex:=Count-1;
      UpdateFtpServer;
      Modified:=true;
      end;
    end;
  end;


{ ------------------------------------------------------------------- }
function TMainForm.NewID : integer;
var
  i : integer;
begin
  Result:=-1;
  with lbFtp do for i:=0 to Items.Count-1 do with (Items.Objects[i] as TVarField) do
    if FID>Result then Result:=FID;
  inc(Result);
  end;

{ ------------------------------------------------------------------- }
procedure TMainForm.btnAddFtpTargetClick(Sender: TObject);
var
  n : integer;
  s : string;
begin
  n:=0; s:='';
  if SelectFtpDialog.Execute(lbFtpServer.Items,n,s) then begin
    with lbFtp do ItemIndex:=Items.AddObject(MakeFtpHint(n,s),TVarField.Create(n,s));
    Modified:=true;
    end;
  end;

procedure TMainForm.btnRemFtpTargetClick(Sender: TObject);
var
  n : integer;
begin
  if ConfirmDialog(BottomLeftPos(btnRemFtpTarget),_('Remove selected FTP target from list?')) then begin
    with lbFtp,Items do if (Count>0) and (ItemIndex>=0) then begin
      n:=ItemIndex;
      Objects[ItemIndex].Free;
      Delete(Itemindex);
      if n<Count then ItemIndex:=n else ItemIndex:=Count-1;
      UpdateFtpServer;
      Modified:=true;
      end;
    end;
  end;

procedure TMainForm.lbFtpDblClick(Sender: TObject);
begin
  with lbFtp do if (ItemIndex>=0) then with (Items.Objects[ItemIndex] as TVarField) do begin
    if SelectFtpDialog.Execute(lbFtpServer.Items,FID,FString) then begin
      Items[ItemIndex]:=MakeFtpHint(FID,FString);
      Modified:=true;
      end;
    end;
  end;

{ ------------------------------------------------------------------- }
procedure TMainForm.btnAddHtmlClick(Sender: TObject);
var
  sh,s : string;
  n    : integer;
begin
  with OpenDialog do begin
    if length(LastTempl)>0 then InitialDir:=ExtractFilePath(LastTempl)
    else InitialDir:=UserPath;
    Filename:='';
    Filter:=_('HTML templates')+'|*.htt'+FSep+_('All files')+'|*.*';
    Title:=_('Select HTML template');
    Options:=Options-[ofAllowMultiSelect]+[ofFileMustExist];
    if Execute then begin
      sh:=Filename;
      LastTempl:=Filename;
      n:=0; s:='';
      if SelectFtpDialog.Execute(lbFtpServer.Items,n,s) then begin
        with lvHtml,Items.Add do begin
          Caption:=sh;
          Data:=TVarField.Create(n,s);
          SubItems.Add(MakeFtpHint(n,s));
          ItemIndex:=Index;
          end;
        Modified:=true;
        end;
      end;
    end;
  end;

procedure TMainForm.lvHtmlDblClick(Sender: TObject);
begin
  with lvHtml do if ItemIndex>=0 then with Items[ItemIndex],TVarField(Data) do begin
    with OpenDialog do begin
      if length(Caption)>0 then InitialDir:=ExtractFilePath(Caption)
      else ExtractFilePath(LastTempl);
      Filename:=ExtractFilename(Caption);
      Filter:=_('HTML templates')+'|*.htt'+FSep+_('All files')+'|*.*';
      Title:=_('Select HTML template');
      Options:=Options-[ofAllowMultiSelect]+[ofFileMustExist];
      if Execute then begin
        Caption:=Filename;
        LastTempl:=Filename;
        if SelectFtpDialog.Execute(lbFtpServer.Items,FID,FString) then begin
          SubItems[0]:=MakeFtpHint(FID,FString);
          Modified:=true;
          end;
        end;
      end;
    end;
  end;

procedure TMainForm.btnRemHtmlClick(Sender: TObject);
var
  i,n : integer;
begin
  if ConfirmDialog(BottomLeftPos(btnRemHtml),_('Remove selected template from list?')) then
      with lvHtml do begin
    n:=ItemIndex;
    for i:=0 to Items.Count-1 do with Items[i] do if Selected then
      TObject(Data).Free;
    lvhtml.DeleteSelected;
    if n<Items.Count then ItemIndex:=n else ItemIndex:=Items.Count-1;
    Modified:=true;
    end;
  end;

{ ------------------------------------------------------------------- }
(* FTP-Kopierfortschritt aktualisieren *)
procedure TMainForm.ShowProgress (AAction : TFileAction; ACount : int64; ASpeedLimit : boolean);
begin
  if ACount<0 then begin
    FSize:=abs(ACount);
    TransferStatus.UpdateProgress(0,FSize);
    end
  else TransferStatus.UpdateProgress(ACount,FSize);
  end;

(* Nachricht von Fortschrittsanzeige bei Klick auf Abbrechen-Button *)
procedure TMainForm.CancelCopy(var Msg: TMessage);
begin
  if assigned(CopyThread) then with CopyThread do if not Done then CancelThread;
  if Connecting then try FTP.Disconnect; except end;
  end;

procedure TMainForm.UpdateActionInfo(AAction : string);
begin
  TransferStatus.Action:=AAction;
  Application.ProcessMessages;
  WriteLog(2,AAction);
  end;

procedure TMainForm.UpdateStatus(AStatus : string);
begin
  TransferStatus.Status:=AStatus;
  Application.ProcessMessages;
  WriteLog(4,AStatus);
  end;

{ ------------------------------------------------------------------- }
function TMainForm.StartFtpCopy (AFtpPar : TFtpParams; const Dir,Mask,SName,VName,LName : string;
                                 Version,AddFiles : boolean; var fc : integer) : boolean;
var
  ok  : boolean;
  s,t,ss : string;
  j   : integer;
  dt  : TDateTime;

  function FtpCopyFile (sn,sd :  string) : integer;
  begin
    dt:=Now;   // begin of copy
    if CopyFiles then begin
      if length(sd)=0 then sd:=sn;
      CopyThread:=TWriteFtpThread.Create(Ftp,sn,ExtractFilePath(sd),ExtractFilename(sd),
               false,false,defFtpBufferSize,tpNormal);
      with CopyThread do begin
        OnProgress:=ShowProgress;
        repeat
          Sleep(10);
          Application.ProcessMessages;
          until Done;
        Result:=ErrorCode;
        Free;
        end;
      if (Result=errOK) and not Ftp.SetTimeStampFromFile(sn,ExtractFilename(sd))
        then Result:=errFileTS;
      end
//    Ftp.SetTimeStamp(Ftp,SName,ExtractFilename(SName));
    else Result:=errOK;
    end;

  procedure CheckForError (n : integer);    // end of copy
  begin
    if n<>errOK then UpdateStatus('  ==> '+GetCopyErrMsg(n))
    else inc(fc);
    WriteLog(4,_('  Elapsed time:')+Space+FormatDateTime('n:ss,zzz',Now-dt));
    end;

begin
  with AFtpPar do begin
    Ftp.SecureTransfer:=SecureMode;
    Ftp.Host:=Host;
    Ftp.Port:=Port;
    Ftp.Username:=Username;
    Ftp.Passive:=Passive;
    Ftp.ForceUtf8:=ForceUtf8;
    Ftp.UseHOST:=UseHost;
    Ftp.WriteLogFile:=WriteLog;
    Ftp.CaseMode:=CaseMode;
    Ftp.TransferTimeout:=defTimeout;
    if length(Password)=0 then begin
      ok:=PasswordDialog.Execute (_('FTP connection'),
          Format(_('The connection to server:'+sLineBreak+'  %s:%u'+
            sLineBreak+'for user: "%s" requires a password!'),
            [Host,Port,Username]),Password)=mrOK;
      end
    else ok:=true;
    if length(Dir)>0 then Directory:=IncludeTrailingSlash(Directory)+Dir;
    if ok and (Proxy.Mode<>fpcmNone) then begin // proxy settings
      Ftp.ProxySettings.ProxyType:=Proxy.Mode;
      Ftp.ProxySettings.Host:=Proxy.Server;
      Ftp.ProxySettings.Port:=Proxy.Port;
      Ftp.ProxySettings.Username:=Proxy.Username;
      with Proxy do if length(Password)=0 then
        ok:=PasswordDialog.Execute (rsFtpConn,
            Format(rsProxyHint,[Server,Port,Username]),Password)=mrOK;
      end;
    if ok then begin
      TransferStatus.Target:=GetFtpHint(AFtpPar);
      UpdateActionInfo(_('Connecting to FTP server'));
      with Ftp.ProxySettings do if ProxyType<>fpcmNone then Password:=Proxy.Password;
      Ftp.Password:=Password;
      Screen.Cursor:=crHourGlass;
      try
        Connecting:=true;
        Ftp.Connect;
      except
        ok:=false;
        end;
      Screen.Cursor:=crDefault;
      Application.ProcessMessages;
      Connecting:=false;
      if ok then ok:=not TransferStatus.Stopped and Ftp.ForceDir(Directory);
      if ok then Ftp.GetFileList;
      end;
    end;
  Application.ProcessMessages;
  if ok then begin
    UpdateStatus(_('Connected to:')+Space+AFtpPar.Host);
    TransferStatus.Directory:='/'+AFtpPar.Directory;
    UpdateStatus(_('Remote directory:')+Space+AFtpPar.Directory);
    if (length(Mask)>0) then begin
      UpdateStatus(TryFormat(_('Delete previous file versions (%s)'),[Mask]));
      if CopyFiles then with Ftp do begin
        DeleteMatchingFiles(Mask);
        DeleteMatchingFiles(NewExt(Mask,Md5Ext));
        DeleteMatchingFiles(NewExt(Mask,Sha256Ext));
        DeleteMatchingFiles(AddExt(Mask,SigExt,2));
        end;
      end;
    UpdateStatus(_('Copy files'));
    FSize:=LongFileSize(SName);
    TransferStatus.Position:=0;
    if AnsiSameText(GetExt(SName),HtmlExt) then begin
      s:=_('  HTML page: %s');
      t:=''; ss:='';
      end
    else begin
      s:=_('  Setup file: %s');
      t:=NewExt(SName,Md5Ext); ss:=AddExt(SName,SigExt,2);
      end;
    UpdateStatus(TryFormat(s,[ExtractFileName(SName)]));
    CheckForError(FtpCopyFile(SName,''));
    if cbHash.Checked and (length(t)>0) and (FileExists(t)) then begin
      UpdateStatus(_('  MD5 checksum')+ColSpace+ExtractFilename(t));
      CheckForError(FtpCopyFile(t,''));
      t:=NewExt(SName,Sha256Ext);
      UpdateStatus(_('  SHA256 checksum')+ColSpace+ExtractFilename(t));
      CheckForError(FtpCopyFile(t,''));
      end;
    if cbGpgSig.Checked and (length(ss)>0) and (FileExists(ss)) then begin
      UpdateStatus(_('  OpenPGP signature')+ColSpace+ExtractFilename(ss));
      CheckForError(FtpCopyFile(ss,''));
      end;
    if Version then begin
      UpdateStatus(_('  Version info')+ColSpace+ExtractFilename(VName));
      CheckForError(FtpCopyFile(VName,''));
      UpdateStatus(_('  Download location')+ColSpace+ExtractFilename(LName));
      CheckForError(FtpCopyFile(LName,''));
      end;
    // zustzliche Dateien
    if AddFiles then with lbFiles do for j:=0 to Items.Count-1 do begin
      s:=Items[j];
      if Checked[j] then t:=edProjPrefix.Text+'-'+ExtractFileName(s)
      else t:=ExtractFileName(s);
      UpdateStatus(Format(_('  Additional file: %s'),[t]));
      Application.ProcessMessages;
      CheckForError(FtpCopyFile(s,t));
      end;
    try
      Ftp.Disconnect;
    except
      if WaitPrompt then ErrorDialog(GetDlgPos,_('Error closing FTP connection'));
      WriteLog(2,_('Error closing FTP connection'));
      end;
    Result:=true;
    end
  else begin
    if WaitPrompt then ErrorDialog(GetDlgPos,Format(rsConnectErr,[AFtpPar.Host]));
    WriteLog(2,Format(rsConnectErr,[AFtpPar.Host]));
    Result:=false;
    end;
  end;

procedure TMainForm.btnCopyClick(Sender: TObject);
begin
  CopyFiles:=true;
  ProcessFiles;
  end;

procedure TMainForm.btnTestClick(Sender: TObject);
begin
  CopyFiles:=false;
  ProcessFiles;
  end;

function TMainForm.ProcessHtmlTemplate(const TemplName,Version,SuName,Md5,Sha : string) : string;
var
  sp,sv,sw,sc,
  s,t,st,sh  : string;
//  fi,ft,fh   : TextFile;
  dmode,j,
  i,k,n,m    : integer;
  dp         : char;
  sli,slh,slt  : TStringList;
  phl        : TStringList;
const
  varcnt = 6;
  vars : array [1..varcnt] of string = ('vers','date','setup','size','md5','sha256');
begin
  sh:=NewExt(TemplName,HtmlExt);
  UpdateActionInfo(Format(_('Processing HTML page: %s'),[ExtractFileName(sh)]));
  if FileExists(TemplName) then begin
    sli:=TStringList.Create;
    slh:=TStringList.Create;
    slt:=TStringList.Create;
    phl:=TStringList.Create;
//    AssignFile(fi,TemplName); reset(fi);
    sli.LoadFromFile(TemplName);
    st:=TemplName+'.tmp';
//    AssignFile(ft,st); rewrite(ft);
//    AssignFile(fh,sh); rewrite(fh);
    m:=0; dp:=Punkt;    // def.
    dmode:=0;  // Date format: 0 = ISO (def.), 1 = German, 2 = English
    for j:=0 to sli.Count-1 do begin
      s:=sli[j];
//    while not Eof(fi) do begin
//      readln(fi,s);
      t:=s;
      if length(s)>0 then begin
        if AnsiStartsText('<!-- #var',s) then begin
          m:=1;
          slt.Add(t);
//          writeln(ft,t);
          end
        else if (m=1) and AnsiStartsText('-->',s) then begin
          m:=2;
          slt.Add(t);
//          writeln(ft,t);
          end
        else begin
          if m=1 then begin  // Variablen
            sv:=Trim(ReadNxtStr(s,'='));
            n:=pos('//',s); sc:='';
            if n=0 then sw:=Trim(s)    // Wert
            else begin
              sw:=Trim(copy(s,1,n-1)); // Wert
              sc:=copy(s,n,length(s)); // Kommentar
              end;
            // analysiere Variable
            if AnsiSameText(sv,'date-format') then begin
              s:=sw;
              dmode:=ReadNxtInt(s,' ',0)
              end
            else if AnsiSameText(sv,'decimal-sep') then begin
              if length(sw)>0 then dp:=sw[1];
              end
            else begin
              s:=sv;
              sp:=ReadNxtStr(s,'-');
              if AnsiSameText(sp,edProjPrefix.Text) then begin // Prfix prfen
                for i:=1 to varcnt do if AnsiSameText(s,vars[i]) then break;
                case i of
                1 : sw:=Version;
                2 : case dmode of
                    1 : sw:=FormatDateTime('d.m.yyyy',Now);
                    2 : sw:=FormatDateTime('mm/dd/yyyy',Now);
                    else sw:=FormatDateTime('yyyy-mm-dd',Now);
                      end;
                3 : sw:=ExtractFilename(SuName);
                4 : sw:=SizeToStr(LongFileSize(SuName),true,false,dp);  // dezimal
                5 : sw:=Md5;
                6 : sw:=Sha;
                  end;
                end;
              phl.AddObject(sv,TVarField.Create(0,sw));
              end;
//            writeln(ft,sv+' = '+sw+'   '+sc);
            slt.Add(sv+' = '+sw+'   '+sc);
            end
          else if m=2 then begin  // HTML
            slt.Add(t);
//            writeln(ft,t);
            t:='';
            repeat
              n:=Pos(phBegin,s);
              if n>0 then begin
                k:=PosEx(phEnd,s,n);
                sv:=copy(s,n+2,k-n-2);
                i:=phl.IndexOf(sv);
                if i>=0 then t:=t+copy(s,1,n-1)+(phl.Objects[i] as TVarField).FString
                else t:=t+copy(s,1,k);
                delete(s,1,k);
                end
              else t:=t+s;
              until (n=0);
            slh.Add(t);
//            writeln(fh,t)
            end
          else slt.Add(t); // writeln(ft,t);
          end;
        end
      else begin
        slt.Add(t);
//        writeln(ft);
        if m=2 then slh.Add(t);// writeln(fh);
        end;
      end;
//    CloseFile(fi); CloseFile(ft); CloseFile(fh);
    slt.SaveToFile(st,sli.Encoding);
    slh.SaveToFile(sh,sli.Encoding);
    sli.Free; slt.Free; slh.Free;
    FreeListObjects(phl);
    phl.Free;
    // alte durch neue Vorlage ersetzen
    if CopyFiles then begin
      DeleteFile(TemplName);
      RenameFile(st,TemplName);
      end;
    Result:=sh;
    end
  else begin
    UpdateStatus('==> '+_('File not found:')+Space+TemplName);
    Result:='';
    end;
  end;

function TMainForm.GetVersionSuffix(t : string; n : integer) : string;
begin
  if n=0 then Result:=''
  else begin
    Result:=IntToStr(ReadNxtInt(t,'.',0));
    if n>1 then Result:=Result+'.'+IntToStr(ReadNxtInt(t,'.',0));
    if n>2 then Result:=Result+'.'+ZStrInt(ReadNxtInt(t,'.',0),2);
    if n>3 then Result:=Result+ZStrInt(ReadNxtInt(t,'.',0),2);
    end;
  end;

function TMainForm.GetVersionName(t : string; n : integer) : string;
begin
  if n=0 then Result:=''
  else begin
    Result:=IntToStr(ReadNxtInt(t,'.',0));
    if n>1 then Result:=Result+'.'+IntToStr(ReadNxtInt(t,'.',0));
    if n>2 then Result:=Result+'.'+IntToStr(ReadNxtInt(t,'.',0));
    if n>3 then Result:=Result+'.'+IntToStr(ReadNxtInt(t,'.',0));
    end;
  end;

function TMainForm.ProcessFiles : boolean;
var
  s,t,sn,sf,sd,
  sl,vl,sv,sm,
  smd5,md5,
  ssha,sha,ssig : string;
  i,j,fCnt,n,ec : integer;
  fv            : TextFile;
  el            : TStringList;
  bvi,ok,psig   : boolean;
  pt            : TPathType;

  (* Datei kopieren mit Attribut und Datum - verwendet CopyFileTS aus FileUtils*)
  function FileCopy (const srcfilename,destfilename : String) : integer;
  begin
    if CopyFiles then begin
      Result:=NO_ERROR;
      if FileExists (srcfilename) and (length(destfilename)>0) then begin
        if FileExists(destfilename) then DeleteFile(destfilename);
        try
          CopyFileTS(srcfilename,destfilename);
        except
          on E:ECopyError do begin
            Result:=E.ErrorCode;    // last system error
            end;
          end;
        end;
      end
    else Result:=NO_ERROR;
    end;

  procedure DeleteMatchingFiles(const APath,AMask : string);
  var
    DirInfo    : TSearchRec;
    Findresult : integer;
  begin
    FindResult:=FindFirst (APath+AMask,faAnyFile,DirInfo);
    while (FindResult=0) do begin
      DeleteFile(APath+DirInfo.Name);
      FindResult:=FindNext(DirInfo)
      end;
    FindClose(DirInfo);
    end;

  function StringToTextFile (const OutName,AString : string) : boolean;
  var
    oText : TextFile;
  begin
    Result:=false;
    try
      AssignFile(oText,OutName); rewrite(oText);
      writeln (oText,AString);
      Result:=true;
    finally
      CloseFile(oText);
      end;
    end;

  function CheckForError (n : integer) : boolean;    // true id successful
  begin
    Result:=n=NO_ERROR;
    if not Result then UpdateStatus(_('  ==> Copy error')+ColSpace+SystemErrorMessage(n));
    end;

begin
  Result:=false;
  OpenLog(Append);
  if CopyFiles then s:=_('Starting copy process')
  else s:=_('Simulating data processing');
  InitLog(s);
  if FileExists(edExeFile.Text) and (length(Version)>0) then begin
    if not FileExists(edSetupFile.Text) then begin
      if WaitPrompt then ErrorDialog(GetDlgPos,Format(_('Setup file "%s" not found!'),[edSetupFile.Text]));
      WriteLog(2,Format(_('Setup file "%s" not found!'),[edSetupFile.Text]));
      CloseLog;
      Exit;
      end;
    sn:=InsertNameSuffix(edSetupFile.Text,'-'+GetVersionSuffix(Version,rgLevel.ItemIndex+1));
    sf:=GetVersionSuffix(Version,rgKeep.ItemIndex);
    if CopyFiles then s:=_('Task:')
    else s:=_('Simulate:');
    TransferStatus.ShowWindow(Handle,s+Space+DelExt(ExtractFilename(Taskname)),
                  '','');
    if length(edVersionInfoFile.Text)>0 then begin
      WriteLog(0,'');
      UpdateActionInfo(_('Update version info'));
      // Versionsdatei schreiben
      sv:=NewExt(edVersionInfoFile.Text,VerExt);
      bvi:=FileExists(sv);
      if bvi then begin
        Application.ProcessMessages;
        StringToTextFile(sv,Version);                 // Version
        sl:=NewExt(edVersionInfoFile.Text,LocExt);
        if FileExists(sl) then begin
          AssignFile(fv,sl); Reset(fv);
          ReadLn(fv,vl); // URL
          CloseFile(fv);
          AssignFile(fv,sl); Rewrite(fv);             // Download
          WriteLn(fv,vl);
          WriteLn(fv,ExtractFileName(sn));
          CloseFile(fv);
          end;
        UpdateStatus(Format(_('Version files updated to %s'),[Version]));
        UpdateStatus(Format(_('  Files: %s (.ver/.loc)'),[ExtractFilename(DelExt(sv))]));
        end
      else begin
        WriteLog(2,_('Template for version info not found!'));
        if not ConfirmDialog(BottomLeftPos(btnCopy),_('Template for version info not found!'+sLineBreak+
                       'Continue without copying this file?')) then begin
          WriteLog(2,_('Terminated by user'));
          TransferStatus.Close;
          CloseLog;
          Exit;
          end;
        end;
      end
    else bvi:=false;
    if FileExists(edSetupFile.Text) and CopyFiles then begin
      s:=ExtractFilePath(sn);
      if not DirectoryExists(s) then ForceDirectories(s);
      DeleteFile(sn);
      FileCopy(edSetupFile.Text,sn);       // Kopie des Setups in Datei mit Version
      end;
    WriteLog(0,'');
    WriteLog(2,_('Generate checksums'));
    if not CopyFiles then sn:=edSetupFile.Text;
    md5:=HashFromFile(htMD5,sn); sha:=HashFromFile(htSHA256,sn);
    if (length(md5)>0) and (length(sha)>0) then begin
      if cbHash.Checked then begin // write to file
        UpdateStatus(_('MD5 checksum generated'));
        smd5:=NewExt(sn,Md5Ext); ssha:=NewExt(sn,Sha256Ext);
        StringToTextFile(smd5,md5);
        UpdateStatus(Format(_('MD5 file created: %s'),[ExtractFileName(smd5)]));
        UpdateStatus(_('SHA256 checksum generated'));
        StringToTextFile(ssha,sha);
        UpdateStatus(Format(_('SHA256 file created: %s'),[ExtractFileName(ssha)]));
        end;
      end
    else UpdateStatus(_('Error on generating MD5/SHA256 checksums!'));
    WriteLog(0,'');
    // OpenPgp-Signatur erzeugen
    psig:=false;
    if GpgInstalled and cbGpgSig.Checked then begin
      ssig:=AddExt(ExtractFileName(sn),SigExt,2);
      if FileExists(ssig) then DeleteFile(ssig);
      el:=TStringList.Create;
      WriteLog(2,_('Create OpenPGP signature'));
      n:=ExecuteConsoleProcess (Format(GpgCommand,[PgpID,ssig,sn]),ExtractFilePath(sn),el);
//      n:=ExecuteProcess (GpgCommand+PgpID+' '+sn,ExtractFilePath(s),true,0);
      psig:=n=0;
      if psig then
        UpdateStatus(Format(_('OpenPGP signature created: %s'),[ssig]))
      else begin
        if n and UserError<>0 then begin
          UpdateStatus(_('Error on creating OpenPGP signature:')+Space+
                       Format(_('GnuGPG return value = %u'),[n and $FF]));
          with el do for i:=0 to Count-1 do UpdateStatus(' * '+el[i]);
          end
        else UpdateStatus(_('Error on creating OpenPGP signature:')+Space+SystemErrorMessage(n));
        end;
      el.Free;
      WriteLog(0,'');
      end;
    // Dateien in Verzeichnisse kopieren
    UpdateActionInfo(_('Copy files'));
    fCnt:=0;
    sm:=ExtractFileName(InsertNameSuffix(edSetupFile.Text,'-'+sf+'*'));
    with lbDirs do for i:=0 to Items.Count-1 do begin
      s:=IncludeTrailingPathDelimiter(Items[i]);
      pt:=CheckPath(s);
      if pt=ptNotAvailable then begin // ev. nicht verbundener Netzwerkpfad?
        ok:=ReconnectPath(s);
        end
      else ok:=true;
      UpdateStatus(_('Destination directory:')+Space+s);
      if ok then begin
        if not DirectoryExists(s) then ForceDirectories(s);
        UpdateStatus(TryFormat(_('Delete previous file versions (%s)'),[sm]));
        if CopyFiles then begin
          DeleteMatchingFiles(s,sm);
          DeleteMatchingFiles(s,NewExt(sm,Md5Ext));
          DeleteMatchingFiles(s,NewExt(sm,Sha256Ext));
          DeleteMatchingFiles(s,AddExt(sm,SigExt,2));
          end;
        UpdateStatus(_('Copy files'));
        UpdateStatus(Format(_('  Setup file: %s'),[ExtractFileName(sn)]));
        if CheckForError(FileCopy(sn,s+ExtractFileName(sn))) then begin
          inc(fCnt);
          if cbHash.Checked then begin
            if (FileExists(smd5)) then begin
              UpdateStatus(_('  MD5 checksum')+ColSpace+ExtractFilename(smd5));
              if CheckForError(FileCopy(smd5,s+ExtractFileName(smd5))) then inc(fCnt);
              end;
            if (FileExists(ssha)) then begin
              UpdateStatus(_('  SHA256 checksum')+ColSpace+ExtractFilename(ssha));
              if CheckForError(FileCopy(ssha,s+ExtractFileName(ssha))) then inc(fCnt);
              end;
            end;
          if psig then begin // Kopiere OpenPGP-Signatur
            UpdateStatus(_('  OpenPGP signature')+ColSpace+ExtractFilename(ssig));
            if CheckForError(FileCopy(ssig,s+ExtractFileName(ssig))) then inc(fCnt);
            end;
          if bvi and Checked[i] then begin
            UpdateStatus(_('  Version info')+ColSpace+ExtractFilename(sv));
            if CheckForError(FileCopy(sv,s+ExtractFileName(sv))) then inc(fCnt);
            UpdateStatus(_('  Download location')+ColSpace+ExtractFilename(sl));
            if CheckForError(FileCopy(sl,s+ExtractFileName(sl))) then inc(fCnt);
            end;
          end
        else begin
          if WaitPrompt then ErrorDialog(GetDlgPos,Format(_('Error copying: %s to %s'),[sn,s]));
          end;
        // zustzliche Dateien
        with lbFiles do for j:=0 to Items.Count-1 do begin
          t:=Items[j];
          if Checked[j] then sd:=edProjPrefix.Text+'-'+ExtractFileName(t)
          else sd:=ExtractFileName(t);
          UpdateStatus(Format(_('  Additional file: %s'),[sd]));
          if CheckForError(FileCopy(t,s+sd)) then inc(fCnt);
          end;
        end
      else begin
        UpdateStatus(_('==> Error')+ColSpace+Format(_('Path not available: %s'),[s]));
        if WaitPrompt then ErrorDialog(GetDlgPos,Format(_('Path not available: %s'),[s]));
        end;
      end;
    WriteLog(0,'');
    // Dateien per FTP kopieren
    with lbFtp,Items do for i:=0 to Count-1 do begin
      if assigned(Objects[i]) then with (Objects[i] as TVarField) do begin
        n:=SearchForID (lbFtpServer.Items,FID); t:=FString;
        end
      else n:=-1;
      with lbFtpServer.Items do if (n>=0) and (n<Count) then with (Objects[n] as TFtpSettings) do begin
        if not ConnectionError and not StartFtpCopy(FtpPar,t,sm,sn,sv,sl,bvi and Checked[i],true,fCnt) then begin
          ConnectionError:=true; // kein Verbindung zu diesem Server
          UpdateStatus(_('  Skip copy to ')+ColSpace+FtpPar.Host);
          end;
        end
      else begin
        UpdateStatus(_('  ==> Error'));
        if WaitPrompt then ErrorDialog(GetDlgPos,_('Illegal reference to FTP server list!'));
        end;
      end;
    with TransferStatus do begin
      Target:=''; Directory:='';
      end;
    // HTML-Schablonen bearbeiten und Web-Seiten kopieren
    with lvHtml do for i:=0 to Items.Count-1 do with Items[i] do begin
      WriteLog(0,'');
      s:=ProcessHtmlTemplate(Caption,GetVersionName(Version,rgLevel.ItemIndex+1),sn,md5,sha);
      // HTML-Datei per FTP kopieren
      if length(s)>0 then begin
        if assigned(Data) then with TVarField(Data) do begin
          n:=SearchForID (lbFtpServer.Items,FID); t:=FString;
          end
        else n:=-1;
        with lbFtpServer.Items do if (n>=0) and (n<Count) then with (Objects[n] as TFtpSettings) do begin
          if not ConnectionError and not StartFtpCopy(FtpPar,t,'',s,'','',false,false,fCnt) then
            ConnectionError:=true; // kein Verbindung zu diesem Server
          end
        else begin
          UpdateStatus(_('  ==> Error'));
          if WaitPrompt then ErrorDialog(GetDlgPos,_('Illegal reference to FTP server list!'));
          end;
//        DeleteFile(s);
        end;
      end;
    with TransferStatus do begin
      Status:=_('Done');
      if WaitPrompt then CloseWindow(0,fCnt) else Close;
      end;
    end
  else begin
    if WaitPrompt then ErrorDialog(GetDlgPos,Format(_('Exe file with version info not found:')+Space+sLineBreak+'"%s"',
                            [edExeFile.Text]));
    WriteLog(2,Format(_('Exe file with version info not found:')+Space+'"%s"',[edExeFile.Text]));
    end;
  CloseLog;
  Result:=true;
  end;

end.
