(* Unit:     AESLib.pas
   Interface to the AES functions written by Brian Gladman

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   Acknowledgements:
     AES functions from http://fp.gladman.plus.com/index.htm

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   Vers. 1.1 - Dec. 2005
   Vers. 1.2 - Jan. 2006
   Vers. 1.3 - Mar. 2006
   Vers. 1.4 - Apr. 2006
   Vers. 1.5 - Jun. 2006 - Callback parameter changed
   Vers. 1.6 - Jul. 2006 - Property "KeyMode" added
         1.6.1 - Aug. 2006 - errors in "DecryptStream" fixed
         1.6.2 - Apr. 2007 - licensing changed to MPL
   Vers. 1.7   - May  2007 - optional fix keymode
   Vers. 1.8   - Oct. 2008 - Password length truncated to MaxPwdLength(= 128)

   Vers. 2    - Feb. 2009 : Unicode Version for Delphi 2009
                            ===============================
        Notes: Passwords always are processed as AnsiString

   Vers. 3    - Jan. 2013 : 32- and 64-bit
                            ==============
         3.1  - Jan. 2016 : clear used buffers in destructor
                            some small cosmetic changes
         3.2  - Oct. 2016 : revised setting of keymode
         3.3  - Feb. 2023 : callback function to abort an encryption or decryption
         3.4  . Feb. 2024 : TEvent/WaitFor added

   Field lengths (in bytes) versus File Encryption Mode (0 < KeyMode < 4)
      KeyMode   Password    Key            Salt  MAC  Overhead
      1         <32         16 (128 bit)   8     10   18
      2         <48         24 (192 bit)   12    10   22
      3         >=48        32 (256 bit)   16    10   26

  *)

unit AesLib;

interface

uses Winapi.Windows, System.Classes, System.SysUtils, System.SyncObjs, CBFunctions;

// type declarations from CBFunctions:
{
  TFileAction = (acNone,acEncrypt,acDecrypt);
  TFileProgressEvent = procedure(AAction : TFileAction; ACount: int64; ASpeedLimit : boolean) of object;
}

const
  PwdLength192 = 32;
  PwdLength256 = 48;
  defCryptBufSize = 256*1024;
  BlockSize = 16;               // see aes.h
  KsLength = 4*BlockSize;
  SaltMax = 16;
  MacMax = 32;
  MacLength = 10;
  MaxKeyLength = 32;            // see filenc.h
  MaxPwdLength = 128;           // WinZip and 7-zip have limitiation to 99 charachters
  PwdVerLength = 2;
  KeyingIterarations = 1000;
  ShaBlockSize = 64;            // see sha1.h
  ShaDigestSize = 20;
  ShaLength = 23;
  PrngPoolLen  = 256;           // see prng.h
  PrngMinMix = 20;
  PrngPoolSize = ShaDigestSize*((PrngPoolLen-1) div ShaDigestSize +1);
  AesContextSize = 4*KsLength+4*3;
  HMacContextSize = ShaBlockSize+4*ShaLength+sizeof(integer);
  PrngContextSize = 2*PrngPoolSize+sizeof(integer)+sizeof(pointer);
  FCryptContextSize = 2*BlockSize+AesContextSize+HMacContextSize+3*sizeof(integer);

  // Saltlength depends on mode (password length)
  SaltLength : array [1..3] of cardinal = (8,12,16);

type
  TPrngContext = packed array[0..PrngContextSize-1] of byte;
  TSaltBuf = packed array[0..SaltMax-1] of byte;
  TMacBuffer = packed array[0..MacMax-1] of AnsiChar;
  TFCryptContext = packed array[0..FCryptContextSize-1] of byte;
  TAesContext = packed array[0..AesContextSize-1] of byte;
  THMacContext = packed array[0..HMacContextSize-1] of byte;
  TPwdVerifier = packed array[0..PwdVerLength-1] of AnsiChar;

  TOpResult = (orOk,orError,orUserBreak);

  TCrypt = class (TObject)
  private
    procedure SetMode (AMode : integer);
  protected
    FMode       : integer;
    FFixedKey   : boolean;
    FCryptCtx   : TFCryptContext;
    FPwd        : AnsiString;
    CBufSize    : cardinal;
    CBuffer     : array of byte;
    FEvent      : TEvent;
  public
    constructor Create (const Password : AnsiString; AMode,ABufSize : integer);
    destructor Destroy; override;
    function GetHeaderSize : integer;
    function GetTrailerSize : integer;
    property KeyMode  : integer read FMode write SetMode;
    property FixedKey : boolean read FFixedKey;
    end;

  TEncryption = class (TCrypt)
  private
    FPrngCtx    : TPrngContext;
    FOnProgress : TFileProgressEvent;
  protected
    procedure DoProgress (AAction : TFileAction; ACount : int64; ASpeedLimit : boolean);
  public
    constructor Create (const Password : AnsiString; AMode,ABufSize : integer);
    destructor Destroy; override;
    procedure EncryptBlock (var Buffer; BLen : cardinal);
    function WriteHeader (sDest : TStream) : boolean;
    function WriteTrailer (sDest : TStream) : boolean;
    function EncryptStreamEx (sSource,sDest : TStream; SpeedLimit : boolean;
                              CancelEncryption : TStatusEvent = nil) : TOpResult;
    function EncryptStream (sSource,sDest : TStream; SpeedLimit : boolean ) : boolean;
    procedure DecryptBlock (var Buffer; BLen : cardinal);
    function ReadHeader (sSource : TStream) : boolean;
    function ReadTrailer (sSource : TStream): boolean;
    function DecryptStreamEx (sSource,sDest: TStream; SLength : int64; SpeedLimit : boolean;
                              CancelDecryption : TStatusEvent = nil) : TOpResult;
    function DecryptStream (sSource,sDest: TStream; SLength : int64; SpeedLimit : boolean) : boolean;
    property Event : TEvent write FEvent;
    property OnProgress : TFileProgressEvent read FOnProgress write FOnProgress;
    end;

{ ---------------------------------------------------------------------------- }
implementation

uses System.StrUtils;

{$L fileenc.obj}
{$L aescrypt.obj}
{$L aeskey.obj}
{$L aestab.obj}
{$L pwd2key.obj}
{$L prng.obj}
{$L hmac.obj}
{$L sha1.obj}

type
  TEntropyFunction = function (var Buffer; Len : cardinal) : integer;

  TLongInteger = record
    case integer of
    0: (AsInt64   : int64);
    1: (Lo,Hi     : Cardinal);
    2: (Cardinals : array [0..1] of Cardinal);
    3: (Words     : array [0..3] of Word);
    4: (Bytes     : array [0..7] of Byte);
    end;

{ ---------------------------------------------------------------------------- }
// entry points for included object files
function fcrypt_init (Mode : integer; const Pwd : PRawByteString; PwdLen : cardinal; const Salt;
              var PwdVerifier : TPwdVerifier; var CryptContext : TFCryptContext) : integer; external;
procedure fcrypt_encrypt (var Data; DataLen : cardinal; var CryptContext : TFCryptContext); external;
procedure fcrypt_decrypt (var Data; DataLen : cardinal; var CryptContext : TFCryptContext); external;
function fcrypt_end (var MacBuf; const CryptContext : TFCryptContext) : integer; external;

function aes_set_encrypt_key (const Key : PRawByteString; KeyLen : cardinal; var AesContext : TAesContext) : cardinal; external;
function aes_set_decrypt_key (const Key : PRawByteString; KeyLen : cardinal; var AesContext : TAesContext) : cardinal; external;
function aes_encrypt_block (const Ib; var Ob; const AesContext : TAesContext) : cardinal; external;
function aes_decrypt_block (const Ib; var Ob; const AesContext : TAesContext) : cardinal; external;

procedure prng_init (Fun : TEntropyFunction; var PrngContext : TPrngContext); external;
procedure prng_rand (var Data; DataLen : cardinal; var PrngContext : TPrngContext); external;
procedure prng_end (var PrngContext : TPrngContext); external;

procedure hmac_sha1_begin (var HMacContext : THMacContext); external;
procedure hmac_sha1_key (const Key : PRawByteString; KeyLen : cardinal; var HMacContext : THMacContext); external;
procedure hmac_sha1_data (const Data : PByteArray; DataLen : cardinal; var HMacContext : THMacContext); external;
procedure hmac_sha1_end (const Mac : PByteArray; MacLen : cardinal; var HMacContext : THMacContext); external;

procedure derive_key (const Pwd : PRawByteString; PwdLen : cardinal; const Salt; SaltLen,Iter : cardinal;
             var Key; KeyLen : cardinal); external;

{ ---------------------------------------------------------------------------- }
// replacement for C library functions
procedure _memset (var Dest; Value,Count : integer); cdecl;
begin
  FillChar (Dest,Count,chr(Value));
  end;

procedure _memcpy (var Dest; const Source; Count : integer); cdecl;
begin
  Move (Source,Dest,Count);
  end;

{ ---------------------------------------------------------------------------- }
// Entropy function for "prng"
function Entropy (var Buffer; Len : cardinal) : integer;
var
  Value : Int64;
  i     : integer;  
begin
  // use Windows performance counter as entropie function
  // if not available use the tick count instead (only 4 low bytes)
  if not QueryPerformanceCounter(Value) then Value:=GetTickCount;
  if Len<8 then i:=Len else i:=8;
  Move (Value,Buffer,i);
  Result:=i;
  end;

{ ---------------------------------------------------------------------------- }
function AesEncKey (KeyStr : RawByteString; KeyLen : cardinal; var AesContext : TAesContext) : cardinal;
begin
  result:=aes_set_encrypt_key(PRawByteString(KeyStr),KeyLen,AesContext);
  end;

function AesDecKey (KeyStr : RawByteString; KeyLen : cardinal; var AesContext : TAesContext) : cardinal;
begin
  result:=aes_set_decrypt_key(PRawByteString(KeyStr),KeyLen,AesContext);
  end;

function AesEncBlk (const AesContext : TAesContext; const Ib; var Ob) : cardinal;
begin
  result:=aes_encrypt_block(PByteArray(Ib),PByteArray(Ob),AesContext);
  end;

function AesDecBlk (const AesContext : TAesContext; const Ib; var Ob) : cardinal;
begin
  result:=aes_decrypt_block(PByteArray(Ib),PByteArray(Ob),AesContext);
  end;

{ ---------------------------------------------------------------------------- }
// expand short passwords
function ExpandPwd (KeyStr : RawByteString; KeyLen : Integer) : RawByteString;
begin
  if length(KeyStr)>0 then begin
    repeat
      KeyStr:=KeyStr+KeyStr;
      until length(KeyStr)>KeyLen;
    Result:=copy(KeyStr,1,KeyLen);
    end
  else Result:='';
  end;

{ ---------------------------------------------------------------------------- }
// Crypt object
// AMode = 1,..,3  ==> AES128, AES192, AES256
//       = 0       ==> dependent on password length
constructor TCrypt.Create (const Password : AnsiString; AMode,ABufSize : integer);
begin
  inherited Create;
  FPwd:=LeftStr(Password,MaxPwdLength);
  CBufSize:=ABufSize; SetLength(CBuffer,CBufSize);
//  if length(FPwd)<8 then FPwd:=ExpandPwd(FPwd,8);
  SetMode(AMode);
  FFixedKey:=true;
  FEvent:=nil;
  end;

destructor TCrypt.Destroy;
begin
  FillChar(FPwd[1],length(FPwd),0); // clear password
  FillChar(CBuffer[0],CBufSize,0);  // clear buffer
  CBuffer:=nil;                     // free buffer
  inherited Destroy;
  end;

procedure TCrypt.SetMode (AMode : integer);
begin
  if (AMode>0) and (AMode<=3) then FMode:=AMode
  else begin
    FFixedKey:=false;
    if length(FPwd)<PwdLength192 then FMode:=1
    else if length(FPwd)<PwdLength256 then FMode:=2
    else FMode:=3;
    end;
  end;

function TCrypt.GetHeaderSize : integer;
begin
  Result:=SaltLength[FMode]+PwdVerLength;
  end;

function TCrypt.GetTrailerSize : integer;
begin
  Result:=MacLength;
  end;

{ ---------------------------------------------------------------------------- }
// Encryption object
constructor TEncryption.Create (const Password : AnsiString; AMode,ABufSize : integer);
begin
  inherited Create (Password,AMode,ABufSize);
  prng_init (Entropy,FPrngCtx);
  FOnProgress:=nil;
  end;

destructor TEncryption.Destroy;
begin
  prng_end (FPrngCtx);
  inherited Destroy;
  end;

{ ------------------------------------------------------------------- }
procedure TEncryption.DoProgress (AAction : TFileAction; ACount : int64; ASpeedLimit : boolean);
begin
  if Assigned(FOnProgress) then FOnProgress(AAction,ACount,ASpeedLimit);
  end;

procedure TEncryption.EncryptBlock (var Buffer; BLen : cardinal);
begin
  fcrypt_encrypt(Buffer,BLen,FCryptCtx);
  end;

function TEncryption.WriteHeader (sDest : TStream) : boolean;
var
  SaltBuf     : TSaltBuf;
  FPwdVer     : TPwdVerifier;
begin
  prng_rand (SaltBuf,SaltLength[FMode],FPrngCtx);
  fcrypt_init (FMode,PRawByteString(FPwd),length(FPwd),SaltBuf,FPwdVer,FCryptCtx);
  try
    // write salt value
    sDest.Write (SaltBuf,SaltLength[FMode]);
    // write password verifier
    sDest.Write (FPwdVer,PwdVerLength);
    result:=true;
  except
    result:=false;
    end;
  end;

function TEncryption.WriteTrailer (sDest : TStream) : boolean;
var
  MacBuf : TMacBuffer;
begin
  FillChar(MacBuf,MacMax,0);
  fcrypt_end (MacBuf,FCryptCtx);
  try
    sDest.Write(MacBuf,MacLength);
    result:=true;
  except
    result:=false;
    end;
  end;

function TEncryption.EncryptStreamEx (sSource,sDest : TStream; SpeedLimit : boolean;
                                      CancelEncryption : TStatusEvent) : TOpResult;
var
  NRead    : cardinal;
  Total    : int64;
begin
  Result:=orError; Total:=0;
  if WriteHeader (sDest) then begin
    repeat
      if assigned(FEvent) then FEvent.WaitFor(INFINITE);
      try
        NRead:=sSource.Read(CBuffer[0],CBufSize);
        inc(Total,NRead);
        DoProgress(acEncrypt,Total,SpeedLimit);
        EncryptBlock (CBuffer[0],NRead);
        sDest.Write(CBuffer[0],NRead);
      except
        Exit;
        end;
      if assigned(CancelEncryption) and CancelEncryption then begin
        Result:=orUserBreak; Break;
        end;
      until (NRead<CBufSize);
    if (Result<>orUserBreak) and WriteTrailer (sDest) then Result:=orOk;
    end;
  end;

function TEncryption.EncryptStream (sSource,sDest : TStream; SpeedLimit : boolean) : boolean;
begin
  Result:=EncryptStreamEx (sSource,sDest,SpeedLimit,nil)=orOk;
  end;

{ ---------------------------------------------------------------------------- }
// Decryption object
procedure TEncryption.DecryptBlock (var Buffer; BLen : cardinal);
begin
  fcrypt_decrypt(Buffer,BLen,FCryptCtx);
  end;

function TEncryption.ReadHeader (sSource : TStream) : boolean;
var
  SaltBuf     : TSaltBuf;
  FPV1,FPV2   : TPwdVerifier;
begin
  try
    // read salt value
    sSource.Read(SaltBuf,SaltLength[FMode]);
    fcrypt_init (FMode,PRawByteString(FPwd),length(FPwd),SaltBuf,FPV1,FCryptCtx);
    // read password verifier
    sSource.Read (FPV2,PwdVerLength);
    // adjust stream length of encrypted data
    result:=FPV1=FPV2;
  except
    result:=false;
    end;
  end;

function TEncryption.ReadTrailer (sSource : TStream) : boolean;
var
  MB1,MB2 : TMacBuffer;
begin
  FillChar(MB1,MacMax,0); FillChar(MB2,MacMax,0);
  fcrypt_end (MB1,FCryptCtx);
  try
    sSource.Read(MB2,MacLength);
    result:=MB1=MB2;
  except
    result:=false;
    end;
  end;

function TEncryption.DecryptStreamEx (sSource,sDest: TStream; SLength : int64; SpeedLimit : boolean;
                                      CancelDecryption : TStatusEvent) : TOpResult;
var
  NRead    : cardinal;
  Total    : int64;
begin
  Result:=orError; Total:=0;
  if ReadHeader (sSource) then begin
    SLength:=SLength-SaltLength[FMode]-PwdVerLength-MacLength;
    repeat
      if assigned(FEvent) then FEvent.WaitFor(INFINITE);
      if SLength<CBufSize then NRead:=SLength
      else NRead:=CBufSize;
      try
        NRead:=sSource.Read(CBuffer[0],NRead);
        inc(Total,NRead);
        DoProgress(acDecrypt,Total,SpeedLimit);
        DecryptBlock (CBuffer[0],NRead);
        if NRead>0 then begin
          if assigned(sDest) then sDest.Write(CBuffer[0],NRead);
          SLength:=SLength-NRead;
          end;
      except
        Exit;
        end;
      if assigned(CancelDecryption) and CancelDecryption then begin
        Result:=orUserBreak; Break;
        end;
      until (SLength<=0);
    if (Result<>orUserBreak) and ReadTrailer (sSource) then Result:=orOk;
    end;
  end;

function TEncryption.DecryptStream (sSource,sDest: TStream; SLength : int64; SpeedLimit : boolean) : boolean;
begin
  Result:=DecryptStreamEx (sSource,sDest,SLength,SpeedLimit,nil)=orOk;
  end;

{ ---------------------------------------------------------------------------- }

end.

