(* Verschlsselung von Strings nach AES (z.B.Passworte)
   ====================================================
   Unicode-Version (ab Delphi 2009)

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   Vers. 1 - Sep. 2003
   Vers. 2 - Aug. 2006
   Vers. 3 - Nov. 2011
   last updated: Feb. 2017
   *)

unit Crypt;

interface

const
  CrBufSize = 4096;
  defUserPwdLength = 64;

type
  TAscii85Buffer = array [0..3]of byte;
  TTextEncoding = (teAscii85,teHex);

// conversion to/from Ascii85
function ValueToAscii85 (const Value : TAscii85Buffer; Len : cardinal) : AnsiString;
function Ascii85ToValue (A85Str : AnsiString) : TAscii85Buffer;

// simple hex coding
function EncodeString (s : AnsiString) : AnsiString;
function DecodeString (s : AnsiString) : AnsiString;

// AES encryption
function EncryptString(const PwdKey,AString : AnsiString;
                       StrLength : integer; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
function DecryptString(const PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;

function EncryptPwdString(PwdKey,AString : AnsiString; DefLength : integer = defUserPwdLength;
                          TextEncoding : TTextEncoding = teAscii85) : AnsiString;
function DecryptPwdString(PwdKey,AString : AnsiString;
                          TextEncoding : TTextEncoding = teAscii85) : AnsiString;

function EncryptCmdString(PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
function DecryptCmdString(PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;

implementation

uses System.SysUtils, System.Classes, AesLib;

{- ASCII85 -------------------------------------------------------------}
// Adobe version used for Postscript
function ValueToAscii85 (const Value : TAscii85Buffer; Len : cardinal) : AnsiString;
var
  i   : integer;
  n   : cardinal;
  s   : AnsiString;
begin
  with LongRec(n) do begin
    Bytes[3]:=Value[0]; Bytes[2]:=Value[1];
    Bytes[1]:=Value[2]; Bytes[0]:=Value[3];
    end;
  if (n=0) and (Len>=4) then Result:='z'
  else begin
    s:='';
    for i:=0 to 4 do begin
      s:=AnsiChar(n mod 85 +33)+s;
      n:=n div 85;
      end;
    if Len<4 then Result:=copy(s,1,Len+1)
    else Result:=s;;
    end;
  end;

function Ascii85ToValue (A85Str : AnsiString) : TAscii85Buffer;
var
  i : integer;
  n : cardinal;
begin
  n:=0;
  while length(A85Str)<5 do A85Str:=A85Str+#33;
  for i:=1 to 5 do n:=85*n+ord(A85Str[i])-33;
  with LongRec(n) do begin
    Result[0]:=Bytes[3]; Result[1]:=Bytes[2];
    Result[2]:=Bytes[1]; Result[3]:=Bytes[0];
    end;
  end;

{ ---------------------------------------------------------------- }
type
  TCrReg = array [1..8] of byte;
  TByteArray = array of byte;

function ShiftReg (var CodeReg : TCrReg;
                   b,pb        : byte) : byte;
begin
  Result:=b xor pb xor CodeReg[3] xor CodeReg[8];
  move (CodeReg[1],CodeReg[2],7);
  end;

procedure SetByte (ba : TByteArray; ndx : integer; b : byte);
begin
  if ndx and 1 = 0 then begin // gerade
    ba[2*ndx]:=b and $F + random(16)*$10;
    ba[2*ndx+1]:=(b and $F0) div $10 + random(16)*$10;
    end
  else begin // ungerade
    ba[2*ndx]:=(b and $F0) + random(16);
    ba[2*ndx+1]:=(b and $F)*$10 + random(16);
    end;
  end;

function GetByte (ba : TByteArray; ndx : integer) : byte;
begin
  if ndx and 1 = 0 then begin // gerade
    Result:=ba[2*ndx] and $F + (ba[2*ndx+1] and $F)*$10;
    end
  else begin // ungerade
    Result:=ba[2*ndx] and $F0 + (ba[2*ndx+1] and $F0) div $10;
    end;
  end;

{ ---------------------------------------------------------------- }
// Routines to obscure plain text,
// e.g. to prevent simple reading of program internal passwords
const
  Key : byte = $93;

function EncodeString (s : AnsiString) : AnsiString;
var
  ba  : TByteArray;
  sn  : byte;
  n,i : integer;
begin
sn:=length(s);
  randomize;
  n:=2*(sn+1)+5+random(5);
  SetLength (ba,n);
  SetByte (ba,0,sn);
  for i:=1 to sn do SetByte (ba,i,ord(s[i]));
  for i:=2*(sn+1) to n-1 do ba[i]:=random(256);
  for i:=0 to n-1 do ba[i]:=ba[i] xor Key;
  Result:='';
  for i:=0 to n-1 do Result:=Result+IntToHex(ba[i],2);
  end;

function DecodeString (s : AnsiString) : AnsiString;
var
  ba  : TByteArray;
  sn  : byte;
  n,i : integer;
begin
  if length(s)>0 then begin
    n:=length(s) div 2;
    SetLength (ba,n);
    for i:=0 to n-1 do ba[i]:=StrToInt('$'+s[2*i+1]+s[2*i+2]);
    for i:=0 to n-1 do ba[i]:=ba[i] xor Key;
    sn:=GetByte(ba,0);
    Result:='';
    for i:=1 to sn do Result:=Result+chr(GetByte(ba,i));
    end
  else Result:='';
  end;

{ ---------------------------------------------------------------- }
// AES encryption - write string with leading length information
function EncryptString(const PwdKey,AString : AnsiString;
                       StrLength : integer; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
var
  ss,sd       : TMemoryStream;
  j,n         : cardinal;
  b           : byte;
  v           : TAscii85Buffer;
begin
  Result:='';
  if length(AString)=0 then Exit;
  ss:=TMemoryStream.Create;
  with ss do begin
    if StrLength>0 then begin
      if StrLength>222 then begin // new: two length bytes
        b:=LongRec(StrLength and $1F00).Bytes[1];
        Write(b,1);  // write 1st length byte (high order)
        b:=StrLength and $FF;
        Write(b,1);  // write 2nd length byte (low order)
        end
      else begin     // for compatibility to previous versions
        b:=(StrLength and $FF)+33;
        Write(b,1);  // write length byte
        end;
      end;
    Write(AString[1],length(AString));
    Seek(0,soFromBeginning);
    end;
  if length(PwdKey)>0 then begin
    sd:=TMemoryStream.Create;
    with TEncryption.Create(PwdKey,0,CrBufSize) do begin
      EncryptStream (ss,sd,false);
      Free;
      end;
    ss.Free;
    end
  else sd:=ss;
  sd.Seek(0,soFromBeginning);
  if TextEncoding=teAscii85 then begin
    n:=sd.Size;
    repeat
      j:=sd.Read(v,4);
      Result:=Result+ValueToAscii85(v,j);
      dec(n,j);
      until n=0;
    end
  else begin
    for j:=1 to sd.Size do begin
      sd.Read(b,1);
      Result:=Result+IntToHex(b,2);
      end;
    end;
  sd.Free;
  end;

// AES decryption - read string with leading length information
function DecryptString(const PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
var
  ss,sd       : TMemoryStream;
  j,n         : integer;
  b           : byte;
  v           : TAscii85Buffer;
  ok          : boolean;
begin
  Result:='';
  if length(AString)=0 then Exit;
  ss:=TMemoryStream.Create;
  n:=0;
  if TextEncoding=teAscii85 then begin
    repeat
      j:=length(AString)-n;
      if j>5 then j:=5;
      v:=Ascii85ToValue(copy(AString,n+1,j));
      ss.Write(v,j-1);
      inc(n,j);
      until n>=length(AString);
    end
  else begin
    repeat
      try
        j:=StrToInt('$'+copy(AString,n+1,2));
      except
        j:=0;
        end;
      ss.Write(j,1);
      inc(n,2);
      until n>=length(AString);
    end;
  if length(PwdKey)>0 then begin
    ss.Seek(0,soFromBeginning);
    sd:=TMemoryStream.Create;
    with TEncryption.Create(PwdKey,0,CrBufSize) do begin
      ok:=DecryptStream (ss,sd,ss.Size,false);
      Free;
      end;
    ss.Free;
    end
  else begin
    sd:=ss;
    ok:=true;
    end;
  if ok then with sd do begin
    sd.Seek(0,soFromBeginning);
    Read(b,1);
    if b>32 then n:=b-33
    else begin
      n:=256*b;
      Read(b,1);
      n:=n+b;
      end;
    SetLength(Result,n);
    Read(Result[1],n);
    end;
  sd.Free;
  end;

function EncryptPwdString(PwdKey,AString : AnsiString; DefLength : integer = defUserPwdLength;
                          TextEncoding : TTextEncoding = teAscii85) : AnsiString;
var
  i,nl : integer;
begin
  if length(AString)=0 then Result:=''
  else begin
    if length(AString)>DefLength then AString:=copy(AString,1,DefLength);
    nl:=length(AString);
    randomize;
    for i:=1 to DefLength-nl-1 do AString:=AString+chr(random(64)+33);
    Result:=EncryptString(PwdKey,AString,nl,TextEncoding);
    end;
  end;

function EncryptCmdString(PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
// max. Lnge von AString = 222
var
  i,nl : integer;
begin
  if length(AString)=0 then Result:=''
  else begin
    nl:=length(AString);
    if nl<32 then begin
      randomize;
      for i:=1 to 32-nl do AString:=AString+chr(random(64)+33);
      end;
    Result:=EncryptString(PwdKey,AString,nl,TextEncoding);
    end;
  end;

function DecryptPwdString(PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
begin
  if length(AString)=0 then Result:=''
  else Result:=DecryptString(PwdKey,AString,TextEncoding);
  end;

function DecryptCmdString(PwdKey,AString : AnsiString; TextEncoding : TTextEncoding = teAscii85) : AnsiString;
begin
  Result:=DecryptPwdString(PwdKey,AString,TextEncoding);
  end;

end.

