(* Delphi unit
   Copy files using file transfer protocol (FTP)

   using Indy 10 routines
   ======================

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   based on FtpFileTools Vers. 3.4 (no compressing, no encryption)
   Current: Indy 10.6.2.0

   last modified: February 2023
   *)

unit FtpCopyUtils;

interface

uses Vcl.Forms, Winapi.Windows, System.Classes, System.SysUtils,
  CBFunctions, StringUtils, FileUtils, FtpUtils, IdFTP, IdSSLOpenSSL,
  IdAllFTPListParsers, IdComponent, IdGlobal, IdLogFile;

type
  TCopyThread = class (TThread)
  private
    FOnProgress      : TFileProgressEvent;
    FCheckTime       : boolean;           // check for illegal filetime
    FSysError        : cardinal;
    FEnableSpLimit   : boolean;
    function GetDone : boolean;
    procedure SetProgress (ACallBack : TFileProgressEvent);
    procedure SetTempDirectory (const ATempDir : string);
  protected
    FError,FErrFlag  : integer;
    FSysErrMsg,
    SourceName,
    DestName,
    SourceNameXL,
    fTmpDir          : string;
    FBuffer          : array of byte;
    sSource,sDest    : TStream;
    FBufSize         : cardinal;
    FUserBreak,
    FCopyTimeStamp,
    FVerify          : boolean;
    FAction          : TFileAction;
    FCount           : int64;
    function CompareStreams (fSource,fDest : TStream; ASpeedLimit : boolean = false) : integer;
    procedure EndThread (Error : integer); virtual;
    procedure UpdateProgress;
    procedure DoProgress (AAction : TFileAction; ACount : int64; ASpeedLimit : boolean = false); virtual;
    procedure Execute; override;
  public
  // ASourceName  = source file,  ADestName = dest. file
    constructor Create (const ASourceName,ADestName : string;
                        ASuspend : Boolean = false; ABufSize : integer = defFtpBufferSize;
                        APriority : TThreadPriority = tpNormal);
    destructor Destroy; override;
    procedure CancelThread; virtual;
    property CheckTime : boolean read FCheckTime write FCheckTime;
    property CopyTimeStamp : boolean read FCopyTimeStamp write FCopyTimeStamp;
    property Done  : boolean read GetDone;
    property ErrorType : integer read FErrFlag;
    property ErrorCode : integer read FError;
    property OnProgress : TFileProgressEvent read FOnProgress write SetProgress;
    property SysErrorMsg : string read FSysErrMsg;
    property TempDir : string read fTmpDir write SetTempDirectory;
    end;

  // Write file via FTP
  TWriteFtpThread = class (TCopyThread)
  private
    FFtp      : TExtFtp;
    FDestDir  : string;
    FFileSize : int64;
  protected
    function WriteFile : integer;
    function VerifyFile (fs : TStream) : integer;
    procedure Execute; override;
    procedure EndThread (Error : integer); override;
    procedure ShowWork(Sender: TObject; AWorkMode: TWorkMode; AWorkCount: Int64);
  public
  // ASourceName  = local source file,  ADestName =  FTP dest file,
    constructor Create (AFtp : TExtFtp; const ASourceName,ADestDir,ADestName : string;
                        AVerify,ASuspend : Boolean; ABufSize : integer; APriority : TThreadPriority);
    destructor Destroy; override;
    procedure CancelThread; override;
    property DestSize : int64 read FFileSize;
    end;

  // Read file via FTP
  TReadFtpThread = class (TCopyThread)
  private
    FFtp      : TExtFtp;
    FModTime  : cardinal;
    FFileSize : int64;
  protected
    function ReadFile : integer;
    procedure Execute; override;
    procedure EndThread (Error : integer); override;
    procedure ShowWork(Sender: TObject; AWorkMode: TWorkMode; AWorkCount: Int64);
    property DestSize : int64 read FFileSize;
  public
  // ASourceName  = FTP source file,  ADestName =  local dest file,
    constructor Create (AFtp : TExtFtp; const ASourceName,ADestName : string; AFileSize : int64;
                        ASuspend : Boolean; ABufSize : integer; APriority : TThreadPriority);
    destructor Destroy; override;
    procedure CancelThread; override;
    property ModTime : cardinal read FModTime;  // Unix time
    end;

  TCompareFtpThread = class (TReadFtpThread)
  protected
    function CompareFile : integer;
    procedure Execute; override;
  public
    property DestSize;
    end;

implementation

uses XlFileUtils, FileErrors, FileConsts, WinApiUtils, ExtSysUtils, PathUtils;

const
  tmpCompName = 'compare.tmp';
  UMaxMemSize = 50*1024*1024;  // max. size of temp. memory stream (uncompressed)

{ ------------------------------------------------------------------- }
constructor TCopyThread.Create (const ASourceName,ADestName : string;
                                ASuspend : Boolean; ABufSize : integer;
                                APriority : TThreadPriority);
begin
  inherited Create (ASuspend);
  SourceName:=ASourceName;
  DestName:=ADestName;
  SourceNameXL:=FilenameToXL(ASourceName);
  Priority:=APriority;
  FOnProgress:=nil;
  FCopyTimeStamp:=true;
  FCheckTime:=true;
  FVerify:=false;
  FError:=errOK; FErrFlag:=errCopy;
  FSysError:=ERROR_SUCCESS; FSysErrMsg:='';
  FBufSize:=ABufSize;
  SetLength(FBuffer,FBufSize);
  FUserBreak:=false;
  fTmpDir:=TempDirectory;
  end;

destructor TCopyThread.Destroy;
begin
  FBuffer:=nil;
  inherited Destroy;
  end;

function TCopyThread.GetDone : boolean;
begin
  Result:=Terminated;
  end;

procedure TCopyThread.SetProgress (ACallBack : TFileProgressEvent);
begin
  FOnProgress:=ACallBack;
  end;

procedure TCopyThread.SetTempDirectory (const ATempDir : string);
begin
  if (length(ATempDir)>0) and DirectoryExists(ATempDir) then fTmpDir:=SetDirName(ATempDir)
  else fTmpDir:=TempDirectory;
  end;

procedure TCopyThread.EndThread (Error : integer);
begin
  if FSysError<>NO_ERROR then FSysErrMsg:=TryFormat(rsErrSystem,[SysErrorMessage(FSysError)]);
  if Error<>errOK then FError:=FErrFlag or Error;
  Terminate;
  end;

procedure TCopyThread.CancelThread;
begin
  FUserBreak:=true;
  end;

{ ------------------------------------------------------------------- }
function TCopyThread.CompareStreams (fSource,fDest : TStream; ASpeedLimit : boolean) : integer;
var
  NRead1,NRead2,fb : integer;
  CBuffer          : array of byte;
  SLength,STotal   : int64;
begin
  sLength:=fSource.Size;
  SetLength(CBuffer,FBufSize);
  STotal:=0;
  DoProgress(acVerify,-SLength);
  if SLength<>fDest.Size then Result:=errError+errSzMismatch
  else begin
    fSource.Seek(0,soFromBeginning);
    fDest.Seek(0,soFromBeginning);
    Result:=errOK;
    repeat
      if SLength<FBufSize then fb:=SLength else fb:=FBufSize;
      try
        NRead1:=fSource.Read(FBuffer[0],fb);
        if NRead1<fb then begin
          Result:=errError+errFileRead;   // z.B. wenn "sSource" gelockt ist
          FSysError:=GetLastError;
          end
        else begin
          STotal:=STotal+NRead1;        // total number of bytes read
          DoProgress(acVerify,STotal,ASpeedLimit);
          end;
      except
        on E:Exception do begin
          FSysErrMsg:=E.Message;
          Result:=errError+errFileRead;
          end;
        end;
      if Result=errOK then begin
        try
          NRead2:=fDest.Read(CBuffer[0],NRead1);
          if NRead2<NRead1 then begin
            Result:=errError2+errFileRead;    // z.B. wenn "sDest" gelockt ist
            FSysError:=GetLastError;
            end;
        except
          on E:Exception do begin
            FSysErrMsg:=E.Message;
            Result:=errError2+errFileRead;
            end;
          end;
        if FUserBreak then Result:=errUserBreak;
        end;
      dec(Slength,NRead1);
      if (Result=errOK) and not CompareMem(@FBuffer[0],@CBuffer[0],NRead1) then
        Result:=errError+errCompare;
      until (SLength<=0) or (Result<>errOK);
    end;
  CBuffer:=nil;
  end;

{ ------------------------------------------------------------------- }
procedure TCopyThread.DoProgress (AAction : TFileAction; ACount : int64; ASpeedLimit : boolean);
begin
  if Assigned(FOnProgress) then begin
    FAction:=AAction; FCount:=ACount; FEnableSpLimit:=ASpeedLimit;
    Synchronize(UpdateProgress);
    end;
  end;

procedure TCopyThread.UpdateProgress;
begin
  FOnProgress(FAction,FCount,FEnableSpLimit);
  end;

{ ------------------------------------------------------------------- }
// execute thread
procedure TCopyThread.Execute;
begin
  EndThread (errOK);
  end;

{ TWriteFtpThread ------------------------------------------------------------- }
// Write file via FTP
constructor TWriteFtpThread.Create (AFtp : TExtFtp;
                                    const ASourceName,ADestDir,ADestName : string;
                                    AVerify,ASuspend : Boolean; ABufSize : integer;
                                    APriority : TThreadPriority);
begin
  inherited Create (ASourceName,ADestName,ASuspend,ABufSize,APriority);
  FFtp:=AFtp;
  FVerify:=AVerify;
  FErrFlag:=errCopy;
  with FFtp do begin
    ValidDirectoryListing:=false;
    OnWork:=ShowWork;
    end;
  FDestDir:=ADestDir; FFileSize:=0;
  end;

destructor TWriteFtpThread.Destroy;
begin
  FFtp.OnWork:=nil;
  inherited Destroy;
  end;

procedure TWriteFtpThread.ShowWork(Sender: TObject; AWorkMode: TWorkMode; AWorkCount: Int64);
begin
  if AWorkMode=wmWrite then DoProgress(acFtpWrite,AWorkCount,true)
  else if AWorkMode=wmRead then DoProgress(acFtpRead,AWorkCount,true)
  end;

procedure TWriteFtpThread.CancelThread;
begin
  FUserBreak:=true;
  with FFtp do if Connected then begin
    AbortTransfer;
    Sleep(200);
    end;
  while not Done do begin
    Sleep(10);
    Application.ProcessMessages;
    end;
  end;

procedure TWriteFtpThread.EndThread (Error : integer);
begin
//  if (Error and errAllCodes)=errUserBreak then FFtp.Abort;
  inherited EndThread(Error);
  end;

{ ------------------------------------------------------------------- }
function TWriteFtpThread.WriteFile : integer;
var
  ec,km         : integer;
  FTime         : TFileTime;
  utime         : cardinal;
  fp            : int64;
  tzn,ten       : string;
begin
  ec:=errOK;
  if not FileExistsXL(SourceName) then ec:=errError+errFileOpen;
  if (ec=errOK) and (length(DestName)=0) then ec:=errError+errFileCreate;
  if ec=errOK then begin
    try
      sSource:=TFileStream.Create(SourceNameXL,fmOpenRead+fmShareDenyNone);
    except
      ec:=errError+errFileOpen;
      end;
    end;
  if ec=errOK then begin
    utime:=FileTimeToUnixTime(GetFileLastWriteTime(SourceNameXL));
    DoProgress(acFtpWrite,-sSource.Size,false);   // Reset progress indicator
    ec:=FFtp.Flush(sSource,DestName,false);
    FFileSize:=sSource.Size;
    if FUserBreak then begin
      Sleep(1000);     // auf Abort warten
      ec:=errUserBreak;
      end;
    if FVerify and (ec=errOK) then ec:=VerifyFile(sSource);
    try sSource.Free; except end;
    end;
  Result:=ec;
  end;

function TWriteFtpThread.VerifyFile (fs : TStream) : integer;
var
  tff    : boolean;
  sRead  : TStream;
  tzf    : string;
begin
  DoProgress(acVerify,-fs.Size,false);   // Reset progress indicator
  tff:=fs.Size>UMaxMemSize;
  if tff then begin
    tzf:=FFtp.TempDir+tmpCompName;
    sRead:=TFileStream.Create(tzf,fmCreate); // write to disk
    end
  else sRead:=TMemoryStream.Create;
  DoProgress(acFtpRead,-fs.Size,false);   // Reset progress indicator
  Result:=FFtp.Fill(DestName,sRead);  // read  from FTP to stream
  if Result=errOK then Result:=CompareStreams(fs,sRead,false);
  end;

procedure TWriteFtpThread.Execute;
var
  ec       : integer;
  FTime    : TFileTime;
  tff      : boolean;
  tzn      : string;
begin
  EndThread (WriteFile);
  end;

{ TReadFtpThread ------------------------------------------------------------- }
  // Read file via FTP
constructor TReadFtpThread.Create (AFtp : TExtFtp;
                                   const ASourceName,ADestName : string;
                                   AFileSize : int64;
                                   ASuspend : Boolean;
                                   ABufSize : integer;
                                   APriority : TThreadPriority);
begin
  inherited Create (ASourceName,ADestName,ASuspend,ABufSize,APriority);
  FFtp:=AFtp;
  FModTime:=0;
  FFileSize:=AFileSize;
  FErrFlag:=errCopy;
  FFtp.OnWork:=ShowWork;
  end;

destructor TReadFtpThread.Destroy;
begin
  FFtp.OnWork:=nil;
  inherited Destroy;
  end;

procedure TReadFtpThread.ShowWork(Sender: TObject; AWorkMode: TWorkMode; AWorkCount: Int64);
begin
  if AWorkMode=wmRead then DoProgress(acFtpRead,AWorkCount,true);
  end;

procedure TReadFtpThread.CancelThread;
begin
  FUserBreak:=true;
  with FFtp do if Connected then begin
    AbortTransfer;
    end;
  while not Done do begin
    Sleep(10);
    Application.ProcessMessages;
    end;
  end;

procedure TReadFtpThread.EndThread (Error : integer);
begin
//  if (Error and errAllCodes)=errUserBreak then FFtp.Abort;
  inherited EndThread(Error);
  end;

{ ------------------------------------------------------------------- }
function TReadFtpThread.ReadFile : integer;
var
  ec,FEncMode   : integer;
  tzn,ten,s     : string;
begin
  ec:=errOK;
  if (length(DestName)=0) then ec:=errError+errFileCreate;
  if ec=errOK then begin
    try
      sDest:=TFileStream.Create(DestName,fmCreate);
    except
      ec:=errError+errFileCreate;
      end;
    end;
  if ec=errOK then begin
    DoProgress(acFtpRead,-FFileSize,false);   // Reset progress indicator
    ec:=FFtp.Fill(SourceName,sDest);
    if FUserBreak then begin
      Sleep(1000);     // auf Abort warten
      ec:=errUserBreak;
      end;
    try sDest.Free; except end;
    end;
  Result:=ec;
  end;

procedure TReadFtpThread.Execute;
begin
  EndThread (ReadFile);
  end;

{ TCompareFtpThread ------------------------------------------------------------- }
  // Compare file from FTP with local file
function TCompareFtpThread.CompareFile : integer;
var
  ec,FEncMode   : integer;
  FEnc,tff,tuf  : boolean;
  sUnp,
  sGZip,sEnc    : TStream;
  CSize,USize   : int64;
  tzn,ten,tun   : string;
begin
  ec:=errOK;
  if not FileExists(DestName) then ec:=errError+errNotFound;
  if ec=errOK then begin
    try
      sDest:=TFileStream.Create(DestName,fmOpenRead);
    except
      ec:=errError+errFileCreate;
      end;
    end;
  if ec=errOK then begin
    tuf:=FFileSize>UMaxMemSize;
    if tuf then begin
      tun:=FFtp.TempDir+tmpCompName;
      sUnp:=TFileStream.Create(tun,fmCreate); // write to disk
      end
    else sUnp:=TMemoryStream.Create;
    DoProgress(acFtpRead,-FFileSize,false);   // Reset progress indicator
    ec:=FFtp.Fill(SourceName,sUnp);
    if (ec=errOK) and FUserBreak then begin
      Sleep(1000);     // auf Abort warten
      ec:=errUserBreak;
      end;
    if ec=errOK then ec:=CompareStreams(sDest,sUnp,false);
    sUnp.Free;
    if tuf then DeleteFile(tun);
    try sDest.Free; except end;
    end;
  Result:=ec;
  end;

procedure TCompareFtpThread.Execute;
begin
  EndThread (CompareFile);
  end;

end.
