(* Delphi Dialog
   Dialog: Eingabe der FTP-Verbindungsparameter
   ============================================

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.
    
   Vers. 1 - Dez. 2006
   Vers. 2 - Jul. 2007
   Vers. 3 - Aug. 2008   
   last modified: October 2021
   *)

unit FtpDlg;

interface

uses Winapi.Windows, System.SysUtils, System.Classes, Vcl.Graphics, Vcl.Forms,
  Vcl.Controls, Vcl.StdCtrls, Vcl.Buttons, Vcl.ExtCtrls,
  NumberEd, FtpUtils, IdAssignedNumbers;

const
  defFtpPort = IdPORT_FTP;
  defProxyPort = IdPORT_FTP_PROXY;

type
  TFtpDialog = class(TForm)
    CancelBtn: TBitBtn;
    OKBtn: TBitBtn;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    edtFtpPwd: TEdit;
    Label5: TLabel;
    edtPort: TRangeEdit;
    btnShow: TSpeedButton;
    cbxCase: TComboBox;
    Label6: TLabel;
    cbProxy: TCheckBox;
    edProxy: TEdit;
    btProxy: TBitBtn;
    pnSettings: TPanel;
    pnTitle: TPanel;
    edName: TLabeledEdit;
    gbOptions: TGroupBox;
    cbPassive: TCheckBox;
    cbUseHost: TCheckBox;
    cbUtf8: TCheckBox;
    gbSecure: TGroupBox;
    rbNone: TRadioButton;
    rbAvail: TRadioButton;
    rbAlways: TRadioButton;
    rbRequired: TRadioButton;
    cbFtpLog: TCheckBox;
    cbQuotes: TCheckBox;
    cbServer: TComboBox;
    cbUser: TComboBox;
    cbDir: TComboBox;
    procedure btnShowClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure cbProxyClick(Sender: TObject);
    procedure btProxyClick(Sender: TObject);
    procedure rbNoneClick(Sender: TObject);
    procedure rbAvailClick(Sender: TObject);
    procedure rbRequiredClick(Sender: TObject);
    procedure rbAlwaysClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormDestroy(Sender: TObject);
  private
    { Private-Deklarationen }
    FIniName : string;
    sp    : boolean;
    FSecureMode : integer;
    FProxyPar   : TFtpProxyParams;
{$IFDEF HDPI}   // scale glyphs and images for High DPI
    procedure AfterConstruction; override;
{$EndIf}
    procedure ShowProxy;
  public
    { Public-Deklarationen }
    procedure Init (const AIniName,DefIni : string);
    function DefaultValues : TFtpParams;
    function Execute (var AFtpPar : TFtpParams; ShowPwd,ShowSecure : boolean;
                      ShowTitle : boolean = false) : boolean;
  end;

function GetFtpHint (FtpPar : TFtpParams; ShowDir : boolean = true) : string;

function GetFtpSecMode (SecMode : integer) : string;

var
  FtpDialog: TFtpDialog;

implementation

{$R *.dfm}

uses IniFiles, IdFtp, StringUtils, WinUtils, ListUtils, MsgDialogs, GnuGetText,
  FtpProxyDlg;

{ ------------------------------------------------------------------- }
function GetFtpHint(FtpPar : TFtpParams; ShowDir : boolean) : string;
begin
  with FtpPar do begin
    if SecureMode>0 then Result:='ftps' else Result:='ftp';
    Result:=Result+'://'+Host+Colon+IntToStr(Port);
    if ShowDir then Result:=Result+'/'+Directory;
    end;
  end;

function GetFtpSecMode (SecMode : integer) : string;
begin
  case SecMode of
  1 : Result:=FtpDialog.rbAvail.Caption;
  2 : Result:=FtpDialog.rbRequired.Caption;
  3 : Result:=FtpDialog.rbAlways.Caption;
  else Result:=FtpDialog.rbNone.Caption;
    end;
  Result:=dgettext('dialogs','Secure mode: ')+sLineBreak+Result;
  end;

{ ------------------------------------------------------------------- }
procedure TFtpDialog.FormCreate(Sender: TObject);
begin
  TranslateComponent (self,'dialogs');
  FIniName:=''; sp:=true;
  end;

{$IFDEF HDPI}   // scale glyphs and images for High DPI
procedure TFtpDialog.AfterConstruction;
begin
  inherited;
  if Application.Tag=0 then
    ScaleButtonGlyphs(self,PixelsPerInchOnDesign,Monitor.PixelsPerInch);
  end;
{$EndIf}

const
  ServerSekt = 'FtpServer';
  UserSekt   = 'Usernames';
  DirSekt    = 'FtpDirs';

procedure TFtpDialog.FormDestroy(Sender: TObject);
var
  IniFile : TMemIniFile;
begin
  if length(FIniName)>0 then begin
    IniFile:=TMemIniFile.Create(FIniName);
    SaveHistory(IniFile,ServerSekt,true,cbServer.Items);
    SaveHistory(IniFile,UserSekt,true,cbUser.Items);
    SaveHistory(IniFile,DirSekt,true,cbDir.Items);
    IniFile.UpdateFile;
    IniFile.Free;
    end;
  end;

procedure TFtpDialog.Init (const AIniName,DefIni : string);
var
  si      : string;
  IniFile : TMemIniFile;
begin
  // nur beim Neustart laden
  if (length(FIniName)=0) and ((length(AIniName)>0) or (length(DefIni)>0)) then begin
    if FileExists(AIniName) then si:=AIniName else si:=DefIni;
    FIniName:=si;
    if FileExists(si) then begin
      IniFile:=TMemIniFile.Create(FIniName);
      LoadHistory(IniFile,ServerSekt,cbServer.Items);
      LoadHistory(IniFile,UserSekt,cbUser.Items);
      LoadHistory(IniFile,DirSekt,cbDir.Items);
      end;
    end;
  end;

function TFtpDialog.DefaultValues : TFtpParams;
begin
  with Result do begin
    Host:='';
    Port:=defFtpPort;
    Username:='';
    Password:='';
    Directory:='';
    SecureMode:=0;
    ConnTimeout:=defTimeOut;
    ConnAttempts:=defConnect;
    Passive:=true;
    UseHost:=false;
    WriteLog:=false;
    ForceUtf8:=false;
    CaseMode:=tcNone;
    Quotes:=false;
    InvalidChars:='';
    with Proxy do begin
      Server:='';
      Username:='';
      Password:='';
      Port:=defProxyPort;
      Mode:=fpcmNone;
      end;
    end;
  end;

procedure TFtpDialog.btnShowClick(Sender: TObject);
begin
  with edtFtpPwd do if btnShow.Down then PasswordChar:=#0 else PasswordChar:='*';
  end;

procedure TFtpDialog.ShowProxy;
begin
  with FProxyPar do begin
    if cbProxy.Checked and (length(Server)>0) then edProxy.Text:=Server+':'+IntToStr(Port)
    else edProxy.Text:='';
    edProxy.Enabled:=cbProxy.Checked;
    btProxy.Enabled:=cbProxy.Checked;
    end;
  end;

procedure TFtpDialog.cbProxyClick(Sender: TObject);
begin
  if Visible and cbProxy.Checked then begin
    EditFtpProxy(FProxyPar,sp);
    cbProxy.Checked:=FProxyPar.Mode<>fpcmNone;
    end;
  ShowProxy;
  end;

procedure TFtpDialog.btProxyClick(Sender: TObject);
begin
  FtpProxyDialog.Execute (FProxyPar,sp);
  cbProxy.Checked:=FProxyPar.Mode<>fpcmNone;
  ShowProxy;
  end;

procedure TFtpDialog.rbNoneClick(Sender: TObject);
begin
  fSecureMode:=0;
  end;

procedure TFtpDialog.rbRequiredClick(Sender: TObject);
begin
  fSecureMode:=2;
  end;

procedure TFtpDialog.rbAvailClick(Sender: TObject);
begin
  fSecureMode:=1;
  end;

procedure TFtpDialog.rbAlwaysClick(Sender: TObject);
begin
  fSecureMode:=3;
  end;

procedure TFtpDialog.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  if ModalResult=mrOK then begin
    CanClose:=(length(cbServer.Text)>0) or (length(cbUser.Text)>0);
    if not Canclose then begin
      if length(cbServer.Text)=0 then begin
        ErrorDialog('',dgettext('dialogs','No host specified!'));
        cbServer.SetFocus;
        end
      else begin
        ErrorDialog('',dgettext('dialogs','No user name specified!'));
        cbUser.SetFocus;
        end;
      end;
    end;
  end;

function TFtpDialog.Execute (var AFtpPar : TFtpParams; ShowPwd,ShowSecure : boolean;
                             ShowTitle : boolean) : boolean;
var
  i : integer;
begin
  sp:=ShowPwd;
  btnShow.Visible:=ShowPwd;
  with gbSecure do begin
    Enabled:=ShowSecure;
    for i:=0 to ControlCount-1 do Controls[i].Enabled:=ShowSecure;
    end;
  if ShowTitle then begin
    pnTitle.Show;
    ClientHeight:=pnSettings.Height+pnTitle.Height;
    end
  else begin
    ClientHeight:=pnSettings.Height;
    pnTitle.Hide;
    end;
  with AFtpPar do begin
    edName.Text:=Caption;
    cbServer.Text:=Host;
    edtPort.Value:=Port;
    cbUser.Text:=Username;
    edtFtpPwd.Text:=Password;
    cbDir.Text:=Directory;
    if not ShowSecure then SecureMode:=0;  // secure mode not available
    fSecureMode:=SecureMode;
    case SecureMode of
    1 : rbAvail.Checked:=true;
    2 : rbRequired.Checked:=true;
    3 : rbAlways.Checked:=true;
    else rbNone.Checked:=true;
      end;
    cbPassive.Checked:=Passive;
    cbUseHost.Checked:=UseHost;
    cbFtpLog.Checked:=WriteLog;
    cbUtf8.Checked:=ForceUtf8;
    cbxCase.ItemIndex:=integer(CaseMode);
    FProxyPar:=Proxy;
    cbProxy.Checked:=FProxyPar.Mode<>fpcmNone;
    ShowProxy;
    if ShowModal=mrOK then begin
      Caption:=edName.Text;
      Host:=cbServer.Text;
      AddToHistory(cbServer,Host);
      Port:=edtPort.Value;
      Username:=cbUser.Text;
      AddToHistory(cbUser,Username);
      Password:=edtFtpPwd.Text;
      Directory:=DosPathToUnixPath(cbDir.Text);
      AddToHistory(cbDir,Directory);
      if length(Caption)=0 then Caption:=GetFtpHint(AFtpPar);
      SecureMode:=fSecureMode;
      Passive:=cbPassive.Checked;
      UseHost:=cbUseHost.Checked;
      WriteLog:=cbFtpLog.Checked;
      ForceUtf8:=cbUtf8.Checked;
      CaseMode:=TTextChange(cbxCase.ItemIndex);
      if not cbProxy.Checked then FProxyPar.Mode:=fpcmNone;
      Proxy:=FProxyPar;
      Result:=true;
      end
    else Result:=false;
    end;
  end;

end.
