(* Dialog for FTP proxy connection parameters

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   Vers. 1 - Jul. 2007
   last modified: July 2022
   *)

unit FtpProxyDlg;

interface

uses Winapi.Windows, System.SysUtils, System.Classes, Vcl.Graphics, Vcl.Forms,
  Vcl.Controls, Vcl.StdCtrls, Vcl.Buttons, Vcl.ExtCtrls, NumberEd, FtpUtils;

type
  TFtpProxyDialog = class(TForm)
    OKBtn: TBitBtn;
    CancelBtn: TBitBtn;
    rgMethod: TRadioGroup;
    cbShowPwd: TCheckBox;
    edProxy: TLabeledEdit;
    edUser: TLabeledEdit;
    edProxyPwd: TLabeledEdit;
    edtPort: TLabeledEdit;
    procedure cbShowPwdClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  private
    { Private declarations }
  public
    { Public declarations }
{$IFDEF HDPI}   // scale glyphs and images for High DPI
    procedure AfterConstruction; override;
{$EndIf}
    function Execute (var AProxyPar : TFtpProxyParams; ShowPwd : boolean) : boolean;
  end;

function EditFtpProxy (var AProxyPar : TFtpProxyParams; ShowPwd : boolean) : boolean;

var
  FtpProxyDialog: TFtpProxyDialog;

implementation

{$R *.dfm}

uses IdFtp, GnuGetText, ExtSysUtils,
  WinUtils, {$IFDEF ACCESSIBLE} ShowMessageDlg {$ELSE} MsgDialogs {$ENDIF};

{------------------------------------------------------------------- }
procedure TFtpProxyDialog.FormCreate(Sender: TObject);
begin
  TranslateComponent (self,'dialogs');
  end;

procedure TFtpProxyDialog.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
{$IFDEF ACCESSIBLE}
  if (Key=VK_F11) then begin
    if ActiveControl is TCustomEdit then begin
      with (ActiveControl as TCustomEdit) do if length(TextHint)>0 then ShowHintInfo(TextHint)
      else if length(Hint)>0 then ShowHintInfo(Hint);
      end
    else with ActiveControl do if length(Hint)>0 then ShowHintInfo(Hint);
    end;
{$ENDIF}
  end;

{$IFDEF HDPI}   // scale glyphs and images for High DPI
procedure TFtpProxyDialog.AfterConstruction;
begin
  inherited;
  if Application.Tag=0 then
    ScaleButtonGlyphs(self,PixelsPerInchOnDesign,Monitor.PixelsPerInch);
  end;
{$EndIf}

procedure TFtpProxyDialog.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
var
  n : integer;
begin
  if (ModalResult=mrOK) then
    CanClose:=(TryStrToInt(edtPort.Text,n)) and (n>=0) and (n<=65535);
  if not CanClose then
    ErrorDialog(TryFormat(dgettext('dialogs','Invalid port number (%s)'),[edtPort.Text]));
  end;

{------------------------------------------------------------------- }
procedure TFtpProxyDialog.cbShowPwdClick(Sender: TObject);
var
  c : char;
begin
  if cbShowPwd.Checked then c:=#0 else c:='*';
  edProxyPwd.PasswordChar:=c;
  end;

{------------------------------------------------------------------- }
function TFtpProxyDialog.Execute (var AProxyPar : TFtpProxyParams; ShowPwd : boolean) : boolean;
begin
  cbShowPwd.Visible:=ShowPwd;
  with AProxyPar do begin
    edProxy.Text:=Server;
    edtPort.Text:=IntToStr(Port);
    edUser.Text:=Username;
    edProxyPwd.Text:=Password;
    with rgMethod do if Mode=fpcmNone then ItemIndex:=0
    else ItemIndex:=integer(Mode)-1;
    ActiveControl:=edProxy;
    if ShowModal=mrOK then begin
      Server:=edProxy.Text;
      Port:=StrToInt(edtPort.Text);
      Username:=edUser.Text;
      Password:=edProxyPwd.Text;
      Mode:=TIdFtpProxyType(rgMethod.ItemIndex+1);
      Result:=true;
      end
    else Result:=false;
    end;
  end;

{------------------------------------------------------------------- }
function EditFtpProxy (var AProxyPar : TFtpProxyParams; ShowPwd : boolean) : boolean;
begin
  if not assigned(FtpProxyDialog) then FtpProxyDialog:=TFtpProxyDialog.Create(Application);
  Result:=FtpProxyDialog.Execute(AProxyPar,ShowPwd);
  FreeAndNil(FtpProxyDialog);
  end;

end.
