(* Delphi-Unit
   FTP basic types and structures

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   based on FtpFileTools Vers. 3.4

   Required changes in Indy10 library files Vers. 10.6.2

   1. Protocols\IdFTP.pas
      - procedure DoAfterLogin; virtual;
      - procedure TIdFTP.SetModTime(const AFileName: String; const ALocalTime: TDateTime);
        ...
          SendCmd('MDTM '+ FTPDateTimeToMDTMD(ALocalTime, False, False) + ' ' + AFileName, [213,253], enDefault); {do not localize}
          // return code 213 is allowed
      - procedure TIdFTP.SetModTimeGMT(const AFileName: String; const AGMTTime: TDateTime);
        ...
          SendCmd('MDTM '+ FTPDateTimeToMDTMD(AGMTTime + OffSetFromUTC, False, False) + ' ' + AFileName, [213,253]); {do not localize}
          // return code 213 is allowed

   2. Protocols\
      // move to public:
      -  property ModifiedDateGMT : TDateTime read FModifiedDateGMT write FModifiedDateGMT; //JR - moved to public 08-12-06

   Current: Indy 10.6.2.0

   Vers. 1.0 - August 2020
   last modified: July 2024
*)

unit FtpUtils;

interface

uses System.Classes, Winapi.Windows, IdFTP, IdSSLOpenSSL, IdAllFTPListParsers, IdComponent,
  IdGlobal, IdLogFile, IdAssignedNumbers, StringUtils, CBFunctions;

const
  defTimeout = 60000;   // 60 s
  defTransferTimeout = 0;   // unlimited
  defConnect = 1;           // default number of attempts to connect
  defFtpBufferSize = 256*1024;   // default buffer size for FTP copy operations
  defFtpReplace = '_';
  defFtpKeepAlive = 0;      // = 1: Send SIO_KEEPALIVE_VALS to socket to keep
                            //      connection alive during data transfer
                            //      see: IdStackWindows.SetKeepAliveValues
  defFtpKeepAliveTime =  1000*30;     // 30 s
  defFtpKeepAliveInterval = 1000*10;  //  10 s

  fsNone = 0;  // secure modes
  fsExplicit = 1;
  fsRequired = 2;
  fsImplicit = 3;

  defLogName = 'FtpLog.txt';

  IdPORT_FTP_PROXY = 2121;
  defFtpPort = IdPORT_FTP;
  defProxyPort = IdPORT_FTP_PROXY;

type
  TFtpEncoding = (feASCII,fe8Bit,feUtf8);

  TFtpReplies = record
    Command : string;
    Reply   : integer
    end;

  TCertInfos = record
    CertDir,RootCertFile,CertFile,KeyFile,Password : string;
    end;

const
  // no longer supported, see above
  KeepAliveCommands : array[0..2] of TFtpReplies= ((Command: 'NOOP'; Reply : 200),
                                                   (Command: 'PWD'; Reply : 257),
                                                   (Command: 'SYST'; Reply : 215));

type
  TFtpProxyParams = record
    Server,
    Username       : string;
    Password       : AnsiString;
    Port           : integer;
    Mode           : TIdFtpProxyType;
    end;

  TCerticateInfo = record
    Version,SerialNumber : integer;
    Server,Issuer,Fingerprint : string;
    ValidUntil : TDateTime;
    procedure Init;
    end;

  TConfirmPeerCert = function (const ACertInfo : TCerticateInfo) : boolean of object;

  TFtpParams = record
    Caption,
    Host,Username,
    Directory,
    InvalidChars,
    Fingerprint         : string;
    Password            : AnsiString;
    Port,
    SecureMode,             // 0 = off, 1 = explicit, 2 = required, 3 = implicit
    TimeOffset,             // UTC offset in minutes
    ConnTimeout,
    ConnAttempts,
    ResponseTimeOut,
    CertVerify,                      // 0 = off, 1 = prompt, 2 = fingerprint
    KeepAlive,
    KeepAliveCmd        : integer;   // 0 = NOOP, 1 = PWD, 2 = SYST
    UseTimeOffset,
    Quotes,
    Passive,
    UseHost,
    WriteLog,
    ForceUtf8,                            // force UTF8 encoding
    UseIPv6,                              // use IPv6 connection
    UseExts,                              // use EPRT/EPSV for IPv6
    UseOpts             : boolean;        // send OPTS UTF8 ON
    CaseMode            : TTextChange;
    Proxy               : TFtpProxyParams
    end;

  TFtpFileInfo = class(TObject)
    FFileName : string;
    FSize     : Int64;
    FModifiedDate    : TDateTime;
    FModifiedDateGMT : TDateTime;
    constructor Create (const AFilename : string; ASize : int64; AMd,AMdG : TDateTime);
    end;

  TDirectoryList = class(TStringlist)
    constructor Create;
    destructor Destroy; override;
    procedure Clear; override;
    end;

  TFileList = class(TStringlist)
    constructor Create;
    destructor Destroy; override;
    procedure Clear; override;
    end;

  TGetPassword = procedure(var APassword : String) of object;

  TExtFtp = class (TIdFTP)
  private
    FList  : TFileList;
    CMode  : TTextChange;
    fQuotes,
    fTLS,
    fUseOpts,              // deprecated, see below
    fForceUtf8,
    fUseIPv6,
    fUseExts,
    fUTF8,
// KeepAlive: time specifications not required on Windows
// The default settings when a TCP socket is initialized sets the keep-alive timeout
// to 2 hours and the keep-alive interval to 1 second (Windows SDK).
    fUseKeepAlive,
    fAborted            : boolean;
    fDir,
    fCurrentDir,
    FChangeDirResponse,
    fRepl,
    fFingerPrint        : string;
    fCertInfo           : TCerticateInfo;
    fCertPwd            : AnsiString;
    fGetSSLPassword     : TGetPassword;
//    fVerifyPeer         : TVerifyPeerEvent;
    fConfirmCert        : TConfirmPeerCert;
    fEncoding           : TFtpEncoding;
    fIdLogFile          : TIdLogFile;
    fPeerFingerprint,
    fRoot,fCert,fKey,
    fTmpDir,fIdError    : string;
    fVerifyPeerMode,                          // 0 = off, 1 = prompt, 2 = fingerprint
    fTimeOffset,
    DCount              : integer;            // count "ChangeDir" in "ForceDir"
    procedure InitIOHandler;
    procedure SetTmpDir (ADir : string);
    function GetLastResponseCode : integer;
    function GetLastResponseString : string;
    function GetFeatureString : string;
    function GetEncoding : TFtpEncoding;
    procedure SetEncoding (Value : TFtpEncoding);
    function GetIPv6 : boolean;
    procedure SetIPv6 (Value : boolean);
    function GetServerName : string;
    function GetLogName : string;
    procedure SetLogName (Value : string);
    procedure SetWriteLog (Value : boolean);
    procedure LogReconnectError (const ErrMsg : string);
    function GetIdError : string;
    procedure SetIdError (const Value : string);
    function GetDirectoryListing : boolean;
  protected
    fValidDirectoryListing : boolean;
    procedure DoAfterLogin; override;
    procedure SetTLSMode(Value : integer);
    procedure GetCertPassword(var Password : String);
    function VerifyPeer (Certificate: TIdX509; AOk: Boolean; ADepth, AError: Integer): Boolean;
  public
    constructor Create (AOwner : TComponent; const Path,Root,Cert,Key : string; Pwd : AnsiString); overload;
    constructor Create (AOwner : TComponent); overload;
    destructor Destroy; override;
    procedure SetParams (const FtpPar : TFtpParams);
    procedure ForceEncoding (Value : TFtpEncoding; UseOpts : boolean = false);
    function ModifyName(const FName : string) : string;
    function RetrieveName(const FName : string) : string;
    function MakeQuotedPath (const s : string) : string;
    function GetFileList (FileList : TFileList) : boolean; overload;
    function GetFileList : boolean; overload;
    function FtpFileExists (const FName : string) : boolean;
    function DeleteSubDirFile (const ADir,FName : string) : integer;
    function DeleteFile (const FName : string; InitListing : boolean = true) : boolean;
    function DeleteExistingFile (const FName : string) : integer;
    function DeleteMatchingFiles (const AMask : string) : integer;
    function RenameFile (const OldName,NewName : string; InitListing : boolean = true) : boolean;
    function GetSize (FileList : TFileList; const FName : string) : int64; overload;
    function GetSize (const FName : string) : int64; overload;
    function CheckSize(const FName : string; ASize : int64) : boolean;
    function GetLocalFileTime (FileList : TFileList; const FName : string) : TFileTime; overload;
    function GetLocalFileTime (const FName : string) : TFileTime; overload;
    function GetModTime(const AFileName: String; AsGMT: Boolean): TDateTime;
    function GetTimeStamp (FileList : TFileList; const FName : string; AsGMT : Boolean = false) : TDateTime; overload;
    function GetTimeStamp (const FName : string; AsGMT : Boolean = false) : TDateTime; overload;
    function CheckTimeStamp (const FName : string; tm : TDateTime; Delta : integer;
                             var fdt : TDateTime; AsGMT : Boolean = false) : integer; overload;
    function CheckTimeStamp (const FName : string; tm : TDateTime;
                             Delta : integer; AsGMT : Boolean = false) : integer;  overload;
    function SetTimeStamp(const FName : string; ATime : TDateTime; UseGMT : Boolean = false) : boolean;
    function SetTimeStampFromFile(const ASourceFile,FName : string; CheckTime : boolean = true) : boolean;
    function GetDirList(DirList : TDirectoryList) : boolean;
    function FtpDirExists (const DName : string) : boolean;
    procedure FtpChangeDir (DName: string);
    function ChangeOneDirUp : boolean;
    function ChangeToDir(const DName: string) : boolean;
    function ChangeToNewDir(const DName: string) : integer;
    function ForceDir(const DName: string) : boolean;
    function ForceDirUp : boolean;
    function SetRemoteDir (const Dir : string) : boolean;
    function DeleteDir(DName: string) : boolean;
    function ReadCurrentDir (var Dir : string) : boolean;
    function GetCurrentDir : string;
    function DeleteEmptyDirs (const DName : string;  DelRoot : boolean;
                              var DCount : integer; WasCanceled : TStatusEvent = nil) : boolean;
    function DeleteTree(const DName : string; DelRoot : boolean; var FCount : integer) : boolean;
    function DirFiles(DName : string) : integer;
    function DoConnect (NRepeat : integer = 1; Verify : boolean = false) : boolean;
    function ReConnect(const Dir : string; Check : boolean = true; UpdateList : boolean = true) : boolean;
    function Connected : boolean; override;
    procedure AbortTransfer;
    function CopyFile (const ASourceFile,ADestFile : string) : integer;
    function Flush(ASource: TStream; const ADestFile: string; AAppend: boolean = false) : integer;
    function Fill(const ASourceFile : string; ADest : TStream) : integer;
    property CanUseTls : boolean read fTls;
    property CaseMode : TTextChange read CMode write CMode;
    property CapabilitiesString : string read GetFeatureString;
    property CertInfo : TCerticateInfo read fCertInfo;
    property ChangeDirResponse : string read FChangeDirResponse;
    property CurrentDir : string read fCurrentDir;
    property Encoding : TFtpEncoding read GetEncoding write SetEncoding;
    property FileList : TFileList read FList;
    property Fingerprint: string read fFingerprint;
    property ForceUtf8 : boolean read fForceUtf8 write fForceUtf8;
    property UseIPv6 : boolean read GetIPv6 write SetIPv6;
    property UseOpts : boolean read fUseOpts write fUseOpts;
    property UseExts : boolean read fUseExts write fUseExts;
    property IndyError : string read GetIdError write SetIdError;
    property LastResponseCode : integer read GetLastResponseCode;
    property LastResponseString : string read GetLastResponseString;
    property Logname : string read GetLogName write SetLogName;
    property PeerFingerprint : string read fPeerFingerprint write fPeerFingerprint;
    property RemoteDir : string read fDir;
    property ReplaceChars : string read fRepl write fRepl;
    property SecureTransfer : integer write SetTLSMode;
    property ServerName : string read GetServerName;
    property TempDir : string read fTmpDir write SetTmpDir;
    property TimeOffset : integer read fTimeOffset write fTimeOffset;
    property UseKeepAlive : boolean write FUseKeepAlive;
    property UseQuotes : boolean read fQuotes write fQuotes;
    property ValidDirectoryListing : boolean read fValidDirectoryListing write fValidDirectoryListing;
    property VerifyPeerMode : integer read fVerifyPeerMode write fVerifyPeerMode;
    property WriteLogFile : boolean write SetWriteLog;
    property OnSSLPassword : TGetPassword read fGetSSLPassword write fGetSSLPassword;
//    property OnSSLVerify : TVerifyPeerEvent read fVerifyPeer write fVerifyPeer;
    property OnConfirmCert : TConfirmPeerCert read fConfirmCert write fConfirmCert;
    end;

function DefaultFtpParams : TFtpParams;

// Bei Verzeichnisangaben Slash ergnzen
function IncludeTrailingSlash (const Dir : string) : string;
function IncludeLeadingSlash (const Dir : string) : string;
function RemoveLeadingSlash (const Dir : string) : string;
function RemoveTrailingSlash (const Dir : string) : string;
function IsAbsolutePath (const Dir : string) : boolean;
function DosPathToUnixPath(const Path: string): string;

// Den letzten Teilnamen extrahieren (siehe auch: ExtractFileName)
function ExtractLastName(const Delimiters,FileName: string): string;
function ExtractPath(const APath : string) : string;

// Unix-Pfad zusammensetzen
function ExpandUnixPath(const APath,AName : string) : string;

// Relativen Unix-Pfad extrahieren
function GetUnixRelativePath(const RootPath,Path : string) : string;

implementation

uses System.SysUtils, System.DateUtils, System.StrUtils, System.Masks, Vcl.Forms,
  IdFTPList, IdReplyRFC, IdReply, IdFTPCommon, IdIOHandler, IdStack,
  IdExplicitTLSClientServerBase, IdGlobalProtocols, IdException, IdExceptionCore,
  IdResourceStringsCore, FtpConsts, FileErrors, FileConsts,
  FileUtils, ExtSysUtils, WinApiUtils, XlFileUtils, PathUtils, WinUtils;

const
  LOldStr : array [0..2] of string =
    ( EOL,  CR, LF );
  LNewStr : array [0..2] of string =
    ( RSLogEOL, RSLogCR, RSLogLF );

  ReconnDelay = 200;  // delay for reconnections in ms

{ ------------------------------------------------------------------- }
procedure TCerticateInfo.Init;
begin
  Version:=0; SerialNumber:=0;
  Server:=''; Issuer:=''; Fingerprint:='';
  ValidUntil:=Now;
  end;

{ ------------------------------------------------------------------- }
constructor TExtFtp.Create (AOwner : TComponent; const Path,Root,Cert,Key : string; Pwd : AnsiString);
begin
  inherited Create(AOwner);
  UseMLIS:=true; ValidDirectoryListing:=false;
  Passive:=true; fUTF8:=false; fUseKeepAlive:=false; fQuotes:=false; fCertPwd:=''; UseHost:=true;
  TransferType:=ftBinary;    // in den verw. Indy-Routinen ist ASCII per default eingestellt
  FList:=TFileList.Create;
  CMode:=tcNone; fIdError:='';
  fDir:=''; fCurrentDir:='';
  fAborted:=false;
  fUseOpts:=false; fForceUtf8:=false; fUseIPv6:=false;
  fIdLogFile:=TIdLogFile.Create(nil);          //create the log file
  with NATKeepAlive do begin
    UseKeepAlive:=false;
    IdleTimeMS:=defFtpKeepAliveTime;
    IntervalMS:=defFtpKeepAliveInterval;
    end;
  with fidLogFile do begin
    FileName:=SetDirName(fTmpDir)+defLogName;
    end;
// if openSSL libraries and certificates are available, use SSL for TLS support
  fTLS:=(length(Cert)>0) and (length(Key)>0) and (length(Root)>0);
  if fTLS then begin
    try
      TIdSSLContext.Create.Free;
      fTLS:=true;
    except
      fTLS:=false;
      end;
    if fTLS then begin
      fRoot:=Path+Root; fCert:=Path+Cert; fKey:=Path+Key;
      fTLS:=FileExists(fRoot) and FileExists(fCert) and FileExists(fKey);
      end;
    end;
  fCertPwd:=Pwd; fGetSSLPassword:=nil; fConfirmCert:=nil;
  InitIOHandler; GetEncoding;
  UseTLS:=utNoTLSSupport;
  fCertInfo.Init;
  end;

procedure TExtFtp.InitIOHandler;
begin
  if IOHandler<>nil then Exit;
  if fTLS then begin
    IOHandler:=TIdSSLIOHandlerSocketOpenSSL.Create(self);
    with (IOHandler as TIdSSLIOHandlerSocketOpenSSL) do begin
      with SSLOptions do begin
        Mode:=sslmClient;
        RootCertFile:=fRoot; CertFile:=fCert; KeyFile:=fKey;
        SSLVersions:=[sslvTLSv1,sslvTLSv1_1,sslvTLSv1_2];
        VerifyMode:=[sslvrfPeer];
        end;
      OnGetPassword:=GetCertPassword;
      OnVerifyPeer:=VerifyPeer;
      end;
    end
  else IOHandler:=TIdIOHandler.MakeDefaultIOHandler(self);
  IOHandler.Intercept:=fIdLogFile;
  ManagedIOHandler:=True;
  end;

constructor TExtFtp.Create (AOwner : TComponent);
begin
  Create(AOwner,'','','','','');
  end;

destructor TExtFtp.Destroy;
begin
  fIdLogFile.Free;
  FList.Free;
  inherited Destroy;
  end;

procedure TExtFtp.DoAfterLogin;
begin
  fUTF8:=IsExtSupported('UTF8');
  GetEncoding;
  if Assigned(FOnAfterClientLogin) then begin
    OnAfterClientLogin(Self);
    end;
  end;

procedure TExtFtp.SetTmpDir (ADir : string);
begin
  fTmpDir:=SetDirName(ADir);
  end;

function TExtFtp.GetLastResponseCode : integer;
begin
  Result:=LastCmdResult.NumericCode;
  end;

function TExtFtp.GetLastResponseString : string;
begin
  Result:=LastCmdResult.Code;
  with LastCmdResult.Text do if Count>0 then Result:=Result+' '+Strings[0];
  if (length(fIdError)>0) and not AnsiContainsText(Result,fIdError) then
    Result:=Result+' ('+fIdError+')';
  end;

function TExtFtp.GetFeatureString : string;
begin
  Result:=Capabilities.CommaText;
  end;

function TExtFtp.GetServerName : string;
begin
  if fTls then Result:='ftps' else Result:='ftp';
  if UseIPv6 then Result:=Result+'://['+Host+']:'+IntToStr(Port)
  else Result:=Result+'://'+Host+':'+IntToStr(Port);
  end;

function TExtFtp.GetLogName : string;
begin
  Result:=fIdLogFile.FileName;
  end;

procedure TExtFtp.SetLogName (Value : string);
begin
  fIdLogFile.FileName:=Value;
  end;

procedure TExtFtp.SetWriteLog (Value : boolean);
begin
  fIdLogFile.Active:=Value;
  end;

function TExtFtp.GetEncoding : TFtpEncoding;
begin
  with IOHandler do begin
    if DefStringEncoding=IndyTextEncoding_UTF8 then Result:=feUtf8
    else if DefStringEncoding=IndyTextEncoding_8Bit then Result:=fe8Bit
    else Result:=feASCII;
    end;
  FEncoding:=Result;
  end;

procedure TExtFtp.SetEncoding (Value : TFtpEncoding);
begin
  if Value<>FEncoding then ForceEncoding(Value,fUseOpts);
  end;

function TExtFtp.GetIPv6 : boolean;
begin
  Result:=IPVersion=Id_IPv6;
  fUseIPv6:=Result;
  end;

procedure TExtFtp.SetIPv6 (Value : boolean);
begin
  if Value then IPVersion:=Id_IPv6 else IPVersion:=Id_IPv4;
  fUseIPv6:=Value;
  end;

procedure TExtFtp.SetParams (const FtpPar : TFtpParams);
begin
  SecureTransfer:=FtpPar.SecureMode;
  Host:=FtpPar.Host;
  Port:=FtpPar.Port;
  Passive:=FtpPar.Passive;
  ForceUtf8:=FtpPar.ForceUtf8;
  UseKeepAlive:=FtpPar.KeepAlive<>0;
  UseIPv6:=FtpPar.UseIPv6;
  UseOpts:=FtpPar.UseOpts;
  UseExts:=FtpPar.UseExts;
  UseHost:=FtpPar.UseHost;
  Username:=FtpPar.Username;
  VerifyPeerMode:=FtpPar.CertVerify;
  PeerFingerprint:=FtpPar.Fingerprint;
  WriteLogFile:=FtpPar.WriteLog;
  CaseMode:=FtpPar.CaseMode;
  UseQuotes:=FtpPar.Quotes;
  ReplaceChars:=FtpPar.InvalidChars;
  if FtpPar.UseTimeOffset then TimeOffset:=FtpPar.TimeOffset
  else TimeOffset:=0;
  ConnectTimeOut:=FtpPar.ConnTimeout;
  TransferTimeout:=defTransferTimeOut;
  ReadTimeout:=FtpPar.ResponseTimeOut;
  ListenTimeout:=FtpPar.ResponseTimeOut;
  end;

procedure TExtFtp.ForceEncoding (Value : TFtpEncoding; UseOpts : boolean);
begin
  if Value=feUtf8 then begin
    if UseOpts then begin         // deprecated
      if not (SendCmd('OPTS UTF8 ON') in [200, 220]) then
      // trying draft-ietf-ftpext-utf-8-option-00.txt next...
        SendCmd('OPTS UTF-8 NLST');
      end;
    IOHandler.DefStringEncoding:=IndyTextEncoding_UTF8;
    end
  else begin
    if UseOpts then SendCmd('OPTS UTF8 OFF');
    if Value=fe8Bit then IOHandler.DefStringEncoding:=IndyTextEncoding_8Bit
    else IOHandler.DefStringEncoding:=IndyTextEncoding_ASCII;
    end;
  fEncoding:=Value;
  end;

procedure TExtFtp.SetTLSMode(Value : integer);
begin
  if fTLS and (Value>0) then begin
    if Value=1 then UseTLS:=utUseExplicitTLS
    else if Value=2 then UseTLS:=utUseRequireTLS
    else UseTLS:=utUseImplicitTLS;
    DataPortProtection:=ftpdpsPrivate;
    end
  else begin
    UseTLS:=utNoTLSSupport;
    DataPortProtection:=ftpdpsClear;
    end;
  end;

procedure TExtFtp.GetCertPassword(var Password : String);
begin
  if Assigned(fGetSSLPassword) then fGetSSLPassword(Password)  // prompt for password
  else Password:=fCertPwd;
  end;

function TExtFtp.VerifyPeer (Certificate: TIdX509; AOk: Boolean; ADepth, AError: Integer): Boolean;

  function HexStringToNumber (sn : string) : integer;
  begin
    sn:='$'+DelChar(sn,':');
    if not TryStrToInt(sn,Result) then Result:=0;
    end;

begin
// possible error codes:
//   X509_V_ERR_CERT_HAS_EXPIRED              = 10
//   X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT   = 18
// returning false to this event will raise a connect exception in TIdSSLSocket.Connect
// after which the connection cannot be completely closed
// save Certificate info for user defined handling
  with Certificate do begin
    fFingerPrint:=FingerprintAsString;    // MD5
    fCertInfo.Server:=GetServerName;
    fCertInfo.Version:=Version;
    fCertInfo.SerialNumber:=HexStringToNumber(SerialNumber);
    fCertInfo.Issuer:=Issuer.OneLine;
    fCertInfo.Fingerprint:=fFingerPrint;
    fCertInfo.ValidUntil:=notAfter;
//      rsCertVersion+ColSpace+IntToStr(Version)+sLineBreak
//      +rsSerialNr+ColSpace+SerialNumber+sLineBreak
//      +rsIssuer+ColSpace+Issuer.OneLine+sLineBreak
//      +rsValidUntil+ColSpace+DateTimeToStr(notAfter)+sLineBreak
//      +rsFingerprint+ColSpace+fFingerPrint;
    end;
//  if Assigned(fVerifyPeer) then begin
//    Result:=AOk or fVerifyPeer(Certificate,AOk,ADepth,AError);
//    end
//  else
  Result:=true;
  end;

function TExtFtp.ModifyName(const FName : string) : string;
begin
  // replace invalid characters and change case
  Result:=TextChangeCase(SpecCharToValue(FName,fRepl),CMode);
  end;

function TExtFtp.RetrieveName(const FName : string) : string;
begin
  if length(fRepl)=0 then Result:=FName else Result:=ValueToSpecChar(FName);
  end;

// Die Konvertierung auf UTF8 wird in den Indy-Routinen vorgenommen, dazu musste
// in IdTCPConnection berall en7Bit durch enDefault ersetzt werden
function TExtFtp.MakeQuotedPath (const s : string) : String;
begin
//  if fUTF8 then Result:=UTF8Encode(s) else
  if fQuotes and (pos(' ',s)>0) then Result:=AnsiQuotedStr(s,StringUtils.Quote)
  else Result:=s;
  end;

function TExtFtp.GetIdError : string;
begin
  if length(fIdError)>0 then Result:=rsIndyError+fIdError
  else Result:=rsFtpResponse+LastResponseString;
  fIdError:='';
  end;

procedure TExtFtp.SetIdError (const Value : string);
begin
  if IsEmptyStr(fIdError) then
    fIdError:=StringsReplace(Trim(Value),LOldStr,LNewStr);
  end;

procedure TExtFtp.LogReconnectError (const ErrMsg : string);
begin
  fIdLogFile.LogError(rsTryReconnect+' - '+ErrMsg);
  end;

function TExtFtp.GetDirectoryListing : boolean;
var
  abort : boolean;
begin
  if ValidDirectoryListing then Result:=true
  else begin
    Result:=false; Abort:=false;
    repeat
      if Connected then begin
        try
          List (nil);
          Result:=true;
        except
          on E:EIdReadTimeout do SetIdError(E.Message);
          on E:EIdSocketError do SetIdError(E.Message);
          on E:EIdConnClosedGracefully do SetIdError(E.Message);
          on E:EIdException do begin
            SetIdError(E.Message); Abort:=true;
            end;
          else begin
            SetIdError(RSICMPConvUnknownUnspecError); Abort:=true;
            end;
          end;
        end
      else SetIdError(rsNoConnect);
      if not Result and not Abort then begin  // try reconnect and repeat
        LogReconnectError(fIdError);
        sleep(ReconnDelay);
        Abort:=not ReConnect(fCurrentDir,false,false);
        end;
      until Result or Abort;
    ValidDirectoryListing:=Result;
    end;
  end;

function TExtFtp.GetFileList (FileList : TFileList) : boolean;
var
  i : integer;
begin
  if assigned(FileList) then begin
    Result:=GetDirectoryListing;
    if Result then with DirectoryListing do begin
      // Dateien
      FileList.Clear;
      for i:=0 to Count-1 do with Items[i] do if ItemType<>ditDirectory then
        FileList.AddObject(RetrieveName(Filename),
            TFtpFileInfo.Create(Filename,Size,ModifiedDate,ModifiedDateGMT));
      end;
    end
  else Result:=false;
  end;

function TExtFtp.GetFileList : boolean;
begin
  Result:=GetFileList(FList);
  end;

// die Objekte mssen von der aufrufenden Funktion entfernt werden
function TExtFtp.GetDirList (DirList : TDirectoryList) : boolean;
var
  i   : integer;
begin
  if assigned(DirList) then begin
    Result:=GetDirectoryListing;
    if Result then with DirectoryListing do begin
      // Verzeichnisse
      DirList.Clear;
      for i:=0 to Count-1 do with Items[i] do
        if (ItemType=ditDirectory) and NotSpecialDir(Filename) then
          DirList.AddObject(RetrieveName(Filename),
            TFtpFileInfo.Create(Filename,Size,ModifiedDate,ModifiedDateGMT));
      end;
    end
  else Result:=false;
  end;

function TExtFtp.FtpDirExists (const DName : string) : boolean;
var
  dl : TDirectoryList;
begin
  Result:=false;
  dl:=TDirectoryList.Create;
  if GetDirList(dl) then Result:=dl.IndexOf(ModifyName(DName))>=0;
  dl.Free;
  end;

function TExtFtp.FtpFileExists (const FName : string) : boolean;
begin
  Result:=false;
  if GetFileList then Result:=FList.IndexOf(ModifyName(FName))>=0;
  end;

// Delete file in subfolder
// Result = 0 : failed
//        = 1 : ok
//        = 2 : not found
function TExtFtp.DeleteSubDirFile (const ADir,FName : string) : integer;
var
  fl : TFileList;
begin
  Result:=0;
  try
    FtpChangeDir(ADir);
    fl:=TFileList.Create;
    if GetFileList(fl) then begin
      if fl.IndexOf(ModifyName(FName))>=0 then begin
        Delete(MakeQuotedPath(ModifyName(FName)));
        Result:=1;
        end
      else Result:=2;
      end;
    fl.Free;
    ChangeOneDirUp;
  except
    end;
  end;

function TExtFtp.DeleteFile (const FName : string; InitListing : boolean) : boolean;
var
  abort : boolean;
begin
  Result:=false; Abort:=false;
  if InitListing then ValidDirectoryListing:=false;
  repeat
    try
      Delete(MakeQuotedPath(ModifyName(FName)));
      Result:=true;
    except
      on E:EIdSocketError do SetIdError(E.Message);
      on E:EIdConnClosedGracefully do SetIdError(E.Message);
      on E:EIdException do begin
        SetIdError(E.Message); Abort:=true;
        end;
      else Abort:=true;
      end;
    if not Result and not Abort then begin  // try reconnect and repeat
      LogReconnectError(fIdError);
      sleep(ReconnDelay);
      Abort:=not ReConnect(fCurrentDir,false);
      end;
    until Result or Abort;
  end;

// delete existing file
// Result = -1 : error
//        = 0  : not found
//        = 1  : existing file was deleted
function TExtFtp.DeleteExistingFile (const FName : string) : integer;
begin
  if FtpFileExists(FName) then begin
    if DeleteFile(FName) then Result:=1 else Result:=-1;
    end
  else Result:=0;
  end;

// delete all files matching to mask
// Result = number of matching files
function TExtFtp.DeleteMatchingFiles (const AMask : string) : integer;
var
  i : integer;
begin
  Result:=0;
  with FList do for i:=0 to Count-1 do if MatchesMask(Strings[i],AMask) then begin
    DeleteFile(MakeQuotedPath(Strings[i]));
    inc(Result);
    end;
  end;

function TExtFtp.RenameFile (const OldName,NewName : string; InitListing : boolean) : boolean;
var
  abort : boolean;
begin
  Result:=false; Abort:=false;
  if InitListing then ValidDirectoryListing:=false;
  repeat
    try
      Rename(MakeQuotedPath(ModifyName(OldName)),MakeQuotedPath(ModifyName(NewName)));
      Result:=true;
    except
      on E:EIdSocketError do SetIdError(E.Message);
      on E:EIdConnClosedGracefully do SetIdError(E.Message);
      on E:EIdException do begin
        SetIdError(E.Message); Abort:=true;
        end;
      else Abort:=true;
      end;
    if not Result and not Abort then begin  // try reconnect and repeat
      LogReconnectError(fIdError);
      sleep(ReconnDelay);
      Abort:=not ReConnect(fCurrentDir,false);
      if not Abort then Result:=FtpFileExists(ModifyName(NewName));
      end;
    until Result or Abort;
  end;

function TExtFtp.ReadCurrentDir (var Dir : string) : boolean;
var
  n  : integer;
begin
  Result:=false; n:=2;
  repeat
    try
      Dir:=RetrieveName(RetrieveCurrentDir);
      Result:=true;
    except
      on E:EIdSocketError do SetIdError(E.Message);
      on E:EIdConnClosedGracefully do SetIdError(E.Message);
      on E:EIdException do begin
        SetIdError(E.Message);
        n:=1;
        end;
      else n:=1;
      end;
    dec(n);
    if not Result then sleep(ReconnDelay); // retry
    until Result or (n=0);
  end;

function TExtFtp.GetCurrentDir : string;
var
  ok : boolean;
begin
  ok:=ReadCurrentDir(Result);
  if not ok then begin
    LogReconnectError(fIdError);
    sleep(ReconnDelay);
    if ReConnect(fCurrentDir,false) then ReadCurrentDir(Result)
    else Result:='';
    end;
  end;

// erfordert vorheriges GetFileList fr das aktuelle Verzeichnis
function TExtFtp.GetSize (const FName : string) : int64;
begin
  Result:=GetSize (FList,FName);
  end;

function TExtFtp.GetSize (FileList : TFileList; const FName : string) : int64;
var
  n : integer;
begin
  Result:=0;
  if not assigned(FileList) then Exit;
  n:=FileList.IndexOf(ModifyName(FName));
  if n>=0 then Result:=(FileList.Objects[n] as TFtpFileInfo).FSize;
  end;

function TExtFtp.CheckSize(const FName : string; ASize : int64) : boolean;
begin
  Result:=GetSize(FName)<>ASize;
  end;

// Replacement for "FileDate"
// Uses the same dependancies on features as SetModTime
function TExtFtp.GetModTime(const AFileName: String; AsGMT: Boolean): TDateTime;
var
  LBuf : String;
begin
  if SendCmd('MDTM ' + AFileName) = 213 then begin {do not localize}
    LBuf := LastCmdResult.Text[0];
    LBuf := Trim(LBuf);
    if AsGMT then Result := FTPMLSToGMTDateTime(LBuf)
    else begin
      if IsExtSupported('MFMT') then Result:=FTPMLSToLocalDateTime(LBuf)
      //syntax 1 + 2
      else if (IndexOfFeatLine('MDTM YYYYMMDDHHMMSS[+-TZ];filename') > 0) or
         (IndexOfFeatLine('MDTM YYYYMMDDHHMMSS[+-TZ] filename') > 0) or IsOldServU or IsBPFTP then
        Result := FTPMLSToLocalDateTime(LBuf)
      //syntax 3 - MDTM [local timestamp] Filename
      else if TZInfo.GMTOffsetAvailable then Result:=FTPMLSToGMTDateTime(LBuf)-TZInfo.GMTOffset
      else Result:=FTPMLSToGMTDateTime(LBuf);
      end;
    end
  else Result := 0;
  end;

// erfordert vorheriges GetFileList fr das aktuelle Verzeichnis
// Wenn MLST untersttzt wird kommt der Zeitstempel von dort,
// andernfalls wird nicht mehr FileDate, sondern GetModTime (s.o.) verwendet
function TExtFtp.GetTimeStamp (const FName : string; AsGMT : Boolean = false) : TDateTime;
begin
  Result:=GetTimeStamp (FList,FName,AsGMT);
  end;

function TExtFtp.GetTimeStamp (FileList : TFileList; const FName : string; AsGMT : Boolean = false) : TDateTime;
var
  n : integer;
begin
  Result:=0;
  if not assigned(FileList) then Exit;
  n:=FileList.IndexOf(ModifyName(FName));
  if n>=0 then begin
    if AsGMT then Result:=(FileList.Objects[n] as TFtpFileInfo).FModifiedDateGMT
    else Result:=(FileList.Objects[n] as TFtpFileInfo).FModifiedDate;
    if not ((FCapabilities.Count>0) and FCanUseMLS) or (Result<1) then begin
      // get local time from server because LIST may return time stamps
      // with only date information (time = 0)
      Result:=GetModTime(MakeQuotedPath(ModifyName(FName)),AsGMT);  // change to local time
      end
    end
  end;

// Compare timestamps as local or UTC time, Delta in s
// fdt = Timestamp of file FName
// Result = 0 : FName not found
//          1 : timestamp of FName < tm - Delta
//          2 : tm - Delta <= timestamp of FName <= tm + Delta
//          3 : tm + Delta < timestamp of FName
function TExtFtp.CheckTimeStamp (const FName : string; tm : TDateTime; Delta : integer;
                             var fdt : TDateTime; AsGMT : Boolean = false) : integer;
begin
  fdt:=GetTimeStamp(FName,AsGMT);
  if fdt>1 then begin
    if fdt>=IncSecond(tm,-Delta) then begin
      if fdt<=IncSecond(tm,Delta) then Result:=2 else Result:=3
      end
    else Result:=1;
    end
  else Result:=0;
  end;

function TExtFtp.CheckTimeStamp (const FName : string; tm : TDateTime;
                                 Delta : integer; AsGMT : Boolean = false) : integer;
var
  dt : TDateTime;
begin
  Result:=CheckTimeStamp(FName,tm,Delta,dt,AsGMT);
  end;

function TExtFtp.GetLocalFileTime (FileList : TFileList; const FName : string) : TFileTime;
begin
  Result:=DateTimeToFileTime(GetTimeStamp(FileList,FName,false));
  end;

function TExtFtp.GetLocalFileTime (const FName : string) : TFileTime;
begin
  Result:=GetLocalFileTime(FList,FName);
  end;

function TExtFtp.SetTimeStamp(const FName : string; ATime : TDateTime; UseGMT : Boolean = false) : boolean;
begin
  Result:=true;
  if (FCapabilities.Count>0) and FCanUseMLS then begin
    try
      // filetime from source is GMT
      if UseGMT then SetModTimeGMT(MakeQuotedPath(ModifyName(FName)),ATime)
      else SetModTime(MakeQuotedPath(ModifyName(FName)),ATime+fTimeOffset/MinsPerDay); // apply optional offset
    except
      Result:=false
      end;
    end
  else begin
    try
      SetModTime(MakeQuotedPath(ModifyName(FName)),ATime+fTimeOffset/MinsPerDay);  // apply optional offset
    except
      Result:=false
      end;
    end;
  end;

function TExtFtp.SetTimeStampFromFile(const ASourceFile,FName : string;
                     CheckTime : boolean = true) : boolean;
var
  dt : TDateTime;
begin
  dt:=FileAgeXL(ASourceFile,0);
  if CheckTime and (dt>Now+1) then dt:=Now;
  Result:=SetTimeStamp(FName,dt);
  end;

procedure TExtFtp.FtpChangeDir (DName: string);
begin
  ValidDirectoryListing:=false;
  DName:=ModifyName(DName);
  if AnsiStartsText(fDir,DName) then    // no case change for fDir
    DName:=fDir+copy(DName,length(fDir)+1,length(DName));
  ChangeDir(MakeQuotedPath(DName));
  end;

// ChangeDirUp with boolean result
function TExtFtp.ChangeOneDirUp : boolean;
begin
  Result:=true;
  ValidDirectoryListing:=false;
  try ChangeDirUp; except Result:=false; end;
  with LastCmdResult do FChangeDirResponse:=Code+' '+Text[0];
  fCurrentDir:=GetCurrentDir;
  end;

// ChangeDir with boolean result
function TExtFtp.ChangeToDir(const DName: string) : boolean;
var
  sd : string;
  n  : integer;

  function RenewConnection : boolean;
  begin
    LogReconnectError(fIdError);
    sleep(ReconnDelay);
    Result:=DoConnect(1);
    if Result then begin
      if not IsAbsolutePath(DName) then begin
        try FtpChangeDir(fCurrentDir); except end;
        end;
      try FtpChangeDir(DName); except end;
      end;
    end;

begin
  Result:=true;
  if length(DName)=0 then Exit;
  try
    FtpChangeDir(DName);
  except
    on E:EIdSocketError do begin
      SetIdError(E.Message);
      Result:=RenewConnection;
      end;
    on E:EIdConnClosedGracefully do begin
      SetIdError(E.Message);
      Result:=RenewConnection;
      end;
    on E:EIdException do begin
      SetIdError(E.Message);
      Result:=false;
      end;
    end;
  if Result then with LastCmdResult do if (NumericCode=250) or (NumericCode=200) then begin
    sd:=Text[0];
    n:=pos(StringUtils.Quote,sd);
    if (n>0) then begin
      System.delete(sd,1,n-1);
      sd:=ReadNxtQuotedStr(sd,Space,StringUtils.Quote);
      end
    else begin
      n:=pos('/',sd);
      if n>0 then System.delete(sd,1,n-1) else sd:='';
      end;
    sd:=Trim(sd);
    if length(sd)>0 then fCurrentDir:=sd // use if available
    else if IsAbsolutePath(DName) then fCurrentDir:=DName
    else Result:=ReadCurrentDir(fCurrentDir);
    end
  else fIdLogFile.LogError(rsIndyError+' - '+fIdError);
  with LastCmdResult do FChangeDirResponse:=Code+' '+Text[0];
//  fCurrentDir:=GetCurrentDir;
  end;

// Change to dir and create if not exists
// return : -1 = error, 0 = ok exists, 1 = ok is new
function TExtFtp.ChangeToNewDir(const DName: string) : integer;
var
  s : string;
  n : integer;
begin
  Result:=0;
  if length(DName)=0 then Exit;
  if ChangeToDir(DName) then Result:=0
  else begin
    Result:=-1;
//  try
//    FtpChangeDir(DName);
//    with LastCmdResult do FChangeDirResponse:=Code+' '+Text[0];
//    Result:=0
//  except
    try
      MakeDir(MakeQuotedPath(ModifyName(DName)));
    except
      end;
    Result:=1; fIdError:='';
    with LastCmdResult do begin
      FChangeDirResponse:=Code+' '+Text[0];
      // Accept "250" as legal response (Some FTP servers need this)
      // Accept "200" as legal response (DSL-EasyBox 803 FTP Server)
      if (NumericCode=257) or (NumericCode=250) or (NumericCode=200) then begin
        s:=Text[0];
        n:=LastDelimiter('"',s);
        if n>0 then begin
          s:=AnsiDequotedStr(copy(s,1,n),'"');
          if Pos(#$fffd,s)>0 then s:=DName       // wird bei proftp zurckgegeben
          else if not AnsiStartsText(fCurrentDir,s) then s:=DName; // bei ProFTPD
          end
        else s:=DName;
        end
      else begin
        s:=DName;
        fIdError:=Code+' '+Text[0];
        end;
      end;
    if not ChangeToDir(s) then Result:=-1;
    end;
//  fCurrentDir:=GetCurrentDir;
  end;

// Create subdirectory(ies)
// Result = true if DName exists or successfully created
// count relative level changes (DCount) for use in "ForceDirUp"
function TExtFtp.ForceDir(const DName: string) : boolean;
var
  ok,root : boolean;
  d,sd  : string;
  n  : integer;
begin
  Result:=true; root:=false;
  DCount:=0;
  if length(DName)>0 then begin
    sd:=ModifyName(DName);
    ok:=true;
    try
      FtpChangeDir(sd);
      DCount:=CountSubStr('/',sd)+1;
    except
      ok:=false;
      end;
    if not ok then begin
      if (length(fDir)>1) and AnsiStartsText(fDir,sd) then begin
        try
          ChangeDir (MakeQuotedPath(fDir));
          if length(fDir)=1 then n:=1 else n:=2;
          sd:=copy(sd,length(fDir)+n,length(sd));
          ok:=true; root:=true;
        except
          end;
        end
      else begin
        d:=ReadNxtStr(sd,'/');
        root:=length(d)=0;
        if root then d:='/'+ReadNxtStr(sd,'/');  // add leading '/'
        ok:=ChangeToNewDir(d)>=0;
        if ok then inc(DCount);
        end;
      while ok and (length(sd)>0) do begin
        d:=ReadNxtStr(sd,'/');
        if ChangeToNewDir(d)>=0 then inc(DCount)
        else ok:=false;
        end;
      if root then DCount:=0; // was abs. path
      Result:=ok;
      end
    end;
  fCurrentDir:=GetCurrentDir;
  end;

// Go up all subdirectories create by last "ForceDir"
function TExtFtp.ForceDirUp: boolean;
var
  i : integer;
begin
  Result:=false;
  ValidDirectoryListing:=false;
  for i:=1 to DCount do begin
    try ChangeDirUp; except exit; end;
    end;
  DCount:=0;
  fCurrentDir:=GetCurrentDir;
  Result:=true;
  end;

function TExtFtp.SetRemoteDir (const Dir : string) : boolean;
begin
  Result:=ForceDir(Dir);
  ValidDirectoryListing:=false;
//  fDir:=fCurrentDir;
  end;

function TExtFtp.DeleteDir (DName : string) : boolean;
begin
  Result:=true;
  ValidDirectoryListing:=false;
  try
    RemoveDir(MakeQuotedPath(ModifyName(DName)));
  except
    Result:=false;
    end;
  end;

// delete empty directories in a tree
function TExtFtp.DeleteEmptyDirs (const DName : string; DelRoot : boolean;
                                  var DCount : integer; WasCanceled : TStatusEvent) : boolean;
var
  i,n   : integer;
  DList : TFileList;
  sc    : string;
begin
  Result:=false;
  ValidDirectoryListing:=false;
  sc:=GetCurrentDir;
  if length(DName)>0 then begin
    try
      FtpChangeDir(DName);
    except
      FtpChangeDir(sc);
      Exit;
      end;
    end;
  List (nil);
  DList:=TFileList.Create;
  Application.ProcessMessages;
  with DirectoryListing do for i:=0 to Count-1 do with Items[i] do
    if (ItemType=ditDirectory) and NotSpecialDir(Filename) then DList.Add(RetrieveName(Filename));
  i:=0;
  Result:=true;
  with DList do while (i<Count) and Result do begin
    Result:=DeleteEmptyDirs(Strings[i],true,DCount,WasCanceled);
    inc(i);
    end;
  if assigned(WasCanceled) then Result:=not WasCanceled;
  Application.ProcessMessages;
  if (length(DName)>0) then begin
    n:=0;
    if Result then begin
      List (nil);
      with DirectoryListing do for i:=0 to Count-1 do with Items[i] do
        if NotSpecialDir(Filename) then inc(n);
        end;
    try ChangeDirUp; except; Result:=false; end;
    if Result and (n=0) and DelRoot then begin
      Result:=DeleteDir(DName);
      if Result then inc(DCount);
      end;
    end;
  DList.Free;
  GetFilelist;
  end;

// delete tree including all files
function TExtFtp.DeleteTree (const DName : string; DelRoot : boolean;
                             var FCount : integer) : boolean;
var
  i     : integer;
  DList : TFileList;
  sc    : string;
begin
  Result:=false;
  ValidDirectoryListing:=false;
  sc:=GetCurrentDir;
  if length(DName)>0 then begin
    try
      FtpChangeDir(DName);
    except
//    if not ChangeToDir(DName) then begin
      FtpChangeDir(sc);
      Exit;
      end;
    end;
  List (nil);
  // Delete files
  with DirectoryListing do for i:=0 to Count-1 do with Items[i] do
      if (ItemType<>ditDirectory) then begin
    try
      self.Delete(MakeQuotedPath(Filename));
      inc(FCount);
    except
      end;
    Application.ProcessMessages;
    end;
  DList:=TFileList.Create;
  with DirectoryListing do for i:=0 to Count-1 do with Items[i] do
    if (ItemType=ditDirectory) and NotSpecialDir(Filename) then DList.Add(RetrieveName(Filename));
  i:=0;
  Result:=true;
  with DList do while (i<Count) and Result do begin
    Result:=DeleteTree(Strings[i],true,FCount);
    inc(i);
    end;
  if length(DName)>0 then begin
    try ChangeDirUp; except; Result:=false; end;
    if Result and DelRoot then begin
      try Result:=DeleteDir(DName); except; end;
      end;
    end;
  DList.Free;
  end;

// Count files in subdirectory
function TExtFtp.DirFiles(DName : string) : integer;
var
  DList : TDirectoryList;
  i     : integer;
begin
  DName:=ModifyName(DName);
  Result:=0;
  if length(DName)>0 then begin
    try FtpChangeDir(DName); except; Exit; end;
    end;
  if GetDirectoryListing then begin
    // Count files
    with DirectoryListing do for i:=0 to Count-1 do with Items[i] do
      if (ItemType<>ditDirectory) then inc(Result);
    DList:=TDirectoryList.Create;
    if GetDirList(DList) then  // subdirectories
      with DList do for i:=0 to Count-1 do Result:=Result+DirFiles(Strings[i]);
    if length(DName)>0 then begin
      try ChangeOneDirUp; except; end;
      end;
    DList.Free;
    end;
  end;

// Connect to FTP,
// if server does not support SSL on utUseExplicitTLS reset DataPortProtection
function TExtFtp.DoConnect (NRepeat : integer; Verify : boolean) : boolean;
begin
  fIdError:='';
  fIdLogFile.LogStatus(TryFormat(rsConnectTo,[Host,Port]));
  ValidDirectoryListing:=false;
  InitIOHandler;
  repeat
    try
      Connect;
      Result:=true;
      if (UseTLS=utUseExplicitTLS) and not FUsingSFTP then begin // TLS not available
        DataPortProtection:=ftpdpsClear;
        end;
    except
      on E:EIdException do begin
        SetIdError(E.Message); Result:=false;
        end;
      end;
    dec(NRepeat);
    if not Result and (NRepeat>0) then begin
      Sleep(ReconnDelay);
      try Disconnect except end;
      end;
    until Result or (NRepeat<=0);
  if Result then begin
    if Verify and (fVerifyPeerMode>0) then begin
      if fVerifyPeerMode=2 then Result:=AnsiSameText(fFingerPrint,fPeerFingerprint)
      else begin
        if Assigned(fConfirmCert) then Result:=fConfirmCert(fCertInfo)
        else Result:=false;
        end;
      if not Result then begin
        try Disconnect except end;
        fIdError:=rsVerCertError;
        end;
      end;
    if fForceUtf8 then ForceEncoding(feUtf8,fUseOpts)
    else if fUseOpts and CanUseUtf8 then SetCmdOpt('UTF8','ON');
    if fUseExts and UseExtensionDataPort then FUsingExtDataPort:=true; // overwrite FEAT request
    Result:=ReadCurrentDir(fDir);
    end
  else fIdLogFile.LogError(rsIndyError+fIdError);
  end;

// Check connection and try to reconnect if connection was canceled
function TExtFtp.ReConnect(const Dir : string; Check,UpdateList : boolean) : boolean;
begin
  if Check and Connected then Result:=true
  else begin
//    try
//      sd:=RetrieveCurrentDir;  // check if still connected
//      if not AnsiSameText(sd,ModifyName(Dir)) then ChangeToDir(Dir);
//      Exit;
//    except
    // ClosedGracefully
    try Disconnect except end;
//      end;
    Result:=DoConnect(3);
    if Result then begin
      ChangeToDir(Dir);
      if UpdateList then GetFileList;
      end;
    end;
  end;

// Replaces original function to check if peer has closed connection
function TExtFtp.Connected : boolean;
begin
  try
    Result:=inherited Connected;
  except
    Result:=false;
    end;
  end;

procedure TExtFtp.AbortTransfer;
begin
  fAborted:=true;
//  try Abort; except GetInternalResponse end;
  end;

// Copy file
function TExtFtp.CopyFile (const ASourceFile,ADestFile : string) : integer;
var
  fs : TFileStream;
begin
  Result:=errOK;
  if not FileExistsXL(ASourceFile) then Result:=errError+errFileOpen;
  if (Result=errOK) and (length(ADestFile)=0) then Result:=errError+errFileCreate;
  if Result=errOK then begin
    try
      fs:=TFileStream.Create(FilenameToXL(ASourceFile),fmOpenRead+fmShareDenyNone);
    except
      Result:=errError+errFileOpen;
      end;
    end;
  if Result=errOK then Result:=Flush(fs,ADestFile,false);
  end;

// Write stream
function TExtFtp.Flush(ASource: TStream; const ADestFile: string; AAppend: boolean) : integer;
var
  abort : boolean;
begin
  if not ReConnect(fCurrentDir) then begin
    Result:=errError+errFtpConnect; exit;
    end;
  Abort:=false; fAborted:=false; NATKeepAlive.UseKeepAlive:=fUseKeepAlive;
  repeat
    try
      Put(ASource,MakeQuotedPath(ModifyName(ADestFile)),AAppend);
      if LastCmdResult.NumericCode<400 then Result:=errOK    // siehe RFC 640
      else Result:=errError+errFtpWrite;
    except
      on E:EIdReadTimeout do begin
        SetIdError(E.Message);
        Result:=errError+errFtpTimeout;
        end;
      on E:EIdSocketError do begin
        SetIdError(E.Message);
        Result:=errError+errFtpBroken;
        end;
      on E:EIdConnClosedGracefully do begin
        SetIdError(E.Message);
        Result:=errError+errFtpBroken;
        end;
      on E:EIdException do begin
        SetIdError(E.Message); Abort:=true;
        if LastCmdResult.NumericCode=425 then Result:=errError+errFtpDatConn
        else Result:=errError+errFtpWrite;
        end;
      else begin
        Abort:=true;
        Result:=errError+errFtpWrite;
        end;
      end;
    Abort:=Abort or fAborted;
    if (Result<>errOK) and not Abort then begin  // try reconnect and repeat transfer
      LogReconnectError(fIdError);
      sleep(ReconnDelay);
      if not ReConnect(fCurrentDir,false) then begin
        Result:=errError+errFtpConnect; Abort:=true;
        end;
      end;
    until (Result=errOK) or Abort;
  NATKeepAlive.UseKeepAlive:=false;
  if (Result=errOK) and not ReConnect(fCurrentDir) then Result:=errError+errFtpBroken;
  end;

// Stream lesen
function TExtFtp.Fill(const ASourceFile : string; ADest : TStream) : integer;
var
  abort : boolean;
begin
  if not ReConnect(fCurrentDir) then begin
    Result:=errError+errFtpConnect; exit;
    end;
  Abort:=false; fAborted:=false; NATKeepAlive.UseKeepAlive:=fUseKeepAlive;
  repeat
    try
      Get(MakeQuotedPath(ModifyName(ASourceFile)),ADest,false);
      ADest.Seek(0,soFromBeginning);
      Result:=errOK;
    except
      on E:EIdReadTimeout do begin
        SetIdError(E.Message);
        Result:=errError+errFtpTimeout;
        end;
      on E:EIdSocketError do begin
        SetIdError(E.Message);
        Result:=errError+errFtpBroken;
        end;
      on E:EIdConnClosedGracefully do begin
        SetIdError(E.Message);
        Result:=errError+errFtpBroken;
        end;
      on E:EIdException do begin
        SetIdError(E.Message); Abort:=true;
        Result:=errError+errFtpRead;
        end;
      else begin
        Abort:=true;
        Result:=errError+errFtpRead;
        end;
      end;
    NATKeepAlive.UseKeepAlive:=false;
    Abort:=Abort or fAborted;
    if (Result<>errOK) and not Abort then begin  // try reconnect and repeat transfer
      LogReconnectError(fIdError);
      sleep(ReconnDelay);
      if not ReConnect(fCurrentDir,false) then begin
        Abort:=true; Result:=errError+errFtpConnect;
        end;
      end;
    until (Result=errOK) or Abort;
  if (Result=errOK) and not ReConnect(fCurrentDir) then Result:=errError+errFtpBroken;
  end;

{ ------------------------------------------------------------------- }
constructor TFtpFileInfo.Create (const AFilename : string; ASize : int64; AMd,AMdG : TDateTime);
begin
  inherited Create;
  FFilename:=AFilename; FSize:=ASize;
  FModifiedDate:=AMd; FModifiedDateGMT:=AMdG;
  end;

{ ------------------------------------------------------------------- }
constructor TDirectoryList.Create;
begin
  inherited Create;
  CaseSensitive:=true;
  end;

destructor TDirectoryList.Destroy;
begin
  Clear;
  inherited Destroy;
  end;

procedure TDirectoryList.Clear;
var
  i : integer;
begin
  for i:=0 to Count-1 do if assigned(Objects[i]) then begin
    try Objects[i].Free; except end; Objects[i]:=nil;
    end;
  inherited Clear;
  end;

{ ------------------------------------------------------------------- }
constructor TFileList.Create;
begin
  inherited Create;
  CaseSensitive:=true;
  end;

destructor TFileList.Destroy;
begin
  Clear;
  inherited Destroy;
  end;

procedure TFileList.Clear;
var
  i : integer;
begin
  for i:=0 to Count-1 do if assigned(Objects[i]) then begin
    try Objects[i].Free; except end; Objects[i]:=nil;
    end;
  inherited Clear;
  end;

{ ------------------------------------------------------------------- }
function DefaultFtpParams : TFtpParams;
begin
  with Result do begin
    Host:='';
    Port:=IdPORT_FTP;
    Username:='';
    Password:='';
    Directory:='';
    Fingerprint:='';
    SecureMode:=fsNone;
    UseTimeOffset:=false;
    with TTimeZone.Local.UtcOffset do TimeOffset:=60*Hours+Minutes;
    ConnTimeout:=defTimeOut;
    ConnAttempts:=defConnect;
    ResponseTimeOut:=defTimeOut;
    CertVerify:=0;
    KeepAlive:=defFtpKeepAlive;
    KeepAliveCmd:=0;
    Passive:=true;
    ForceUtf8:=false;
    UseIPv6:=false;
    UseExts:=false;
    UseOpts:=false;
    UseHost:=false;
    WriteLog:=false;
    CaseMode:=tcNone;
    Quotes:=false;
    InvalidChars:='';
    with Proxy do begin
      Server:='';
      Username:='';
      Password:='';
      Port:=IdPORT_FTP_PROXY;
      Mode:=fpcmNone;
      end;
    end;
  end;

{ ------------------------------------------------------------------- }
(* Bei Verzeichnisangaben Slash ergnzen *)
function IncludeTrailingSlash (const Dir : string) : string;
begin
  if (length(Dir)=0) or (Dir[length(Dir)]='/') then Result:=Dir
  else Result:=Dir+'/';
  end;

function IncludeLeadingSlash (const Dir : string) : string;
begin
  if (length(Dir)=0) or (Dir[1]<>'/') then Result:='/'+Dir
  else Result:=Dir;
  end;

function RemoveLeadingSlash (const Dir : string) : string;
begin
  Result:=Dir;
  if (length(Result)>0) and (Result[1]='/') then Delete(Result,1,1);
  end;

function RemoveTrailingSlash (const Dir : string) : string;
begin
  Result:=Dir;
  if (length(Result)>0) and (Result[length(Result)]='/') then Delete(Result,length(Result),1);
  end;

function IsAbsolutePath (const Dir : string) : boolean;
begin
  Result:=(length(Dir)>0) and (Dir[1]='/');
  end;

function DosPathToUnixPath(const Path: string): string;
begin
  Result:=Path.Replace('\', '/');
  end;

// Den letzten Teilnamen extrahieren (siehe auch: ExtractFileName)
function ExtractLastName(const Delimiters,FileName: string): string;
var
  I: Integer;
begin
  I:=LastDelimiter(Delimiters, FileName);
  Result:=Copy(FileName,I+1,MaxInt);
  end;

function ExtractPath(const APath : string) : string;
begin
  Result:=IndyGetFilePath(APath);
  end;

// Unix-Pfad zusammensetzen
function ExpandUnixPath(const APath,AName : string) : string;
begin
  if length(APath)>0 then Result:=APath+'/'+AName
  else Result:=AName;
  end;

// Relativen Unix-Pfad extrahieren
function GetUnixRelativePath(const RootPath,Path : string) : string;
begin
  Result:=DOSPathToUnixPath(Path);
  if AnsiStartsText(DOSPathToUnixPath(RootPath),Result) then begin
    Delete(Result,1,length(RootPath));
    end
  else Result:='';
  end;

end.
