(* Delphi-Unit
   Compute hashes from file or stream

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   Acknowledgements:
     based on the routines from
     http://forum.vingrad.ru/forum/topic-230076.html

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   Vers. 1 - Jul. 2018, uses Syste.Hash
   *)

unit Hashes;

interface

uses System.SysUtils, System.Classes, System.Hash, CBFunctions;

type
  THashType = (htMD5,htSHA1,htSHA256);
  THashTypes = set of THashType;
  THashStrings = array [THashType] of string;

const
  defBufSize = 1024*1024*32;  // 32 MiB
  HashNames : array [THashType] of string = ('MD5','SHA1','SHA256');

function HashStream (ht : THashType; fs : TStream; CallBack : TProgressEvent = nil;
                     ABufSize : integer = defBufSize) : String;
function HashFromFile (ht : THashType; const InName : string; CallBack : TProgressEvent = nil;
                       ABufSize : integer = defBufSize) : string;
procedure HashBreak;


function HashToTextFile (ht : THashType; const InName,OutName,HashString : string) : boolean;
function HashesToTextFile (hts : THashTypes; const InName,OutName : string;
                           const HashStrings : THashStrings) : boolean;

implementation

var
  UserBreak : boolean;

function HashStream (ht : THashType; fs : TStream; CallBack : TProgressEvent = nil;
                     ABufSize : integer = defBufSize) : String;
var
  HashMd5       : THashMD5;
  HashSHA1      : THashSHA1;
  HashSHA256    : THashSHA2;
  Buffer        : array of byte;
  BufSize,BSize : integer;
begin
  case ht of
  htSHA1   : HashSHA1:=THashSHA1.Create;
  htSHA256 : HashSHA256:=THashSHA2.Create(SHA256);
  else HashMD5:=THashMD5.Create;
    end;
  BufSize:=ABufSize;
  SetLength(Buffer,BufSize);
  UserBreak:=false;
  try
    with fs do begin
      if assigned(CallBack) then CallBack(ptStart,Size);
      while (Position<Size) and not UserBreak do begin
        BSize:=Read(Buffer[0],BufSize);
        if assigned(CallBack) then begin
          CallBack(ptInc,BSize); CallBack(ptPos,Position);
          end;
        if BSize>0 then begin
          case ht of
          htSHA1   : HashSHA1.Update(Buffer[0],BSize);
          htSHA256 : HashSHA256.Update(Buffer[0],BSize);
          else HashMD5.Update(Buffer[0],BSize);
            end;
          end;
        end;
      if assigned(CallBack) then CallBack(ptEnd,Size);
      end;
  finally
    Buffer:=nil;
    end;
  if UserBreak then Result:=''
  else begin
    case ht of
    htSHA1   : Result:=HashSHA1.HashAsString;
    htSHA256 : Result:=HashSHA256.HashAsString;
    else Result:=HashMD5.HashAsString;
      end;
    end;
  end;

procedure HashBreak;
begin
  UserBreak:=true;
  end;

function HashFromFile (ht : THashType; const InName : string; CallBack : TProgressEvent = nil;
                       ABufSize : integer = defBufSize) : string;
var
  fs    : TFileStream;
begin
  Result:='';
  if not FileExists(InName) then Exit;
  try
    fs:=TFileStream.Create(InName,fmOpenRead);
    Result:=HashStream(ht,fs,CallBack,ABufSize);
  finally
    fs.Free;
    end;
  end;

function HashToTextFile (ht : THashType; const InName,OutName,HashString : string) : boolean;
var
  oText : TextFile;
begin
  Result:=false;
  if length(HashString)=0 then Exit;
  try
    AssignFile(oText,OutName); rewrite(oText);
    writeln (oText,ExtractFilename(InName));
    writeln (oText,HashNames[ht]:7,': ',HashString);
    Result:=true;
  finally
    CloseFile(oText);
    end;
  end;

function HashesToTextFile (hts : THashTypes; const InName,OutName : string;
                           const HashStrings : THashStrings) : boolean;
var
  ht : THashType;
  oText : TextFile;
begin
  Result:=false;
  if hts=[] then Exit;
  try
    AssignFile(oText,OutName); rewrite(oText);
    writeln (oText,ExtractFilename(InName));
    for ht:=low(THashType) to High(THashType) do if ht in hts then
      writeln (oText,HashNames[ht]:7,': ',HashStrings[ht]);
    Result:=true;
  finally
    CloseFile(oText);
    end;
  end;
end.
