(* Delphi Dialog
   Password query
   ==============

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.
    
   Vers. 1 - Dez. 2006
   last modified: July 2022
   *)

unit PwdDlg;

interface

uses Winapi.Windows, System.SysUtils, System.Classes, Vcl.Graphics, Vcl.Forms,
  Vcl.Controls, Vcl.StdCtrls, Vcl.Buttons, Vcl.ExtCtrls;

type
  TPasswordDialog = class(TForm)
    Label1: TLabel;
    edtPwd: TEdit;
    CancelBtn: TBitBtn;
    OKBtn: TBitBtn;
    btnSkip: TBitBtn;
    gbEncrypt: TGroupBox;
    rbEnc128: TRadioButton;
    rbEnc256: TRadioButton;
    rbEncAuto: TRadioButton;
    pnTop: TPanel;
    pnInput: TPanel;
    pnBottom: TPanel;
    bbPaste: TBitBtn;
    lblPrompt: TStaticText;
    procedure FormCreate(Sender: TObject);
    procedure bbPasteClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  private
    { Private-Deklarationen }
    Hgt : integer;
  public
    { Public-Deklarationen }
{$IFDEF HDPI}   // scale glyphs and images for High DPI
    procedure AfterConstruction; override;
{$EndIf}
    function Execute (APos : TPoint; const Title,Prompt : string; Skip,Enc  : boolean;
                      var Pwd : String; var EncMode  : integer) : TModalResult;  overload;
    function Execute (APos : TPoint; const Title,Prompt : string; Skip,Enc  : boolean;
                      var Pwd : AnsiString; var EncMode  : integer) : TModalResult;  overload;
    function Execute (APos : TPoint; const Title,Prompt : string; var Pwd : AnsiString) : TModalResult;  overload;
    function Execute (const Title,Prompt : string; var Pwd : AnsiString) : TModalResult;  overload;
  end;

// Result:  Skip = mrYes, Ok = mrOK, Cancel = mrCancel
function ReadPassword (APos : TPoint; const Title,Prompt : string; Skip,Enc : boolean;
                       var Pwd : String; var EncMode  : integer) : TModalResult; overload;

function ReadPassword (APos : TPoint; const Title,Prompt : string; Skip,Enc : boolean;
                       var Pwd : AnsiString; var EncMode  : integer) : TModalResult; overload;

function ReadPassword (APos : TPoint; const Title,Prompt : string; Skip : boolean;
                       var Pwd : AnsiString) : TModalResult; overload;

function ReadPassword (const Title,Prompt : string; Skip : boolean;
                       var Pwd : AnsiString) : TModalResult; overload;

function ReadPassword (APos : TPoint; const Title,Prompt : string;
                       var Pwd : String) : boolean; overload;

function ReadPassword (APos : TPoint; const Title,Prompt : string;
                       var Pwd : AnsiString) : boolean; overload;

var
  PasswordDialog: TPasswordDialog;

implementation

{$R *.dfm}

uses GnuGetText, StringUtils, WinUtils, {$IFDEF ACCESSIBLE} ShowMessageDlg {$ELSE} MsgDialogs {$ENDIF};

{------------------------------------------------------------------- }
procedure TPasswordDialog.FormCreate(Sender: TObject);
begin
  TranslateComponent (self,'dialogs');
//  AdjustClientSize(self,CancelBtn);
  Hgt:=gbEncrypt.Height;
  end;

procedure TPasswordDialog.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
{$IFDEF ACCESSIBLE}
  if (Key=VK_F11) then begin
    if ActiveControl is TCustomEdit then begin
      with (ActiveControl as TCustomEdit) do if length(TextHint)>0 then ShowHintInfo(TextHint)
      else if length(Hint)>0 then ShowHintInfo(Hint);
      end
    else with ActiveControl do if length(Hint)>0 then ShowHintInfo(Hint);
    end;
{$ENDIF}
  end;

{$IFDEF HDPI}   // scale glyphs and images for High DPI
procedure TPasswordDialog.AfterConstruction;
begin
  inherited;
  if Application.Tag=0 then
    ScaleButtonGlyphs(self,PixelsPerInch,Monitor.PixelsPerInch);
  end;
{$EndIf}

procedure TPasswordDialog.bbPasteClick(Sender: TObject);
begin
  with edtPwd do begin
    Clear; PasteFromClipboard;
    end;
  end;

{ ------------------------------------------------------------------- }
(* Passworteingabe, Ergebnis: mrOK bei "ok", mrYes bei "skip" *)
function TPasswordDialog.Execute (APos         : TPoint;
                                  const Title,Prompt : string;
                                  Skip,Enc     : boolean;
                                  var Pwd      : String;
                                  var EncMode  : integer) : TModalResult;
var
  h,w : integer;
begin
  AdjustFormPosition(Screen,self,APos);
  Caption:=Title;
  w:=GetMaxTextWidth(Prompt,lblPrompt.Font);
  h:=MulDiv(2,Monitor.PixelsPerInch,PixelsPerInchOnDesign);
  with lblPrompt do begin
    Height:=TextLineCount(Prompt)*(abs(Font.Height)+h)+2*h;
    Width:=w;
    Caption:=Prompt;
    end;
  pnTop.Height:=lblPrompt.Height+2*h;
  edtPwd.Text:='';
  btnSkip.Visible:=Skip;
  ActiveControl:=edtPwd;
  with gbEncrypt do begin
    Visible:=Enc;
    end;
  if w>ClientWidth then ClientWidth:=w+2*lblPrompt.Left;
  h:=pnTop.Height+pnInput.Height+pnBottom.Height;
  if Enc then ClientHeight:=h+Hgt else ClientHeight:=h;
  case Encmode of
  1 : rbEnc128.Checked:=true;
  2 : rbEnc256.Checked:=true;
  else  rbEncAuto.Checked:=true;
    end;
  Result:=ShowModal;
  if Result=mrOk then begin
    Pwd:=edtPwd.Text;
    if Enc then begin
      if rbEncAuto.Checked then EncMode:=0
      else if rbEnc128.Checked then EncMode:=1 else EncMode:=3;
      end;
    end;
  end;

function TPasswordDialog.Execute (APos : TPoint; const Title,Prompt : string; Skip,Enc  : boolean;
                      var Pwd : AnsiString; var EncMode  : integer) : TModalResult;
var
  s : string;
begin
  Result:=Execute (APos,Title,Prompt,Skip,false,s,EncMode);
  if Result=mrOK then Pwd:=s;
  end;

function TPasswordDialog.Execute (APos : TPoint; const Title,Prompt : string;
                                  var Pwd : AnsiString) : TModalResult;
var
  n : integer;
begin
  Result:=Execute(APos,Title,Prompt,false,false,Pwd,n);
  end;

function TPasswordDialog.Execute (const Title,Prompt : string; var Pwd : AnsiString) : TModalResult;
begin
  Result:=Execute(CenterPos,Title,Prompt,Pwd);
  end;

function ReadPassword (APos         : TPoint;
                       const Title,Prompt : string;
                       Skip,Enc     : boolean;
                       var Pwd      : String;
                       var EncMode  : integer) : TModalResult;
begin
  if not assigned(PasswordDialog) then PasswordDialog:=TPasswordDialog.Create(Application);
  Result:=PasswordDialog.Execute(APos,Title,Prompt,Skip,Enc,Pwd,EncMode);
  FreeAndNil(PasswordDialog);
  end;

function ReadPassword (APos : TPoint; const Title,Prompt : string; Skip,Enc : boolean;
                       var Pwd : AnsiString; var EncMode  : integer) : TModalResult; overload;
var
  s : string;
begin
  Result:=ReadPassword (APos,Title,Prompt,Skip,false,s,EncMode);
  if Result=mrOK then Pwd:=s;
  end;

function ReadPassword (APos : TPoint; const Title,Prompt : string;
                       Skip         : boolean;
                       var Pwd      : AnsiString) : TModalResult;
var
  n : integer;
begin
  Result:=ReadPassword (APos,Title,Prompt,Skip,false,Pwd,n);
  end;

function ReadPassword (const Title,Prompt : string;
                       Skip         : boolean;
                       var Pwd      : AnsiString) : TModalResult;
var
  n : integer;
begin
  Result:=ReadPassword (CenterPos,Title,Prompt,Skip,false,Pwd,n);
  end;

function ReadPassword (APos : TPoint; const Title,Prompt : string;
                       var Pwd : String) : boolean;
var
  n : integer;
begin
  Result:=ReadPassword (APos,Title,Prompt,false,false,Pwd,n)=mrOK;
  end;

function ReadPassword (APos : TPoint; const Title,Prompt : string;
                       var Pwd : AnsiString) : boolean;
var
  n : integer;
begin
  Result:=ReadPassword (APos,Title,Prompt,false,false,Pwd,n)=mrOK;
  end;

end.
