(* Delphi Dialog
   Hinweisfenster fr Status mit Fortschrittsanzeige und Dateiliste
   ================================================================

    Dr. J. Rathlev, D-24222 Schwentinental (kontakt(a)rathlev-home.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.
    
   Vers. 1 - Jul. 2007
   last modified: October 2021
   *)

unit TransferStat;

interface

uses Winapi.Windows, System.SysUtils, System.Classes, Vcl.Graphics, Vcl.Forms,
  Vcl.Controls, Vcl.StdCtrls, Winapi.Messages, Vcl.Buttons, Vcl.ComCtrls,
  Vcl.ExtCtrls, Prgrss, System.ImageList, Vcl.ImgList;

const
  wmCancelAction = WM_USER+1;

type
  TTransferStatus = class(TForm)
    StatusLabel: TLabel;
    Button: TBitBtn;
    lbxLog: TListBox;
    Label1: TLabel;
    ProgLabel: TLabel;
    Panel1: TPanel;
    ActionLabel: TLabel;
    DestLabel: TLabel;
    DirLabel: TLabel;
    Panel2: TPanel;
    Panel3: TPanel;
    pbTransfer: TNewProgressBar;
    ImageList: TImageList;
    procedure ButtonClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
    FHandle : HWND; // Handle of calling form
    cw      : integer;
    procedure WritePosition (Value : integer);
    procedure SetStatus (AStat : string);
    procedure SetError (AStat : string);
    procedure SetAction (AAction : string);
    procedure SetTarget (ATarget : string);
    procedure ShowDir (ADir : string);
  public
    { Public declarations }
    WStatus,Stopped      : boolean;
    FIniName,FIniSection : string;
    property Position : integer write WritePosition;
    property Action : string write SetAction;
    property Error : string write SetError;
    property Status : string write SetStatus;
    property Target : string write SetTarget;
    property Directory : string write ShowDir;

    procedure StartDelete;
    procedure StartSync;
    procedure StartCopy;
    procedure UpdateProgress (Bytes,Size : int64);
    procedure LoadFromIni (IniName, Section : string);
    procedure ShowWindow (CfHandle : HWND; ATitle,ADest,AAction : string);
    procedure CloseWindow (NDel,NCop : integer);
  end;

var
  TransferStatus: TTransferStatus;

implementation

{$R *.DFM}

uses Vcl.FileCtrl, System.IniFiles, NumberUtils, GnuGetText;

{ ---------------------------------------------------------------- }
procedure TTransferStatus.FormCreate(Sender: TObject);
begin
  TranslateComponent (self,'dialogs');
  FHandle:=0;
  FIniName:=''; FIniSection:='';
  cw:=ClientWidth;
  end;

const
  iniTop = 'Top';
  iniLeft = 'Left';
  IniWidth = 'Width';
  IniHeight = 'Height';

(* load history list *)
procedure TTransferStatus.LoadFromIni(IniName, Section : string);
begin
  FIniName:=IniName; FIniSection:=Section;
  if (length(FIniName)>0) and (length(FIniSection)>0) then begin
    with TMemIniFile.Create(IniName) do begin
      Top:=ReadInteger(FIniSection,iniTop,Top);
      Left:=ReadInteger(FIniSection,iniLeft,Left);
      ClientWidth:=ReadInteger (FIniSection,IniWidth,ClientWidth);
      ClientHeight:=ReadInteger (FIniSection,IniHeight,ClientHeight);
      Free;
      end;
    end;
  end;

procedure TTransferStatus.FormDestroy(Sender: TObject);
begin
  if (length(FIniName)>0) and (length(FIniSection)>0) then begin
    with TMemIniFile.Create(FIniName) do begin
      WriteInteger(FIniSection,iniTop,Top);
      WriteInteger(FIniSection,iniLeft,Left);
      WriteInteger (FIniSection,IniWidth,ClientWidth);
      WriteInteger (FIniSection,IniHeight,ClientHeight);
      UpdateFile;
      Free;
      end;
    end;
  end;

{ ---------------------------------------------------------------- }
procedure TTransferStatus.WritePosition (Value : integer);
begin
  with pbTransfer do begin
    if Value<Min then Value:=Min;
    if Value>Max then Value:=Max;
    Position:=Value;
    end;
  end;

procedure TTransferStatus.SetStatus (AStat : string);
begin
  with StatusLabel do begin
    Font.Color:=clBlue;
    AStat:=MinimizeName(AStat,Canvas,pbTransfer.Width);
    with lbxLog do ItemIndex:=Items.Add('  '+AStat);
    Caption:=Trim(AStat);
    end;
  end;

procedure TTransferStatus.SetError(AStat : string);
begin
  with StatusLabel do begin
    Font.Color:=clRed;
    Caption:=AStat;
    lbxLog.Items.Add(Caption);
    end;
  end;

procedure TTransferStatus.SetAction (AAction: string);
begin
  with ActionLabel do begin
    Caption:=MinimizeName(AAction,Canvas,cw-2*Left);
    end;
  end;

procedure TTransferStatus.SetTarget (ATarget : string);
begin
  DestLabel.Caption:=ATarget;
  end;

procedure TTransferStatus.ShowDir (ADir : string);
begin
  with DirLabel do Caption:=MinimizeName(ADir,Canvas,cw-2*Left);
  end;

procedure TTransferStatus.StartDelete;
begin
  Action:=dgettext('dialogs','Deleting');
  lbxLog.Items.Add(dgettext('dialogs','Delete files:'));
  end;

procedure TTransferStatus.StartSync;
begin
  Action:=dgettext('dialogs','Synchronizing');
  lbxLog.Items.Add(dgettext('dialogs','Delete files:'));
  end;

procedure TTransferStatus.StartCopy;
begin
  Action:=dgettext('dialogs','Copying to');
  with lbxLog do begin
    if Items.Count=1 then Clear;
    Items.Add(dgettext('dialogs','Copy files:'));
    end;
  end;

procedure TTransferStatus.UpdateProgress (Bytes,Size : int64);
begin
  if Size>0 then begin
    WritePosition(round(Bytes/Size*100));
    ProgLabel.Caption:=Format(dgettext('dialogs','%s of %s copied'),[SizeToStr(Bytes),SizeToStr(Size)]);
    end
  else begin
    pbTransfer.Position:=0;
    ProgLabel.Caption:='';
    end;
  end;

procedure TTransferStatus.ButtonClick(Sender: TObject);
begin
  Stopped:=true;
  if WStatus then Close
  else if FHandle<>0 then PostMessage (FHandle,wmCancelAction,0,0);
  end;

{ ---------------------------------------------------------------- }
(* Transferstatus anzeigen *)
procedure TTransferStatus.ShowWindow (CfHandle             : HWND;
                                      ATitle,ADest,AAction : string);
(* ATitle  : Fensterberschrift
   ADest   : Ziel des Datentransfers
   AAction : Name der Aktion *)
begin
  Stopped:=false; WStatus:=false;
  FHandle:=CfHandle;
  Caption:=ATitle;
  DestLabel.Caption:=ADest;
  ProgLabel.Caption:='';
  Action:=AAction;
  pbTransfer.Show;
  with StatusLabel do begin
    WordWrap:=false;
    Height:=13;
    Caption:='';
    end;
  lbxLog.Clear;
  DirLabel.Caption:='';
  with Button do begin
    Glyph:=nil;
    ImageList.GetBitmap(0,Glyph);
    Caption:=dgettext('dialogs','Cancel');
    end;
  pbTransfer.Position:=0;
  Show;
  end;

procedure TTransferStatus.CloseWindow (NDel,NCop : integer);
begin
  if Visible then Hide;
  pbTransfer.Hide;
  ProgLabel.Caption:='';
  with StatusLabel do begin
    WordWrap:=true;
    Height:=26;
    if Stopped then Caption:=dgettext('dialogs','Canceled by user!')
    else begin
      if NDel<0 then begin  // Fehler
        Font.Color:=clRed;
        Caption:=dgettext('dialogs','Error during file transfer!');
        end
      else begin
        Font.Color:=clBlue;
        if NDel>0 then begin
          if NCop>0 then
            Caption:=Format(dgettext('dialogs','%u file(s) deleted'+sLineBreak+'%u file(s) copied!'),[NDel,NCop])
          else Caption:=Format(dgettext('dialogs','%u file(s) deleted!'),[NDel])
          end
        else if NCop>0 then Caption:=Format(dgettext('dialogs','%u file(s) copied'),[NCop])
        else Caption:=dgettext('dialogs','Nothing to do!');
        end;
      end;
    end;
  WStatus:=true;
  with Button do begin
    Glyph:=nil;
    ImageList.GetBitmap(1,Glyph);
    Caption:=dgettext('dialogs','OK');
    end;
  messagebeep (mb_OK);
  Application.BringToFront;
  BringToFront;
  ShowModal;
  end;

end.
