(* Process translations using GunGetText for Delphi
   replaces context menu startable programs
   - ggdxgettext
   - ggmerge

    J. Rathlev, IEAP, Uni-Kiel, (rathlev(a)physik.uni-kiel.de)

   The contents of this file may be used under the terms of the
   Mozilla Public License ("MPL") or
   GNU Lesser General Public License Version 2 or later (the "LGPL")

   Software distributed under this License is distributed on an "AS IS" basis,
   WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
   the specific language governing rights and limitations under the License.

   J. Rathlev, Oct. 2010
   last changes:
   *)

unit TransMain;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, ComCtrls, Buttons, HListBox, xgettext, XPMan;

const
  ProgName = 'Process GnuGetText translations';
  Vers = '2.1';
  CopRgt = ' 2011 Dr. J. Rathlev, 24222 Schwentinental';
  EmailAdr = 'info(a)rathlev-home.de';

  dxgIni = 'dxgettext.ini';
  CoCmd = 'copy-mo.cmd';
  PoEdit = 'poedit.exe';

  PoExt = 'po';
  PoxExt = 'pox';
  MoExt = 'mo';
  DxGetTextIni = 'dxgettext.ini';
  defOutput = 'default';
  defCopyDir ='locale\%s\LC_MESSAGES\';

type
  TfrmTransMain = class(TForm)
    Label1: TLabel;
    cbProjDir: THistoryCombo;
    btProjDir: TSpeedButton;
    edLangSubDir: TLabeledEdit;
    cbLanguage: TComboBox;
    EditMask: TLabeledEdit;
    cbRecurse: TCheckBox;
    cbCreateIgnore: TCheckBox;
    cbRemoveIgnore: TCheckBox;
    cbAllowNonAscii: TCheckBox;
    gbDomain: TGroupBox;
    rbDefault: TRadioButton;
    rbOther: TRadioButton;
    OutputName: TLabeledEdit;
    pnTop: TPanel;
    pnLeft: TPanel;
    meProgress: TMemo;
    pnStatus: TPanel;
    laProgress: TLabel;
    btExtract: TBitBtn;
    EndeBtn: TBitBtn;
    cbOverwrite: TCheckBox;
    btMerge: TBitBtn;
    cbBackup: TCheckBox;
    gbCopy: TGroupBox;
    rbSingle: TRadioButton;
    rbMulti: TRadioButton;
    edTargetDir: TLabeledEdit;
    btCopyDir: TSpeedButton;
    btCopy: TBitBtn;
    cbPoEdit: TCheckBox;
    Label2: TLabel;
    XPManifest1: TXPManifest;
    btPoEdit: TBitBtn;
    Label3: TLabel;
    btAssemble: TBitBtn;
    btnInfo: TBitBtn;
    btEditIgnore: TBitBtn;
    OpenDialog: TOpenDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure btProjDirClick(Sender: TObject);
    procedure EndeBtnClick(Sender: TObject);
    procedure rbDefaultClick(Sender: TObject);
    procedure btExtractClick(Sender: TObject);
    procedure cbProjDirCloseUp(Sender: TObject);
    procedure btMergeClick(Sender: TObject);
    procedure btCopyDirClick(Sender: TObject);
    procedure btCopyClick(Sender: TObject);
    procedure btPoEditClick(Sender: TObject);
    procedure btAssembleClick(Sender: TObject);
    procedure rbSingleClick(Sender: TObject);
    procedure btnInfoClick(Sender: TObject);
    procedure btEditIgnoreClick(Sender: TObject);
  private
    { Private-Deklarationen }
    ProgVersName,
    ProgVersDate,
    AppPath,UserPath,
    IniName,ProgPath,
    LastDir,TemplName,
    TemplPath,MergePath,
    TextEditor,
    PoPath            : string;
    procedure LoadGetTextSettings (const ADir : string);
    procedure SaveGetTextSettings;
    procedure LoadMergeSettings(const AFilename : string);
    procedure SaveMergeSettings(const AFilename : string);
    procedure Progress (CurrentTask:widestring;CurrentFileName:widestring;LineNumber:Integer);
    procedure Warning (WarningType:TWarningType;Msg,Line:widestring;Filename:widestring;LineNumber:Integer);
    procedure OverwriteQuestion (sender: TObject; const aFileName: wideString; var Overwrite: boolean);
    function StartProgram(const Appl,Para,Dir : string) : integer;
  public
    { Public-Deklarationen }
  end;

var
  frmTransMain: TfrmTransMain;

implementation

{$R *.dfm}

uses IniFiles, JrUtils, GnuGetText, GnuGetTextInit, ShellDirDlg, WinApiUtils,
  FileUtils, WinExecute, StrUtils, RunxGetText, PoParser, ShellApi,
  SelectListItems, AssembleEngine, Registry;

{ ---------------------------------------------------------------- }
const
  IniExt = 'ini';

  (* INI-Sektionen *)
  CfgSekt  = 'Config';
  DirSekt  = 'Directories';

  (* INI-Variablen *)
  iniLangDir  = 'LanguageDir';
  iniLanguage = 'Language';
  iniLast     = 'LastDir';
  iniTop      = 'Top';
  iniLeft     = 'Left';
  IniWidth    = 'Width';
  IniHeight   = 'Height';
  IniEditor   = 'Editor';

procedure TfrmTransMain.FormCreate(Sender: TObject);
var
  IniFile  : TIniFile;
  n        : integer;
begin
  TranslateComponent(self);
  Application.Title:=_('Process GnuGetText translations for Delphi');
  InitPaths(AppPath,UserPath,ProgPath);
  InitVersion(Application.Title,Vers,CopRgt,3,3,ProgVersName,ProgVersDate);
  Caption:=Application.Title+' ('+VersInfo.Comments+')';
  IniName:=Erweiter(AppPath,PrgName,IniExt);
  IniFile:=TIniFile.Create(IniName);
  with IniFile do begin
    edLangSubdir.Text:=ReadString(CfgSekt,iniLangDir,'languages');
    with cbLanguage do begin
      n:=ReadInteger(CfgSekt,iniLanguage,ItemIndex);
      if (n>=0) and (n<Items.Count) then ItemIndex:=n;
      end;
    LastDir:=ReadString(CfgSekt,IniLast,'');
    Top:=ReadInteger(CfgSekt,iniTop,Top);
    Left:=ReadInteger(CfgSekt,iniLeft,Left);
    ClientWidth:=ReadInteger (CfgSekt,IniWidth,ClientWidth);
    ClientHeight:=ReadInteger (CfgSekt,IniHeight,ClientHeight);
    TextEditor:=ReadString(CfgSekt,IniEditor,'');
    Free;
    end;
  cbProjDir.LoadFromIni(IniName,DirSekt);
  TemplName:='';
  end;

procedure TfrmTransMain.FormDestroy(Sender: TObject);
begin
  with TIniFile.Create(IniName) do begin
    WriteString(CfgSekt,iniLangDir,edLangSubdir.Text);
    WriteInteger(CfgSekt,iniLanguage,cbLanguage.ItemIndex);
    WriteString(CfgSekt,IniLast,cbProjDir.Text);
    WriteInteger(CfgSekt,iniTop,Top);
    WriteInteger(CfgSekt,iniLeft,Left);
    WriteInteger (CfgSekt,IniWidth,ClientWidth);
    WriteInteger (CfgSekt,IniHeight,ClientHeight);
    WriteString(CfgSekt,IniEditor,TextEditor);
    Free;
    end;
  end;

procedure TfrmTransMain.FormShow(Sender: TObject);
var
  i : integer;
  s  : string;
begin
  if (length(TextEditor)=0) or not FileExists(TextEditor) then begin
    with TRegistry.Create do begin
      Access:=KEY_READ;
      RootKey:=HKEY_CLASSES_ROOT;
      s:='';
      try
        if OpenKey('.txt',false) then begin
          s:=ReadString('');
          CloseKey;
          if OpenKey(s+'\shell\open\Command',false) then s:=ReadString('')
          else s:='';
          s:=ReadNxtQuotedStr(s,#32,'"');
          end;
      finally
        Free;
        end;
      end;
    if (length(s)>0) and FileExists(s) then TextEditor:=s
    else with OpenDialog do begin
      Title:=_('Select text editor');
      InitialDir:=ProgPath;
      Filter:=_('Programs')+'|*.exe|'+_('all')+'|*.*';
      Filename:='';
      if Execute then begin
        TextEditor:=Filename;
        btEditIgnore.Enabled:=true;
        end
      else btEditIgnore.Enabled:=false;
      end;
    end;
  with cbProjDir do begin
    if not DirectoryExists(LastDir) then begin
      for i:=0 to Items.Count-1 do if DirectoryExists(Items[i]) then Break;
      if i<Items.Count then LastDir:=Items[i]
      else begin
        LastDir:='';
        if not ShellDirDialog.Execute(_('Select project directory'),true,true,UserPath,LastDir) then Close
        else AddItem(LastDir);
        end;
      end;
    ItemIndex:=Items.IndexOf(LastDir);
    if ItemIndex<0 then Items.Add(LastDir);
    LoadGetTextSettings(Text);
    end;
  end;

procedure TfrmTransMain.btnInfoClick(Sender: TObject);
begin
  InfoDialog(ProgName,ProgVersName+' - '+ProgVersDate+sLineBreak+
           VersInfo.CopyRight+sLineBreak+'E-Mail: '+EmailAdr,CenterPos);
  end;

procedure TfrmTransMain.EndeBtnClick(Sender: TObject);
begin
  Close;
  end;

// "ggdxgettext" inifile
const
  ggtSect = 'ggdxgettext';
  dxSect = 'dxgettext';
  trSect = 'ggttranslate';

  iniRecurse = 'recurse';
  iniDefault = 'nodefault';
  iniUpdate = 'updateignore';
  iniIgnore = 'useignore';
  iniAAscii = 'allownonascii';
  iniOutput =  'output';
  iniMask =  'mask';
  iniOvwr = 'overwrite';
  iniPoedit = 'StartEditor';
  iniMulti  = 'MultiCopy';
  iniCopyPath = 'CopyPath';

  defMask = '*.pas *.dfm *.c *.cpp *.inc *.rc *.dfm *.xfm *.dpr';

procedure TfrmTransMain.LoadGetTextSettings (const ADir : string);
var
  n : integer;
  s : string;
begin
  // dxgettext.ini einlesen
  with TIniFile.Create(IncludeTrailingPathDelimiter(ADir)+DxGetTextIni) do begin
    cbRecurse.Checked:=ReadBool(ggtSect,iniRecurse,false);
    cbCreateIgnore.Checked:=ReadBool(ggtSect,iniUpdate,false);
    cbRemoveIgnore.Checked:=ReadBool(ggtSect,iniIgnore,false);
    cbAllowNonAscii.Checked:=ReadBool(ggtSect,iniAAscii,false);
    if ReadBool(ggtSect,iniDefault,False) then begin
      rbOther.Checked:=true;
      with OutputName do begin
        Text:=ReadString(dxSect,iniOutput,defOutput);
        Enabled:=true;
        end;
      end
    else begin
      rbDefault.Checked:=true;
      with OutputName do begin
        Text:=''; Enabled:=false;
        end;
      end;
    EditMask.Text:=ReadString(dxSect,iniMask,defMask);
    cbOverwrite.Checked:=ReadBool(dxSect,iniOvwr,false);
    with edLangSubdir do Text:=ReadString(trSect,iniLangDir,Text);
    with cbLanguage do begin
      n:=ReadInteger(trSect,iniLanguage,ItemIndex);
      if (n>=0) and (n<Items.Count) then ItemIndex:=n;
      end;
    cbPoedit.Checked:=ReadBool(trSect,iniPoedit,true);
    if ReadBool(trSect,iniMulti,false) then rbMulti.Checked:=true
    else rbSingle.Checked:=true;
    edTargetDir.Text:=ReadString(trSect,iniCopyPath,'');
    Free;
    end;
  btMerge.Enabled:=false;
  meProgress.Clear;
  if rbOther.Checked then TemplName:=OutputName.Text
  else TemplName:='default';
  s:=edLangSubDir.Text;
  if TextPos('%s',s)=0 then s:=IncludeTrailingPathDelimiter(s)+'%s';
  MergePath:=IncludeTrailingPathDelimiter(ADir)+
    IncludeTrailingPathDelimiter(Format(s,[copy(cbLanguage.Text,1,2)]));
  PoPath:=NewExt(MergePath+TemplName,PoExt);
  LoadMergeSettings(MergePath+TemplName);
  laProgress.Caption:=_('Settings loaded for this project.');
  end;

procedure TfrmTransMain.SaveGetTextSettings;
begin
  // dxgettext.ini einlesen
  with TIniFile.Create(IncludeTrailingPathDelimiter(cbProjDir.Text)+DxGetTextIni) do begin
    WriteBool(ggtSect,iniRecurse,cbRecurse.Checked);
    WriteBool(ggtSect,iniUpdate,cbCreateIgnore.Checked);
    WriteBool(ggtSect,iniIgnore,cbRemoveIgnore.Checked);
    WriteBool(ggtSect,iniAAscii,cbAllowNonAscii.Checked);
    WriteBool(ggtSect,iniDefault,rbOther.Checked);
    if rbOther.Checked then WriteString(dxSect,iniOutput,OutputName.Text)
    else WriteString(dxSect,iniOutput,'');
    WriteString(dxSect,iniMask,EditMask.Text);
    WriteBool(dxSect,iniOvwr,cbOverwrite.Checked);
    WriteString(trSect,iniLangDir,edLangSubdir.Text);
    WriteInteger(trSect,iniLanguage,cbLanguage.ItemIndex);
    WriteBool(trSect,iniPoedit,cbPoedit.Checked);
    WriteBool(trSect,iniMulti,rbMulti.Checked);
    WriteString(trSect,iniCopyPath,edTargetDir.Text);
    Free;
    end;
  end;

const
  ggmSect = 'ggmerge';

  iniTemplate = 'template';
  iniBackup ='createbackup';
  iniSAscii = 'supportnonascii';

procedure TfrmTransMain.LoadMergeSettings(const AFilename : string);
begin
  with TIniFile.Create(NewExt(AFilename,IniExt)) do begin
    TemplPath:=ReadString(ggmSect,iniTemplate,NewExt(IncludeTrailingPathDelimiter(cbProjDir.Text)+TemplName,PoExt));
    cbBackup.Checked:=ReadBool(ggmSect,iniBackup,true);
    Free;
    end;
  end;

procedure TfrmTransMain.SaveMergeSettings(const AFilename : string);
begin
  with TIniFile.Create(NewExt(AFilename,IniExt)) do begin
    WriteString(ggmSect,iniTemplate,TemplPath);
    WriteBool(ggmSect,iniBackup,cbBackup.Checked);
    WriteBool(ggmSect,iniSAscii,cbAllowNonAscii.Checked);
    Free;
    end;
  end;

procedure TfrmTransMain.btProjDirClick(Sender: TObject);
var
  s : string;
begin
  s:=cbProjDir.Text;
  if ShellDirDialog.Execute(_('Select project source directory'),false,true,'',s) then begin
    SaveGetTextSettings;
    with cbProjDir do begin
      AddItem(s);
      ItemIndex:=0;
      LoadGetTextSettings(Text);
      end;
    end;
  end;

procedure TfrmTransMain.cbProjDirCloseUp(Sender: TObject);
begin
  SaveGetTextSettings;
  with cbProjDir do LoadGetTextSettings (Items[ItemIndex]);
  end;

procedure TfrmTransMain.btCopyDirClick(Sender: TObject);
var
  s : string;
begin
  s:=edTargetDir.Text;
  if length(s)=0 then s:=ExtractFilePath(cbProjDir.Text);
  if ShellDirDialog.Execute(_('Select program directory of project'),false,true,'',s) then begin
    edTargetDir.Text:=s;
    end;
  end;

procedure TfrmTransMain.rbDefaultClick(Sender: TObject);
begin
  OutputName.Enabled:=rbOther.Checked;
  end;

procedure TfrmTransMain.rbSingleClick(Sender: TObject);
begin
  btAssemble.Enabled:=rbSingle.Checked;
  end;

procedure TfrmTransMain.Progress(CurrentTask, CurrentFileName: widestring;
  LineNumber: Integer);
begin
  laProgress.Caption:=CurrentTask;
  laProgress.Update;
  if LineNumber<=1 then begin
    meProgress.Lines.Add(CurrentTask);
    Application.ProcessMessages;
    end;
  end;

procedure TfrmTransMain.Warning(WarningType: TWarningType; Msg, Line,
  Filename: widestring; LineNumber: Integer);
begin
  with meProgress do begin
    Lines.Add(Msg);
    Lines.Add(Format(_('Line: %s'),[Line]));
    Lines.Add('');
    end;
  end;

procedure TfrmTransMain.OverwriteQuestion(sender: TObject; const aFileName: wideString; var Overwrite: boolean);
begin
  Overwrite:=cbOverwrite.Checked
  or ConfirmDialog(Format(_('Do you want to overwrite the file named %s?'),[aFilename]));
  end;

{ ---------------------------------------------------------------- }
procedure TfrmTransMain.btExtractClick(Sender: TObject);
var
  xgt  : TXGetText;
  rxgt : TRunXGettext;

  procedure Explode (line : string; sl : TStrings);
  var
    i,last : integer;
    item   : string;
  begin
    last:=1;
    line:=line+' ';
    for i:=1 to length(line) do begin
      if line[i]<=#32 then begin
        item:=trim(copy(line,last,i-last));
        if item<>'' then sl.Add (item);
        last:=i;
        end;
      end;
    end;

begin
  SaveGetTextSettings;
  btMerge.Enabled:=false;
  xgt:=TXGetText.Create;
  meProgress.Clear;
  with xgt do begin
    Recurse:=cbRecurse.Checked;
    UpdateIgnore:=cbCreateIgnore.Checked;
    UseIgnoreFile:=cbRemoveIgnore.Checked;
    DestinationPath:=IncludeTrailingPathDelimiter(cbProjDir.Text);
    AddBaseDirectory(DestinationPath);
    AllowNonAscii:=cbAllowNonAscii.Checked;
    if rbOther.Checked then defaultDomain:=OutputName.Text;    
    Explode(EditMask.Text,filemasks);
    OnProgress:=Progress;
    OnWarning:=Warning;
    OnOverwrite:=OverwriteQuestion;
    Application.ProcessMessages;
    try
      Execute;
      if CFiles.Count<>0 then begin
        if Assigned(OnProgress) then
          OnProgress (Format(_('Scanning %u C/C++ files'),[CFiles.Count]),'C/C++ files',0);
        rxgt:=TRunXGettext.Create;
        with rxgt do begin
          FileList.Assign(CFiles);
          OutputDir:=DestinationPath;
          try
            Execute;
          finally
            Free;
            end;
          end;
        end;
    finally
      Free;
      end;
    end;
  with meProgress.Lines do begin
    if Count=0 then Add(_('No warnings or errors.'));
    Add('');
    end;
  laProgress.Caption:=_('Template was created.');
  btMerge.Enabled:=true;
  end;

{ ---------------------------------------------------------------- }
function TfrmTransMain.StartProgram(const Appl,Para,Dir : string) : integer;
const
  BUFSIZE = 1024;
var
  si        : TStartupInfo;
  pi        : TProcessInformation;
  ec        : dword;
begin
// Create process to start Assembler
  FillChar(si, SizeOf(TStartupInfo), 0);
  with si do begin
    cb := Sizeof(TStartupInfo);
    dwFlags := STARTF_USESTDHANDLES or STARTF_USESHOWWINDOW;
    wShowWindow:=SW_HIDE;
    end;
  if CreateProcess(nil,                // Anwendungsname
                   pchar(Quote+Appl+Quote+Space+Para),
                   nil,                // Security
                   nil,                // Security
                   true,               // use InheritHandles
                   NORMAL_PRIORITY_CLASS, // Prioritt
                   nil,                   // Environment
                   pchar(Dir),     // Verzeichnis
                   si,pi) then begin
    while WaitForSingleObject(pi.hProcess,100)<>WAIT_OBJECT_0 do Application.ProcessMessages;
    GetExitCodeProcess(pi.hProcess,ec);
    CloseHandle(pi.hProcess);
    Result:=ec;
    end
  else Result:=-GetLastError;
  end;

procedure TfrmTransMain.btMergeClick(Sender: TObject);
var
  translist : TPoEntryList;
  pe,petr   : TPoEntry;
  parser    : TPoParser;
  tf        : TextFile;
  fs        : TFileStream;
  hd,tmp    : TStringList;
  sv,sx,s   : string;
  res,n,i   : integer;
  ok,cphd   : boolean;

  function shellescape (s:string):string;
  var
    i:integer;
  begin
    Result:='';
    for i:=1 to length(s) do begin
      if s[i]='"' then Result:=Result+'\"'
      else if s[i]='\' then Result:=Result+'\\'
      else Result:=Result+s[i];
      end;
    end;

begin
  // Template
  sv:=NewExt(IncludeTrailingPathDelimiter(cbProjDir.Text)+TemplName,PoExt);
  // Translation
  if not DirectoryExists(MergePath) then ForceDirectories(MergePath);
  cphd:=FileExists(PoPath);  // copy header after merging
  if not cphd then FileCopy(sv,Popath); // Init translation
  sx:=NewExt(PoPath,PoxExt);
  meProgress.Lines.Add(Format(_('Merging with %s template.'),[copy(cbLanguage.Text,5,255)]));
  ok:=true;
  if cbAllowNonAscii.Checked then begin  // Non-ASCII support required. Use internal function.
    FileMode:=fmOpenRead;
    hd:=TStringList.Create;
    if cphd then begin // read header  - JR 2011-07-28
      AssignFile (tf,PoPath);
      Reset(tf); ok:=false;
      while not Eof(tf) and not ok do begin
        readln(tf,s);
        if uppercase(copy(s,1,5))='MSGID' then begin
          ok:=length(Trim(copy(s,6,length(s))))>2;                               //  first non zero Id
          end;
        if not ok then hd.Add(s);
        end;
      CloseFile(tf);
      end;
    translist:=TPoEntryList.Create;
    try
      translist.LoadFromFile(PoPath);
      AssignFile (tf,sv);
      Reset (tf);
      try
        parser:=TPoParser.Create;
        try
          fs:=TFileStream.Create (sx,fmCreate);
          try
            while true do begin
              pe:=parser.ReadNextEntry(tf);
              if pe=nil then break;
              petr:=translist.Find(pe.MsgId);
              if petr<>nil then begin
                pe.MsgStr:=petr.MsgStr;
                pe.UserCommentList.Text:=petr.UserCommentList.Text;
                end;
              pe.WriteToStream(fs);
              end;
          finally
            FreeAndNil (fs);
            end;
        finally
          FreeAndNil (parser);
          end;
      finally
        CloseFile (tf);
        end;
    finally
      FreeAndNil (translist);
      end;
    if cphd then begin // insert original header - JR 2011-07-28
      tmp:=TStringList.Create;
      with tmp do begin
        LoadFromFile(sx);  ok:=false;
        repeat
          s:=Strings[0];
          if uppercase(copy(s,1,5))='MSGID' then begin
            system.delete(s,1,5);
            ok:=length(Trim(s))>2;                          //  first non zero Id
            end;
          if not ok then delete(0);
          until ok or (Count=0);
        for i:=0 to hd.Count-1 do Insert(i,hd.Strings[i]);  // insert old header
        SaveToFile(sx);
        end;
      end;
    hd.Free;
    end
  else begin
    // ASCII only. Use external msgmerge.exe
//      cmdline:='msgmerge.exe -q "'+shellescape(st)+'" "'+shellescape(sv)+'" -o "'+shellescape(sx)+'" 2>&1';
//      res:=ExecConsoleApp('bash.exe','-c "'+shellescape(cmdline)+'"',sl,nil);
    res:=StartProgram('msgmerge.exe','-q "'+PoPath+'" "'+sv+'" -o "'+sx,MergePath);
    ok:=res=0;
    if res<0 then ErrorDialog(Format(_('msgmerge.exe failed with exit code %s.'),[IntToStr(-res)]))
    else if res>0 then ErrorDialog(_('Because there was unexpected output from msgmerge.exe, no merging has been done.'));
    end;
  if ok then begin
    sv:=NewExt(PoPath,'~'+PoExt);
    if FileExists(sv) then DeleteFile(sv);
    if cbBackup.Checked then begin
      if FileExists(PoPath) then ok:=RenameFile(PoPath,sv)
      else ok:=true;
      if not ok then ErrorDialog(Format(_('Cannot rename %s to %s'),[PoPath,sv]));
      end
    else begin
      ok:=deletefile (PoPath);
      if not ok then ErrorDialog(Format(_('Cannot delete %s'),[PoPath]));
      end;
    if ok then begin
      ok:=FileExists(sx);
      if ok then begin
        ok:=RenameFile(sx,PoPath);
        if not ok then ErrorDialog(Format(_('Cannot rename %s to %s'),[sx,PoPath]))
        end
      end;
    end;
  if ok then begin
    SaveMergeSettings(MergePath+TemplName);
    laProgress.Caption:=_('The template was merged into the translation file.');
    if cbPoEdit.Checked then ShellExecute (Application.Handle,'open',PChar(PoPath),nil,nil,SW_RESTORE)
    else begin
      res:=StartProgram('msgfmt.exe',PoPath+' -o '+NewExt(PoPath,MoExt),MergePath);
      if res<0 then ErrorDialog(Format(_('msgfmt.exe failed with exit code %s.'),[IntToStr(-res)]))
      else if res>0 then ErrorDialog(_('Compiling to  MO file failed.'))
      else meProgress.Lines.Add(Format(_('compiled to MO %s'),[PoPath]));
      end
    end
  else laProgress.Caption:=_('Merging of template failed.');
  meProgress.Lines.Add('');
  end;

{ ---------------------------------------------------------------- }
procedure TfrmTransMain.btCopyClick(Sender: TObject);
var
  sm,sd : string;
  ok    : boolean;

  function CopyToDirs (const Filename,Dir,SubFolder : string) : boolean;
  var
    DirInfo    : TSearchRec;
    Findresult : integer;
  begin
    Result:=true;
    FindResult:=FindFirst (Erweiter(Dir,'*',''),faAnyFile,DirInfo);
    while FindResult=0 do with DirInfo do begin
      if NotSpecialDir(Name) then begin
        if((Attr and faDirectory)<>0) then Result:=Result and CopyToDirs(Filename,Erweiter(Dir,Name,''),SubFolder);
        end;
      FindResult:=FindNext (DirInfo);
      end;
    FindClose(DirInfo);
    if AnsiEndsText(SubFolder,IncludeTrailingPathDelimiter(Dir)) then begin
      try
        meProgress.Lines.Add('  ==> '+Dir);
        FileCopy(sm,IncludeTrailingPathDelimiter(Dir)+ExtractFilename(sm));
      except
        on E:EInOutError do begin
          meProgress.Lines.Add(_('  *** Error copying file:'));
          meProgress.Lines.Add('  '+E.Message);
          Result:=false;
          end;
        end;
      end;
    end;

begin
  if length(edTargetDir.Text)= 0 then btCopyDirClick(Sender);
  if DirectoryExists(edTargetDir.Text) then begin
    sm:=NewExt(PoPath,MoExt);
    meProgress.Lines.Add(Format(_('Copying MO file %s'),[sm]));
    sd:=Format(defCopyDir,[copy(cbLanguage.Text,1,2)]);
    ForceDirectories(IncludeTrailingPathDelimiter(edTargetDir.Text)+sd);
    if rbSingle.Checked then begin
      sd:=IncludeTrailingPathDelimiter(edTargetDir.Text)+sd+ExtractFilename(sm);
      try
        meProgress.Lines.Add('  ==> '+sd);
        FileCopy(sm,sd);
        ok:=true;
      except
        on E:EInOutError do begin
          meProgress.Lines.Add(_('  *** Error copying file:'));
          meProgress.Lines.Add('  '+E.Message);
          ok:=false;
          end;
        end;
      end
    else begin  // multi copy
      ok:=CopyToDirs(sm,edTargetDir.Text,sd);
      end;
    if ok then laProgress.Caption:=_('MO file was copied.')
    else laProgress.Caption:=_('Copying of MO file failed.');;
    meProgress.Lines.Add('');
    end
  else ErrorDialog(Format(_('Folder for executables not found: '),[edTargetDir.Text]));
  SaveGetTextSettings;
  end;

{ ---------------------------------------------------------------- }   
procedure TfrmTransMain.btPoEditClick(Sender: TObject);
begin
  ShellExecute (Application.Handle,'open',PChar(PoPath),nil,nil,SW_RESTORE)
  end;

procedure TfrmTransMain.btEditIgnoreClick(Sender: TObject);
begin
  ShellExecute (Application.Handle,'open',PChar(TextEditor),
    PChar(NewExt(IncludeTrailingPathDelimiter(cbProjDir.Text)+'Ignore',PoExt)),nil,SW_RESTORE);
  ShellExecute (Application.Handle,'open',PChar(TextEditor),
    PChar(NewExt(IncludeTrailingPathDelimiter(cbProjDir.Text)+TemplName,PoExt)),nil,SW_RESTORE);
  end;

{ ---------------------------------------------------------------- }
procedure TfrmTransMain.btAssembleClick(Sender: TObject);
var
  DirInfo    : TSearchRec;
  Findresult : integer;
  sl         : TStringList;
  Ext        : string;
  i,n        : integer;
  AssEng     : TAssembleEngine;
  ok         : boolean;
begin
  if DirectoryExists(edTargetDir.Text) then begin
    AssEng:=TAssembleEngine.Create(edTargetDir.Text);
    sl:=TStringList.Create;
    ok:=false;
    with AssEng do begin
      SetGnuGettextPatchCode;
      filemask:='*.mo';
      PrepareFileList;
      if FileList.Count=0 then ErrorDialog(_('No translations (.mo files) found.'))
      else begin
        with Filelist do for i:=0 to Count-1 do sl.AddObject(Strings[i],pointer(true));
        if SelectListItemsDialog.Execute(TopRightPos(btAssemble,Point(0,-100)),_('Embed translations'),
            _('Select the translations you want to embed into your executables'),
            edTargetDir.Text,sl) then begin
          with sl do for i:=0 to Count-1 do if not boolean(Objects[i]) then SkipFile (Strings[i]);
          ok:=true;
          end;
        end;
      end;
    n:=-1;
    if ok then begin
      sl.Clear;
      FindResult:=FindFirst (Erweiter(edTargetDir.Text,'*',''),faAnyFile,DirInfo);
      while FindResult=0 do with DirInfo do begin
        if NotSpecialDir(Name) then begin
          Ext:=GetExt(Name);
          if((Attr and faDirectory)=0) and (AnsisameText(Ext,'exe') or AnsiSameText(Ext,'dll')) then
            sl.AddObject(Name,pointer(true));
          end;
        FindResult:=FindNext (DirInfo);
        end;
      FindClose(DirInfo);
      if SelectListItemsDialog.Execute(TopRightPos(btAssemble,Point(0,-100)),_('Executable files'),
                  _('Select all executables where you want to embed the translations'),
                  edTargetDir.Text,sl) then with sl do begin
        n:=0;
        for i:=0 to Count-1 do if boolean(Objects[i]) then begin
          meProgress.Lines.Add(Format(_('Embedding translation into %s'),[Strings[i]]));
          with AssEng do begin
            exefilename:=IncludeTrailingPathDelimiter(edTargetDir.Text)+Strings[i];
            try
              Execute;
              inc(n);
            except
              on E: Exception do begin
                meProgress.Lines.Add(_('  *** Error on embedding translations:'));
                meProgress.Lines.Add('  '+E.Message);
                end;
              end;
            end;
          end;
        end;
      end;
    sl.Free;
    AssEng.Free;
    if n>0 then laProgress.Caption:=Format(_('%u files processed.'),[n])
    else laProgress.Caption:=_('No files processed.');
    meProgress.Lines.Add('');
    end
  else ErrorDialog(Format(_('Folder for executables not found: '),[edTargetDir.Text]));
  end;

end.
