unit xgettext;
(****************************************************************)
(*                                                              *)
(*  (C) Copyright by Lars B. Dybdahl, Jens Berke and            *)
(*        Jacques Garcia Vazquez                                *)
(*  E-mail: Lars@dybdahl.dk, phone +45 70201241                 *)
(*  You received this file under the Mozilla Public License 1.1 *)
(*                                                              *)
(*  See http://dybdahl.dk/dxgettext/ for more information       *)
(*                                                              *)
(****************************************************************)

// changes for Windows 7: J. Rathlev, Nov. 2009

interface

uses
  Classes, poparser;

type
  {TXExcludeFormClassProperties: represents 1..n properties of a certain class
   that shall be excluded from string extraction in form files. }
  TXExcludeFormClassProperties = class(TCollectionItem)
  private
    FProperties: TStringList;
    FNameOfClass: string;
    procedure SetNameOfClass(const Value: string);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    function ExcludeFormClassProperty(aPropertyname: string): boolean;
    procedure AddFormClassProperty(aPropertyname: string);
    property NameOfClass: string read FNameOfClass write SetNameOfClass; // "Classname" already used by TObject => needed other name
  end;

  {TXExcludeFormClassPropertyList: represents a collection of
   TXExcludeFormClassProperties}
  TXExcludeFormClassPropertyList = class(TCollection)
  private
    function GetItems(Index: integer): TXExcludeFormClassProperties;
    procedure SetItem(Index: integer;
      const Value: TXExcludeFormClassProperties);
    function Add: TXExcludeFormClassProperties;
    function AddFormClass(aClassname: string): TXExcludeFormClassProperties;
  public
    function FindItem(aClassname: string): TXExcludeFormClassProperties;
    function ExcludeFormClassProperty(aClassname, aPropertyname: string): Boolean;
    function AddFormClassProperty(aClassPropertyname: string): TXExcludeFormClassProperties;
    property Items[Index: integer]: TXExcludeFormClassProperties read GetItems write SetItem; default;
  end;

  {TXExcludes: holds all information about what shall be excluded from string
   extraction, specified in a "ggexclude.cfg" file }
  TXExcludes = class(TObject)
  private
    FFormClasses: TStringList;
    FFormInstances: TStringList;
    FDirectories: TStringList;
    FFiles: TStringList;
    FBaseDirectory: string;
    FExcludeFormClassPropertyList: TXExcludeFormClassPropertyList;
    FLastErrorMsg: widestring;
    function GetFullInternalPath(s:string): string;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function AddDirectory(aDirectory: string): boolean;
    function AddFormFile(aFilename: string): boolean;
    function AddFormClass(aClassname: string): boolean;
    function AddFormClassProperty(aPropertyname: string): boolean;
    function AddFormInstance(aInstanceName: string): boolean;
    function ExcludeDirectory(aDirectory: string): Boolean;
    function ExcludeFormFile(aFilename: string): Boolean;
    function ExcludeFormClass(aClassname: string): Boolean;
    function ExcludeFormClassProperty(aClassname, aPropertyname: string): Boolean; overload;
    function ExcludeFormClassProperty(aClassname: string): Boolean; overload;
    function ExcludeFormInstance(aFilename, aInstanceName: string): boolean;
    function FormClassHasWildcard(aClassname: string): Boolean;
    property BaseDirectory: string read FBaseDirectory write FBaseDirectory;
    property LastErrorMsg: widestring read FLastErrorMsg;
  end;

  TOnOverwrite = Procedure (sender: TObject; const aFileName: wideString; var Overwrite: boolean) of object;
  TWarningType=
    (wtGenericWarning, wtUnexpectedException, wtConstantReplaced,
     wtSyntaxError, wtParameterError, wtExtendedDirectiveError, wtNonAscii,
     wtNotImplemented,
     wtExcludeFile);
  TXGTDomain=
    class
    public
      msgid:TStringList;  // Sorted for binary lookups, objects are TItem
      order:TStringList;  // same as msgid, but sorted after occurence. Points to the same objects as msgid, so don't free them!
      constructor Create;
      destructor Destroy; override;
    end;
  TOnProgress=
    procedure (CurrentTask:widestring;CurrentFileName:widestring;LineNumber:Integer) of object;
  TOnWarning=
    procedure (WarningType:TWarningType;Msg,Line:widestring;Filename:widestring;LineNumber:Integer) of object;

  TXGetText =
    class
    private
      ignorelist: TPoEntryList;
      domainlist: TStringList; // Strings are domain name, values are TXGTDomain
      constlist:TStringList;   // List of consts. Strings are names, Objects are TConst
      definedDomain: string;
      procedure doHandleExtendedDirective (line: wideString);
      procedure ClearConstList;
      function GetDomain(domain: widestring): TXGTDomain;
      procedure AddTranslation(domain:widestring; msgid: widestring; Comments, Location: widestring);
      procedure WriteAll(Destinationpath:widestring; domain: widestring);
      function MakePathLinuxRelative(path: widestring): widestring;
    private
      resourcestringmode: Integer;  // 0=None, 1=Const, 2=Resourcestring
      CurrentFilename:widestring;
      LastLineRead:widestring;
      linenr:Integer;
      commentmode:widestring; // Empty means that dxreadln is not inside a comment
      lastcomment:widestring;
      BaseDirectoryList:TStringList; // Always ends in a pathdelimiter
      BaseDirectory:string;
      Excludes: TXExcludes;
      procedure WritePoFiles (DestinationPath: widestring);
      procedure Warning (WarningType:TWarningType;msg:widestring); overload;
      procedure Warning (WarningType:TWarningType;Msg,Line:widestring;Filename:widestring;LineNumber:Integer); overload;
      procedure dxreadln (var src:TextFile;var line:widestring); // same as system.readln, but takes care of comments
      procedure extractstring(var source:widestring;var res: widestring);
      function readstring(var line: widestring; var src: TextFile): widestring; // Reads a pascal ansistring constant
      procedure ExtractFromPascal(sourcefilename: widestring);
      procedure ExtractFromDFM(sourcefilename: widestring);
      procedure ExtractFromRC(sourcefilename: widestring);
      {$ifdef mswindows}
      procedure ExtractFromEXE(sourcefilename: widestring);
      {$endif}
      procedure ExtractFromFile(sourcefilename: widestring);
      procedure ExtractFromFileMasks(mask: widestring);
      procedure ParseExcludeFile;
    public
      // When set, only default domain is written to a file, and this file has it's filename from this variable
      SingleOutputFilename:string;
      
      OnProgress:TOnProgress;
      OnWarning:TOnWarning;
      Recurse:boolean;
      UpdateIgnore:boolean;  // Will create ignore.po if not exists, and put obvious untranslatable items into it
      UseIgnoreFile:boolean; // Will make sure that no item from ignore.po is stored in other files
      AllowNonAscii:boolean;
      OrderbyMsgid:boolean;
      NoWildcards:boolean;
      defaultDomain:string;
      filemasks:TStringList;
      DestinationPath:string;
      CFiles:TStringList;   // This will contain filenames of C/C++ source files to be scanned elsewhere
      OnOverwrite: TOnOverwrite;
      constructor Create;
      destructor Destroy; override;
      procedure AddBaseDirectory (path:string);
      procedure AddDelphiFilemasks;
      procedure AddKylixFilemasks;
      procedure HandleIgnores;
      procedure Execute;
    end;




implementation

uses
{$ifdef MSWINDOWS}
  Windows, ExeImage, rxtypes,
{$endif}
  SysUtils, Math, appconsts, gnugettext, xgettexttools, Masks,
  ignoredetector, StrUtils;

type
  TConst=
    class
      name:widestring;
      value:widestring;
    end;
  EGetText=
    class (Exception)
    end;

const
  cDefineDirective  = '{gnugettext:'; // Start of an option inside the source code
  cScanOption       = 'scan-all';     // Means that all strings in the source code should be extracted
  cDomainDefinition = 'text-domain';  // Changes default text domain for strings
  cScanResetOption  = 'reset';        // Changes back to default behaviour

  { consts for exclusion of files, directories, properties and classes from extraction: }
  cExcludeFormInstance = 'exclude-form-instance';
  cExcludeFormClassProperty = 'exclude-form-class-property';
  cExcludeFormClass = 'exclude-form-class';
  cExcludeFile = 'exclude-file';
  cExcludeDir = 'exclude-dir';

function RemoveNuls (const s:widestring):widestring;
// Since #0 is used to separate msgid_plural values inside msgid strings
// in this software, #0 cannot be present in msgid values. In order to
// prevent this, this function replaces #0 with '#0'.
var
  p:integer;
begin
  Result:=s;
  while true do begin
    p:=pos(#0,Result);
    if p=0 then break;
    Result:=MidStr(Result,1,p-1)+'#0'+MidStr(Result,p+1,maxint);
  end;
end;

procedure TXGetText.extractstring(var source: widestring; var res: widestring);
const whitespace=[#0..#32];
// Extracts the Pascal coded string at the beginning of source.
// Returns the result in res.
// Removes the extracted data from source.
var
  charset: set of char;
  s: widestring;
  constname,uconstname:widestring;
  idx:integer;
begin
  res := '';
  while source <> '' do begin
    case source[1] of
      '#':
        begin
          if copy(source, 2, 1) = '$' then begin
            s := '$';
            delete(source, 1, 2);
            charset := ['0'..'9', 'a'..'f', 'A'..'F'];
          end else begin
            delete(source, 1, 1);
            s := '';
            charset := ['0'..'9'];
          end;
          while (source <> '') and (ord(source[1])<=255) and (char(ord(source[1])) in charset) do begin
            s := s + source[1];
            delete(source, 1, 1);
          end;
          res := res + widechar(StrToInt(s));
          while (source<>'') and (ord(source[1])<=255) and (char(ord(source[1])) in whitespace) do delete (source,1,1);
          if (length(trim(source))>=2) and (copy(source,1,1)='+') then delete (source,1,1);
        end;
      '''':
        begin
          delete(source, 1, 1);
          while true do begin
            if source = '' then begin
              Warning (wtSyntaxError,_('Single quote detected - string starts but does not end'));
              exit;
            end;
            if copy(source, 1, 1) = '''' then begin
              if copy(source, 2, 1) = '''' then begin
                // Double quote detected
                res := res + '''';
                delete(source, 1, 2);
              end else begin
                // End of text part detected
                delete(source, 1, 1);
                break;
              end
            end else begin
              res := res + copy(source, 1, 1);
              delete(source, 1, 1);
            end;
          end;
        end;
      'a'..'z','A'..'Z','_':
        begin
          constname:='';
          while (source<>'') and (ord(source[1])<=255) and (char(ord(source[1])) in ['a'..'z','A'..'Z','_','0'..'9']) do begin
            constname:=constname+source[1];
            delete (source,1,1);
          end;
          uconstname:=uppercase(constname);
          if constlist.Find(uconstname,idx) then begin
            res:=res+(constlist.Objects[idx] as TConst).value;
          end else
          if uconstname='CRLF' then begin
            res:=res+#10;
            if (resourcestringmode<>1) then
              Warning (wtConstantReplaced,Format(_('CRLF substituted with #10 for %s. Consider to use sLineBreak instead.'),[constname]));
          end else
          if uconstname='SLINEBREAK' then begin
            // Don't make a warning on this one because it is so common
            res:=res+#10;
          end else
          if uconstname='EOF' then begin
            // Don't make a warning on this one because it is so common
            res:=res+#26;
          end else
          if uconstname='EOL' then begin
            // Don't make a warning on this one because it is so common
            res:=res+#10;
          end else
          if (uconstname='DEPRECATED') or (uconstname='PLATFORM') or (uconstname='LIBRARY') then begin
            // The hinting directive was detected and ignored.
          end else
          begin
            if resourcestringmode=1 then // Don't handle consts that don't work
              break;
            Warning (wtGenericWarning,Format(_('Constant %s is not known.'),[constname]));
          end;
        end;
    else
      break;
    end;
    while (source<>'') and (ord(source[1])<=255) and (char(ord(source[1])) in whitespace) do delete (source,1,1);
    if (length(trim(source))>=2) and (copy(source,1,1)='+') then delete (source,1,1);
    while (source<>'') and (ord(source[1])<=255) and (char(ord(source[1])) in whitespace) do delete (source,1,1);
  end;
end;

function TXGetText.readstring(var line: widestring; var src: TextFile): widestring;
var
  s: widestring;
  pluscoming:boolean;
  i:integer;
  ansis:ansistring;
  found:boolean;
begin
  Result := '';
  while true do begin
    if line='' then
      dxreadln(src, line);
    extractstring(line, s);
    Result := Result + s;
    line := trim(line);
    pluscoming:=(line='');
    if (line='+') or pluscoming then begin
      // This is a multi-line string
      dxreadln(src, line);
      line := trim(line);
      if pluscoming then begin
        if copy(line,1,1)='+' then begin
          delete (line,1,1);
          line:=trim(line);
        end else begin
          if resourcestringmode<>1 then
            Warning (wtSyntaxError,_('This line is not connected with the previous line using a plus (+).'));
          break;
        end;
      end;
    end else
      break;
  end;
  // Find out if there is just one character above 255
  found:=False;
  for i:=1 to length(Result) do begin
    if ord(Result[i])>=256 then begin
      found:=True;
      break;
    end;
  end;
  if not found then begin
    // Assume the string is not unicode, but the local character set.
    // Move all characters to an ansistring
    SetLength (ansis,length(Result));
    for i:=1 to length(Result) do
      ansis[i]:=char(ord(Result[i]));
    // Convert from local character set to widestring
    Result:=StringToWidestring(ansis);
  end;
end;

function TXGetText.MakePathLinuxRelative (path:widestring):widestring;
var
  baselen:integer;
begin
  baselen:=length(BaseDirectory);
  {$ifdef MSWINDOWS}
  if uppercase(copy(path,1,baselen))=uppercase(BaseDirectory) then begin
    Result:=copy(path,baselen+1,maxint);
  end else begin
    Result:=copy(path,3,maxint);
  end;
  Result:=WindowsPathDelim2LinuxPathDelim(Result);
  {$endif}
  {$ifdef LINUX}
  if copy(path,1,baselen)=BaseDirectory then begin
    Result:=copy(path,baselen+1,maxint);
  end else begin
    Result:=path;
  end;
  {$endif}
end;

procedure TXGetText.ExtractFromPascal(sourcefilename: widestring);
// I didn't have a Pascal parser available when this code was written.
var
  src: TextFile;
  line, uline:widestring;
  s:widestring;
  msgid: widestring;
  p, p2, idx:Integer;
  domain: widestring;
  co:TConst;
  constident:widestring;
  idlength,idoffset:integer;
  idplural:boolean;
begin
  if lowercase(extractfilename(sourcefilename)) = 'gnugettext.pas' then exit;
  if lowercase(extractfilename(sourcefilename)) = 'gnugettextd5.pas' then exit;
  ClearConstList;
  FileMode:=fmOpenRead;
  AssignFile(src, sourcefilename);
  Reset(src);
  try
    definedDomain := '';
    lastcomment := '';
    resourcestringmode := 0;
    linenr := 0;
    while not eof(src) do begin
      dxreadln(src, line);
      line := trim(line);

      s := ConvertWhitespaceToSpaces (uppercase(line)) + ' ';

      // This should catch resourcestring start
      if (copy(s, 1, 15) = 'RESOURCESTRING ') then begin
        resourcestringmode := 2;
        delete (line,1,15);
      end;
      if (copy(s, 1, 6) = 'CONST ') then begin
        resourcestringmode := 1;
        delete (line,1,6);
      end;
      // This should catch all ends of resourcestring areas
      if (copy(s, 1, 9) = 'FUNCTION ') or (copy(s, 1, 10) = 'PROCEDURE ') or
        (copy(s, 1, 6) = 'BEGIN ') or (copy(s, 1, 4) = 'VAR ') or
        (copy(s, 1, 5) = 'TYPE ') or
        (copy(s, 1, 12) = 'CONSTRUCTOR ') or (copy(s, 1, 11) = 'DESTRUCTOR ') then
        resourcestringmode := 0;

      if resourcestringmode<>0 then begin
        while true do begin
          line:=trim(line);
          p := pos('''', line);
          if p = 0 then
            break;

          s:=trim(copy(line,1,p-1));
          if copy(s,length(s),1)='=' then begin
            // Identifier probably precedes the string
            s:=trim(copy(s,1,length(s)-1));
            if is_identifier(s) then
              constident:=s
            else
              constident:='';
          end;

          delete(line, 1, p - 1);
          // Extract the string
          msgid:=RemoveNuls(readstring(line, src));
          if resourcestringmode=2 then begin
            if constident<>'' then begin
              if lastcomment<>'' then
                lastcomment:=lastcomment+sLinebreak;
              lastcomment:=lastcomment+'Programmer''s name for it: '+constident;
            end;
            AddTranslation(defaultDomain, msgid, lastcomment, MakePathLinuxRelative(sourcefilename)+':'+IntToStr(linenr));
            lastcomment := '';
          end;
          if constident<>'' then begin
            if constlist.Find(uppercase(constident),idx) then begin
              co:=constlist.Objects[idx] as TConst;
            end else begin
              co:=TConst.Create;
              co.Name:=constident;
              constlist.AddObject(uppercase(co.name),co);
            end;
            co.Value:=msgid;

            // If source-code comments for gnugettext enable it,
            // extract the constant even though it is not a resourcestring.
            if Length (definedDomain) > 0 then begin
              if lastcomment <> '' then
                lastcomment := lastcomment + sLinebreak;
              lastcomment := lastcomment + 'Programmer''s name for it: ' + constident;
              AddTranslation (definedDomain, msgid, lastcomment, MakePathLinuxRelative(sourcefilename)+':'+IntToStr(linenr));
              lastcomment := '';
            end;
          end;

          // Check what comes next in the line
          if copy(line, 1, 1) <> ';' then begin
            // First parameter is line number, second is the contents of the line
            if resourcestringmode=2 then
              Warning (wtSyntaxError,_('resourcestring does not end in semicolon.'));
            line:='';
            break;
          end else begin
            // If it ended with a semicolon, analyze the rest of the line as if it was a new line
            delete(line, 1, 1);
          end;
        end;
      end else begin
        // Check for occurence of gettext()
        while true do begin
          uline:=uppercase(line);
          p:=poscode('_',uline);
          p2:=poscode('GETTEXT', uline);
          if p=0 then begin
            p:=p2;
          end else
            if p2<>0 then
              p:=min(p,p2);
          if p=0 then
            break;
          if (poscode('FUNCTION',uline)<>0) or
             (poscode('PROCEDURE',uline)<>0) then
            break;

          domain := defaultDomain;
          idoffset:=0;
          if copy(uline,p,1)='_' then begin
            idlength:=1;
            idplural:=False;
          end else begin
            idlength:=7;
            if uppercase(copy(line, p - 1, 1)) = 'D' then begin
              domain := '';
              idlength:=8;
              idoffset:=-1;
            end;
            if uppercase(copy(line, p - 2, 2)) = 'DC' then begin
              domain := '';
              idlength:=9;
              idoffset:=-2;
            end;
            idplural:=False;
            if uppercase(copy(line, p - 2, 2)) = 'DN' then begin
              domain := '';
              idlength:=9;
              idoffset:=-2;
              idplural:=True;
            end else
            if uppercase(copy(line, p - 1, 1)) = 'N' then begin
              idlength:=8;
              idoffset:=-1;
              idplural:=True;
            end;
          end;
          if ((p+idoffset=1) or (not ((ord(uline[p+idoffset-1])<=255) and (char(ord(uline[p+idoffset-1])) in ['a'..'z','A'..'Z','_','0'..'9'])))) and
              (length(line)>=p+idlength+idoffset) and (not ((ord(uline[p+idoffset+idlength])<=255) and (char(ord(uline[p+idoffset+idlength])) in ['a'..'z','A'..'Z','_','0'..'9']))) then begin
            line := trim(copy(line, p + idlength+idoffset, maxint));
            if copy(line, 1, 1) = '(' then begin
              line := trim(copy(line, 2, maxint));
              if domain = '' then begin
                // get first parameter
                extractstring(line, domain);
                line := trim(line);
                if copy(line, 1, 1) = ',' then begin
                  delete(line, 1, 1);
                  line:=trim(line);
                end else begin
                  // First parameter is line number, second is line contents
                  Warning (wtSyntaxError,_('Missing comma after first parameter'));
                end;
              end;

              // Get parameter that contains the msgid
              msgid := RemoveNuls(readstring(line, src));
              if idplural then begin
                line := trim(line);
                if copy(line, 1, 1) = ',' then begin
                  delete(line, 1, 1);
                  line:=trim(line);
                end else begin
                  Warning (wtSyntaxError,_('Missing comma after first parameter'));
                end;
                if line='' then
                  dxreadln(src, line);
                msgid := msgid+#0+RemoveNuls(readstring(line,src));
              end;
              AddTranslation(domain, msgid, lastcomment, MakePathLinuxRelative(sourcefilename) + ':' + IntToStr(linenr));
              lastcomment := '';
            end { if a parenthesis is found };
          end else begin
            line := trim(copy(line, p + idlength+idoffset, maxint));
          end { if it looks like a function call identifier };
        end { loop that finds several occurences in the same line };
      end { if resourcestringmode };
    end;
  finally
    CloseFile(src);
  end;

  If length (definedDomain) > 0 then begin
    Warning (wtExtendedDirectiveError, _('$gnugettext: end directive is missing !'));
  end;
end;

constructor TXGetText.Create;
begin
  inherited Create;
  ignorelist:=TPoEntryList.Create;
  CFiles:=TStringList.Create;
  CFiles.Duplicates:=dupError;
  CFiles.CaseSensitive:=True;
  CFiles.Sorted:=True;
  BaseDirectoryList:=TStringList.create;
  filemasks:=TStringList.Create;
  filemasks.Sorted:=True;
  filemasks.Duplicates:=dupIgnore;
  filemasks.CaseSensitive:=True;
  domainlist := TStringList.Create;
  domainlist.Sorted := True;
  domainlist.Duplicates:=dupError;
  domainlist.CaseSensitive:=True;
  constlist:=TStringList.Create;
  constlist.Sorted:=True;
  constlist.Duplicates:=dupError;
  constlist.CaseSensitive:=True;
  Excludes := TXExcludes.Create;
  defaultDomain:='default';
end;

destructor TXGetText.Destroy;
begin
  ClearConstList;
  FreeAndNil (constlist);
  while domainlist.Count <> 0 do begin
    domainlist.Objects[0].Free;
    domainlist.Delete(0);
  end;
  FreeAndNil(domainlist);
  FreeAndNil (BaseDirectoryList);
  FreeAndNil (filemasks);
  FreeAndNil (CFiles);
  FreeAndNil (ignorelist);
  FreeAndNil(Excludes);
  inherited;
end;

procedure TXGetText.ExtractFromDFM(sourcefilename: widestring);
var
  src: TStream;
  mem: TMemoryStream;
  line, lastline:widestring;
  s: widestring;
  i:integer;
  indent: integer;
  comment: widestring;
  p, linenr: integer;
  scope: TStringList;
  propertyname: widestring;
  multilinevalue: boolean;
  mvalue: widestring;
  p1, p2, p3: integer;
  pClassname: integer;
  c:char;
  classnamepart: widestring;
  linechar:string;
  currentclassname: widestring;
  classnames: TStringList;
  instancenames: TStringList;
  excludeclass:boolean;
  excludeinstance:boolean;
  collectionlevel:integer; // will be increased which each occurence of a collection, in order to recognize nested collections
  collectionpropertyname:widestring; // will be the propertyname of the highest-level collection property

  procedure AddEntry(aValue:widestring);
  var
    propname:widestring;
  begin
    if collectionlevel > 0 then
      propname := collectionpropertyname
    else
      propname := propertyname;
    if not excludeclass and not excludeinstance and not Excludes.ExcludeFormClassProperty(classnames[indent], propname) then begin
      comment := scope2comment(scope, propertyname);
      AddTranslation(defaultDomain, RemoveNuls(aValue), comment, MakePathLinuxRelative(sourcefilename) + ':' + IntToStr(linenr));
    end;
  end;

begin
  src:=TFileStream.Create(sourcefilename,fmOpenRead);
  try
    // Check for empty file
    if src.Read(c,1)=0 then
      exit;
    // Check for binary dfm file
    src.Seek(0, soFromBeginning);
    if c=#$FF then begin
      // Convert the file into text form in a memory stream
      mem:=TMemoryStream.Create;
      ObjectResourceToText(src,mem);
      FreeAndNil (src);
      src:=mem;
    end;
    src.Seek(0,soFrombeginning);

    scope := TStringList.Create;
    classnames := TStringlist.Create;
    instancenames := TStringlist.Create;
    try
      classnames.Add(''); // we need that one because "indent" might start with 0
      instancenames.Add('');
      linenr := 0;
      line := '';
      propertyname := '';
      collectionpropertyname := '';
      multilinevalue := false;
      collectionlevel := 0;
      while true do begin
        // Get next line and check it out
        lastline := line;
        if not StreamReadln (src, line) then break;
        inc(linenr);
        indent := measureindent(line);
        line := trim(line);
        if line='' then continue;  // *** ABORT IF LINE IS EMPTY ***

        // Check if a collection starts or ends in this line.
        // If we have nested collections, the nesting-level
        // will be remembered                                                    
        if RightStr(line, 3) = '= <' then
          inc(collectionlevel);
        if RightStr(lowercase(line), 4) = 'end>' then begin
          dec(collectionlevel);
          if collectionlevel = 0 then
            collectionpropertyname := '';
        end;

        // Always adjust the count of "classnames" to the current "indent"
        // and make sure, the a bigger indent gets the same classname as the
        // smaller indent before. This will be overwritten as soon as we reach
        // an line containing "object", "inherited" or "inline", like this:
        //
        // object Form1: TForm      indent = 0, classname[0] = 'TForm'
        //  Caption = 'Form1'       indent = 1, classname[1] = 'TForm'
        //  object Edit1: TEdit     indent = 1, classname[1] = 'TEdit'
        //   Left = 1;              indent = 2, classname[2] = 'TEdit'
        while indent < classnames.Count-1 do begin
          classnames.Delete(classnames.Count-1);
          instancenames.Delete(instancenames.Count-1);
        end;
        while indent > classnames.Count-1 do begin
          classnames.Add(classnames[classnames.Count-1]);
          instancenames.Add(instancenames[instancenames.Count-1]);
        end;

        // check for occurence of a classname and remember it at the current indention.
        // Take into account that some properties might contain identifiers as part
        // of their name, e.g. "InlineSkaterCount" or "InheritedFromGrandPa"
        if (Pos(':', line) > 0) and ((Pos('object ', lowercase(line)) > 0) or (Pos('inherited ', lowercase(line)) > 0) or (Pos('inline ', lowercase(line)) > 0)) then begin
          pClassname := Pos(':', line);
          if pClassname > 0 then begin
            currentclassname := '';
            classnamepart := Trim(Copy(line, pClassname+1, Length(line)-pClassname+1));
            for i := 1 to Length(classnamepart) do begin
              // be aware of constructs like "TScrollbox [0]" or other unlikely things, simply just get only the chars that are valid for classnames
              linechar := UTF8Encode(classnamepart[i]);
              if (Length(linechar) > 1) or (not (linechar[1] in ['a'..'z','A'..'Z','_','0'..'9'])) then
                break
              else
                currentclassname := currentclassname + classnamepart[i];
            end;
            classnames[indent] := currentclassname;
            // remember the name of instance of that class as well in the same way
            p := Pos(' ', line);
            instancenames[indent] := Copy(line, p +1, pClassname -p -1);
          end;
        end;

        // check if the whole class shall be excluded
        excludeclass := Excludes.ExcludeFormClass(classnames[indent]);
        excludeinstance := false;
        if not excludeclass then begin
          for i := indent downto 0 do // check parent classes if they contain a wildcard
            if Excludes.FormClassHasWildcard(classnames[i]) then begin
              excludeclass := true;
              break;
            end;
          if not excludeclass then begin
            excludeinstance := Excludes.ExcludeFormInstance(sourcefilename, instancenames[indent]);
            if not excludeinstance then begin
              for i := indent downto 0  do
                if Excludes.ExcludeFormInstance(sourcefilename, instancenames[i]) then begin
                  excludeinstance := true;
                  break;
                end;
            end;
          end;
        end;

        // Check for changes in scope
        if (indent < scope.Count) and multilinevalue then begin
          multilinevalue := false;
          AddEntry(mvalue);
          scope.Delete(scope.count - 1);
        end;
        while indent < scope.Count do begin
          scope.Delete(scope.count - 1);
        end;

        if indent > scope.Count then begin
          p := pos(' ', lastline);
          if p = 0 then s := lastline else s := copy(lastline, p + 1, maxint);
          p := pos(':', s);
          multilinevalue := true;
          mvalue := '';
          if p = 0 then s := '' else s := copy(s, 1, p - 1);
        end;
        while indent > scope.Count do begin
          scope.Add(s);
          s := '';
        end;

        // Analyze the line
        p := pos(' =', line);
        p1 := pos('''', line);
        p2 := pos('#', line);
        if p1 = 0 then p1 := maxint;
        if p2 = 0 then p2 := maxint;
        p3 := min(p1, p2);

        // Extract property name if the line contains such one
        if (p <> 0) and (p < p3) then begin
          propertyname := trim(copy(line, 1, p - 1));
          // is we're in a collection (and it's the highest level if there are nested collections), remember the property name of that collection
          if (collectionlevel = 1) and (collectionpropertyname = '') then
            collectionpropertyname := propertyname;
          multilinevalue := false;
        end;

        // Extract string, if the line contains such one
        if p3 <> maxint then begin
          delete(line, 1, p3 - 1);
          extractstring(line, s);
          if multilinevalue then begin
            mvalue := mvalue + s;
            if trim(line) <> '+' then begin
              AddEntry(mvalue);
              mvalue:='';
            end;
          end else begin
            AddEntry(s);
          end;
        end;
      end;
    finally
      FreeAndNil(scope);
      FreeAndNil(classnames);
    end;
  finally
    FreeAndNil (src);
  end;
end;

procedure TXGetText.AddTranslation(domain, msgid: widestring; Comments,
  Location: widestring);
// Adds something to translate to the list
var
  it: TPoEntry;
  i, e: integer;
  sl: TStringList;
  dom:TXGTDomain;
  lookupvalue:ansistring;
begin
  // Check, that all parts of msgid are nonempty, if there are multiple parts
  if msgid<>'' then begin
    for i:=1 to length(msgid)+1 do begin
      if copy(#0+msgid+#0,i,2)=#0#0 then
        raise Exception.Create('Illegal msgid_plural value: It contained empty strings.');
    end;
  end;

  // Check for non-ascii characters
  if not AllowNonAscii then begin
    for i:=1 to length(msgid) do begin
      if ord(msgid[i])>=128 then begin
        Warning (wtNonAscii,format(_('msgid contains non-ascii characters: "%s"'),[msgid]));
        // Don't add an invalid msgid
        exit;
      end;
    end;
  end;

  // Remove any Carriage Returns
  while true do begin
    i:=pos(#13,msgid);
    if i=0 then break;
    delete (msgid,i,1);
  end;

  // Don't add empty strings
  if msgid = '' then exit;

  // Don't add numbers
  val(msgid, i, e);
  if (e = 0) and (msgid = IntToStr(i)) then exit;

  dom:=GetDomain(domain);
  sl:=TStringList.Create;
  try
    sl.Text := utf8encode(msgid);
    if sl.Count=0 then
      lookupvalue:='Weird, but happens if the string contains weird ascii chars'
    else
      lookupvalue:=sl.Strings[0];
  finally
    FreeAndNil(sl);
  end;
  it:=nil;
  if dom.msgid.Find(lookupvalue,i) then begin
    // Scroll back to the first in the list that has the same
    // first line in msgid
    while (i > 0) and (dom.msgid.Strings[i - 1] = lookupvalue) do
      dec(i);
    // Now loop through all those in the list it may be
    while true do begin
      it := dom.msgid.Objects[i] as TPoEntry;
      // Check if we found the correct one
      if it.msgid = msgid then break;
      // Check if we have scrolled past the last one
      if (i = dom.msgid.Count - 1) or (dom.msgid.Strings[i+1] <> lookupvalue) then begin
        it := nil;
        break;
      end;
      inc(i);
    end;
  end;
  if it = nil then begin
    it := TPoEntry.Create;
    dom.msgid.AddObject(lookupvalue, it);
    it.msgid := msgid;
    dom.order.AddObject(lookupvalue, it);
  end;
  if comments<>'' then begin
    sl:=TStringList.Create;
    try
      sl.Text:=utf8encode(comments);
      for i:=0 to sl.Count-1 do begin
        it.AutoCommentList.Add('#. '+sl.Strings[i]);
      end;
    finally
      FreeAndNil (sl);
    end;
  end;

  it.AutoCommentList.Add('#: '+RemoveFilenameSpaces(utf8encode(Location)));
end;

procedure TXGetText.WriteAll(Destinationpath, domain: widestring);
// Outputs a .po file
var
  destination: TFileStream;
  i: integer;
  item: TPoEntry;
  dom:TXGTDomain;
  filename: widestring;
  orderlist:TStrings;
  overwrite: boolean;
begin
  dom:=GetDomain(domain);
  if SingleOutputFilename<>'' then begin
    if domain=defaultDomain then begin
      filename:=SingleOutputFilename;
    end else begin
      exit;
    end;
  end else begin
    filename := destinationpath + domain + '.po';
  end;

  // Check for overwriting. Call custom handler if present, and abort if overwriting is not permitted.
  if FileExists (fileName) then begin
    overwrite := True;
    if assigned (OnOverwrite) then OnOverwrite (self, fileName, overwrite);
    if not overwrite then begin
      OnProgress (format (_('Overwrite %s aborted.'), [fileName]), filename, 0);
      Exit;
    end;
  end;

  // %s will be replaced by the filename
  if Assigned(OnProgress) then
    OnProgress (Format(_('Writing %s'),[filename]),filename,0);
  destination:=TFileSTream.Create (filename, fmCreate);
  try
    // Write a dummy header that the user can modify
    StreamWriteDefaultPoTemplateHeader(destination,Format(_('dxgettext %s'),[version]));

    // Write out all msgids
    if OrderbyMsgid then orderlist:=dom.msgid
                    else orderlist:=dom.order;
    for i := 0 to orderlist.Count - 1 do begin
      item := orderlist.Objects[i] as TPoEntry;
      item.WriteToStream(destination);
    end;
  finally
    FreeAndNil (destination);
  end;
end;

procedure TXGetText.ExtractFromFile(sourcefilename: widestring);
var
  ext:widestring;
begin
  CurrentFilename:=sourcefilename;
  linenr:=0;
  if ExpandFileName(sourcefilename)<>sourcefilename then
    sourcefilename:=BaseDirectory+SourceFilename;
  if Excludes.ExcludeDirectory(ExtractFilePath(sourcefilename)) or Excludes.ExcludeFormFile(sourcefilename) then
    Exit;
  try
    ext:=uppercase(ExtractFileExt(sourcefilename));
    if (ext='.C') or (ext='.CPP') then
      CFiles.Add(sourcefilename)
    else begin
      if Assigned(OnProgress) then
        OnProgress (Format(_('Reading %s'),[sourcefilename]),sourcefilename,0);
      if (ext='.DFM') or (ext='.XFM') then
        ExtractFromDFM(sourcefilename)
      else
      if ext='.RC' then
        ExtractFromRC(sourcefilename)
      else
{$ifdef mswindows}
      if (ext='.DLL') or (ext='.EXE') or (ext='.BPL') then
        ExtractFromEXE(sourcefilename)
      else
{$endif}
      if (ext='.PAS') or (ext='.DPR') or (ext='.INC') then
        ExtractFromPascal(sourcefilename)
      else begin
        Warning (wtParameterError,Format(_('WARNING: Unknown file extension %s. Reading file as being pascal source.'),[ext]));
        ExtractFromPascal(sourcefilename)
      end;
    end;
  except
    on e:EControlC do begin
      raise;
    end;
    on e:Exception do begin
      Warning (wtUnexpectedException,'Exception '+e.ClassName+sLineBreak+e.Message);
    end;
  end;
  CurrentFilename:='';
end;

procedure TXGetText.ExtractFromFileMasks(mask:widestring);
var
  sr: TSearchRec;
  more: boolean;
  curdir:widestring;
  dirlist:TStringList;
  sl:TStringList;
  i, idx:integer;
  maskcheck:TMask; // This is only necessary because of a bug in the Windows API FindFirst()
begin
  mask:=ExpandFileName(BaseDirectory+mask);
  dirlist:=TStringList.Create;
  try
    dirlist.Add(ExtractFilePath(mask));
    mask:=ExtractFileName(mask);

    if recurse then begin
      idx:=0;
      while idx<dirlist.count do begin
        curdir:=dirlist.Strings[idx];

        // Find all subdirectories
        more := FindFirst(curdir+'*', faAnyFile, sr) = 0;
        while more do begin
          if (sr.Attr and faDirectory<>0) and (sr.Name<>'.') and (sr.Name<>'..') then
            dirlist.Add(curdir+sr.Name+PathDelim);
          more := FindNext(sr) = 0;
        end;
        SysUtils.FindClose (sr);
        inc (idx);
      end;
    end;

    dirlist.Sort;

    for idx:=0 to dirlist.Count-1 do begin
      curdir:=dirlist.Strings[idx];

      maskcheck:=TMask.Create (mask);
      sl:=TStringList.Create;
      try
        // Extract from all files in current directory
        more := FindFirst(curdir+mask, faAnyFile-faDirectory, sr) = 0;
        while more do begin
          // The following if is only necessary, because several Windows versions
          // have a bug in FindFirst, that makes "test.cpp,v" match on the
          // file mask "*.cpp"
          if maskcheck.Matches(sr.Name) then
            sl.Add (curdir + sr.Name);
          more := FindNext(sr) = 0;
        end;
        SysUtils.FindClose(sr);
        sl.Sort;
        for i:=0 to sl.count-1 do
          ExtractFromFile(sl.Strings[i]);
      finally
        FreeAndNil (sl);
        FreeAndNil (maskcheck);
      end;
    end;
  finally
    FreeAndNil (dirlist);
  end;
end;

function TXGetText.GetDomain(domain: widestring): TXGTDomain;
var
  i: integer;
begin
  if domainlist.Find(domain, i) then begin
    Result := domainlist.Objects[i] as TXGTDomain;
  end else begin
    Result := TXGTDomain.Create;
    domainlist.AddObject(domain, Result);
  end;
end;

procedure TXGetText.dxreadln(var src: TextFile; var line: widestring);
var
  i:integer;
  procedure cutuntil (endtag:widestring);
  var p:integer;
  begin
    p:=i+length(endtag)-1;
    while p<=length(line) do begin
      if copy(line,p,length(endtag))=endtag then begin
        delete (line,i,p+length(endtag)-i);
        exit;
      end;
      inc (p);
    end;
    // At this place, the end tag was not found in the line
    line:=copy(line,1,i-1);
    commentmode:=endtag;
  end;
begin
  line:='';
  while (not eof(src)) and (line='') do begin
    if commentmode<>'' then begin
      while true do begin
        if eof(src) then begin
          line:='';
          exit;
        end;
        readln (src, line);
        line:=trim(line);
        LastLineRead:=line;
        inc (linenr);
        i:=pos(commentmode,line);
        if i<>0 then begin
          delete (line,1,i+length(commentmode)-1);
          commentmode:='';
          break;
        end;
      end;
    end else begin
      readln (src, line);
      line:=trim(line);
      LastLineRead:=line;
      inc (linenr);
      if trim(line)='' then
        lastcomment:='';
    end;
    i:=1;
    while i<=length(line) do begin
      if copy(line,i,1)='''' then begin
        // A string was detected - find the end of it.
        inc (i);
        while true do begin
          if copy(line,i,1)='''' then begin
            inc (i);
            break;
          end;
          // If the string doesn't end until the line is over, finish the procedure
          if i>=length(line) then
            exit;
          inc (i);
        end;
      end else
      if copy(line,i,2)='//' then begin
        // The rest of the line is a comment
        if lastcomment<>'' then
          lastcomment:=lastcomment+sLineBreak;
        lastcomment:=trim(copy(line,i+2,maxint));
        line:=copy(line,1,i-1);
        exit;
      end else
      if copy(line,i,1)='{' then begin
        if pos (cDefineDirective, lowercase(copy(line,1,length(cDefineDirective)))) = 1 then
          doHandleExtendedDirective (line);

        // Bracket comment
        cutuntil ('}');
      end else
      if copy(line,i,2)='(*' then begin
        // Bracket comment, Danish style
        cutuntil ('*)');
      end else
        inc (i);
    end;
    line:=trim(line);
  end;
end;

{ TXGTDomain }

constructor TXGTDomain.Create;
begin
  msgid:=TStringList.Create;
  order:=TStringList.Create;
  msgid.Sorted:=True;
  msgid.Duplicates:=dupAccept;
  msgid.CaseSensitive:=True;
end;

destructor TXGTDomain.Destroy;
begin
  while msgid.count<>0 do begin
    msgid.Objects[0].Free;
    msgid.Delete (0);
  end;
  FreeAndNil (msgid);
  FreeAndNil (order);
  inherited;
end;

procedure TXGetText.WritePoFiles (DestinationPath:widestring);
var
  i:integer;
begin
  for i:=0 to domainlist.Count-1 do begin
    // Write all domain.po files
    WriteAll(DestinationPath,domainlist.Strings[i]);
  end;
end;

procedure TXGetText.ClearConstList;
begin
  while constlist.Count<>0 do begin
    constlist.Objects[0].Free;
    constlist.Delete (0);
  end;
end;

procedure TXGetText.Warning(WarningType:TWarningType;msg: widestring);
begin
  if Assigned(OnWarning) then
    OnWarning (WarningType,msg,LastLineRead,CurrentFilename,linenr);
end;

procedure TXGetText.Warning(WarningType: TWarningType; Msg,
  Line: widestring; Filename:widestring;LineNumber: Integer);
begin
  if Assigned(OnWarning) then
    OnWarning (WarningType,msg,Line,Filename,linenumber);
end;

procedure TXGetText.ExtractFromRC(sourcefilename: widestring);
var
  tf:TextFile;
  line:widestring;
  p, i:integer;
  ident:widestring;
begin
  // Currently, this scanner is based on the RC file that was included
  // with DBISAM version 3. It may not work with other RC files, but
  // if you find an RC file that it does not work with, please send that
  // RC file to Lars@dybdahl.dk
  FileMode:=fmOpenRead;
  AssignFile (tf,sourcefilename);
  Reset (tf);
  try
    linenr:=0;
    while not eof(tf) do begin
      // Get next line
      readln (tf,line);
      inc (linenr);
      line:=trim(line);
      LastLineRead:=line;

      if copy(line,1,1)<>'#' then begin
        p:=pos('"',line);
        if p<>0 then begin
          // Find identifier in the beginning of the line
          ident:=trim(copy(line,1,p-1));
          if copy(ident,length(ident),1)=',' then
            delete (ident,length(ident),1);
          if ident<>'' then
            ident:='Programmer''s name: '+ident;

          // Find the msgid
          delete (line,1,p);
          i:=1;
          while i<=length(line) do begin
            if copy(line,i,2)='\n' then begin
              delete (line,i,1);
              line[i]:=#10;
            end else
            if copy(line,i,2)='\r' then begin
              delete (line,i,1);
              line[i]:=#13;
            end else
            if copy(line,i,2)='\t' then begin
              delete (line,i,1);
              line[i]:=#9;
            end else
            if copy(line,i,2)='\f' then begin
              delete (line,i,1);
              line[i]:=#26;
            end else
            if line[i]='\' then begin
              case line[i+1] of
                'n': line[i+1] := #10;
                'r': line[i+1] := #13;
                't': line[i+1] := #9;
              end;
              delete (line,i,1);
            end else
            if line[i]='"' then begin
              delete (line,i,maxint);
            end;
              inc (i);
          end;
          AddTranslation(defaultDomain,RemoveNuls(line),ident,MakePathLinuxRelative(sourcefilename) + ':' + IntToStr(linenr));
        end;
      end;
    end;
  finally
    CloseFile (tf);
  end;
end;

procedure TXGetText.AddBaseDirectory(path: string);
begin
  if path<>'' then
    BaseDirectoryList.Add(IncludeTrailingPathDelimiter(path))
  else
    BaseDirectoryList.Add('');
end;

procedure TXGetText.Execute;
var
  i,j:integer;
begin

  // If no base directories, make one
  if BaseDirectoryList.Count=0 then
    AddBaseDirectory(IncludeTrailingPathDelimiter(ExpandFileName('.')));

  // Find destination path
  // does not work on Windows 7 ==> see changes in uWork
  if DestinationPath='' then
    DestinationPath:=IncludeTrailingPathDelimiter(ExpandFileName('.'));

  // Read current ignore.po file
  if FileExists(DestinationPath+'ignore.po') then
    ignorelist.LoadFromFile(DestinationPath+'ignore.po');

  // Iterate base directories
  for j:=0 to BaseDirectoryList.Count-1 do begin
    BaseDirectory:=BaseDirectoryList.Strings[j];
    ParseExcludeFile;
    for i:=0 to filemasks.count-1 do begin
      if NoWildcards then begin
        ExtractFromFile(filemasks.Strings[i]);
      end else begin
        ExtractFromFileMasks(filemasks.Strings[i]);
      end;
    end;
  end;

  // Handle ignores
  HandleIgnores;

  // Write files
  if UpdateIgnore then
    ignorelist.SaveToFile(DestinationPath+'ignore.po');
  WritePoFiles (DestinationPath);
end;

procedure TXGetText.AddDelphiFilemasks;
begin
  filemasks.add ('*.pas');
  filemasks.add ('*.inc');
  filemasks.Add ('*.rc');
  filemasks.add ('*.dpr');
  filemasks.add ('*.xfm');
  filemasks.add ('*.dfm');
end;

procedure TXGetText.AddKylixFilemasks;
begin
  filemasks.add ('*.pas');
  filemasks.add ('*.inc');
  filemasks.Add ('*.rc');
  filemasks.add ('*.dpr');
  filemasks.add ('*.xfm');
end;


procedure TXGetText.doHandleExtendedDirective(line: wideString);
Const
  cErrOptionUnknown = '{gnugettext: Unknonw option.';
  cErrMissingStart = '{gnugettext: reset found without scan-all.';
  cErrDomainSyntax = '{gnugettext: error in the domain name definition.';
Var
  i : integer;
  tmp : string;
begin
  delete (line, 1, length(cDefineDirective));
  line := trim (line);
  if IsDirective(cScanOption, line) then begin
    delete (line, 1, length (cScanOption));
    line := trim (line);
    if pos (cDomainDefinition, lowerCase (copy (line, 1, length(cDomainDefinition)))) = 1 then begin
      delete (line, 1, Length (cDomainDefinition));
      line := trim (line);
      if (length (line) > 0) and (line[1] = '=') then begin
        delete (line, 1, 1);
        line := trim (line);
        if (length (line) > 0) and (line[1] = '''') then begin
          delete (line, 1, 1);
          i := 1;
          tmp := '';
          while (i <= length (line)) and (line[i] <> '}') do begin
            if (line[i] = '''') then begin
              if (i = length (line)) or (line[i+1] <> '''') then begin
                definedDomain := tmp;
                break;
              end
              else inc (i);
            end;
            tmp := tmp + line[i];
            inc (i);
          end;
        end;
      end;

      if length (definedDomain) = 0 then begin
        Warning (wtExtendedDirectiveError, _(cErrDomainSyntax));
      end;
    end
    else definedDomain := defaultDomain;
  end
  else if IsDirective(cScanResetOption, line) then begin
    if length (definedDomain) = 0 then Warning(wtExtendedDirectiveError, _(cErrMissingStart))
    else definedDomain := ''
  end
  else begin
    Warning (wtExtendedDirectiveError, _(cErrOptionUnknown))
  end;
end;

{$ifdef mswindows}
procedure TXGetText.ExtractFromEXE(sourcefilename: widestring);
  procedure recurse (rl:TResourceList);
  var
    r:TResourceItem;
    i,j:integer;
    ws:widestring;
    itemno:integer;
  begin
    for i:=0 to rl.Count-1 do begin
      r:=rl.Items[i];
      if r.IsList then begin
        recurse (r.List)
      end else begin
        case r.ResType of
          rtString:
            begin
              itemno:=0;
              ws:=PWideChar(r.RawData);
              while ws<>'' do begin
                inc (itemno);
                j:=ord(ws[1]);
                AddTranslation(defaultDomain,RemoveNuls(copy(ws,2,j)),'Resource '+r.Name+', item no. '+IntToStr(itemno),MakePathLinuxRelative(sourcefilename)+':'+IntToStr(r.Offset));
                delete (ws,1,j+1);
              end;
            end;
        end;
      end;
    end;
  end;
var
  exe:TExeImage;
begin
  exe := TExeImage.CreateImage(nil, sourceFileName);
  try
    recurse (exe.Resources);
  finally
    FreeAndNil (exe);
  end;
end;
{$endif}

procedure TXGetText.HandleIgnores;
var
  j:integer;
  dom:TXGTDomain;
  item:TPoEntry;
  newitem:TPoEntry;
  ignoreitem:TPoEntry;
begin
  // Only default domain is affected by ignore.po
  dom:=GetDomain(defaultDomain);

  // Add new ignores to new ignore list and update autocomments
  if UpdateIgnore then begin
    for j := 0 to dom.order.Count-1 do begin
      item := dom.order.Objects[j] as TPoEntry;
      ignoreitem:=ignorelist.Find(item.MsgId);
      if ignoreitem=nil then begin
        newitem:=TPoEntry.Create;
        newitem.Assign(item);
        if not IsProbablyTranslatable(newitem) then 
          ignorelist.Add(newitem)
        else
          FreeAndNil (newitem);
      end else begin
        ignoreitem.AutoCommentList.Text:=item.AutoCommentList.Text;
      end;
    end;
  end;

  // Remove ignores from default list
  if UseIgnoreFile then begin
    for j:=dom.order.Count-1 downto 0 do begin
      item:=dom.order.Objects[j] as TPoEntry;
      if ignorelist.Find(item.MsgId)<>nil then
        // Only delete from order list
        dom.order.Delete (j);
    end;
  end;
end;

procedure TXGetText.ParseExcludeFile;
const
 cExcludeFilename = 'ggexclude.cfg';
var
  excludefile: widestring;
  F: TextFile;
  section,
  line: widestring;
  lnr: integer;
  added:boolean;
begin
  lnr := 0;
  Excludes.Clear;
  Excludes.Basedirectory := BaseDirectory;
  excludefile :=BaseDirectory;
  if RightStr(excludefile, 1) <> PathDelim then
    excludefile := excludefile + PathDelim;
  excludefile := ExpandFilename(excludefile + cExcludeFilename);
  if not FileExists(excludefile) then
    Exit;
  section := '';
  FileMode:=fmOpenRead;
  AssignFile(F, excludefile);
  Reset(F);
  try
    if Assigned(OnProgress) then
      OnProgress (Format(_('Reading %s'),[excludefile]),excludefile,0);
    while not EOF(F) do begin
      Readln(F, line);
      line := Trim(line);
      inc(lnr);
      if line <> '' then begin // skip empty lines
        if line[1] = '#' then // skip remarks
          Continue;
        if line[1] = '[' then begin // entering new section
          if RightStr(line, 1) = ']' then begin
            section := LowerCase(Copy(line, 2, Length(line) - 2));
            if (section <> cExcludeDir)
              and (section <> cExcludeFile)
              and (section <> cExcludeFormClass)
              and (section <> cExcludeFormClassProperty)
              and (section <> cExcludeFormInstance) then
                Warning(wtExcludeFile, Format(_('Line %d: Unknown section'), [lnr]), section, excludefile, lnr);
             continue;
          end else
            Warning(wtExcludeFile, Format(_('Line %d: Looks like a section but has no closing square brackets'), [lnr]), line, excludefile, lnr);
        end;
        added := true;
        if section = cExcludeDir then
          added := Excludes.AddDirectory(line)
        else if section = cExcludeFile then
          added := Excludes.AddFormFile(line)
        else if section = cExcludeFormClass then
          added := Excludes.AddFormClass(line)
        else if section = cExcludeFormClassProperty then
          added := Excludes.AddFormClassProperty(line)
        else if section = cExcludeFormInstance then
          added := Excludes.AddFormInstance(line);
        if not added then
          Warning(wtExcludeFile, Format(_('Line %d: %s'), [lnr, Excludes.LastErrorMsg]), line, excludefile, lnr);
      end;
    end;
  finally
    CloseFile(F);
  end;
end;

{ TXExcludes }

function TXExcludes.AddDirectory(aDirectory: string): boolean;
begin
  Result := True;
  aDirectory := Trim(aDirectory);
  if aDirectory = '' then
    Exit;
  if BaseDirectory <> '' then begin
    aDirectory := GetFullInternalPath(aDirectory);
    if RightStr(aDirectory, 1) = PathDelim then
      aDirectory := Copy(aDirectory, 1, Length(aDirectory) -1);
    if DirectoryExists(aDirectory) then begin
      {$ifdef mswindows}
      FDirectories.Add(AnsiLowerCase(aDirectory));
      {$else}
      FFiles.Add(aDirectory);
      {$endif}
    end else begin
      Result := False;
      FLastErrorMsg := Format(_('Directory %s doesn''t exist'), [aDirectory]);
    end;
  end;
end;


function TXExcludes.AddFormClass(aClassname: string): boolean;
begin
  Result := True;
  if aClassname = '' then
    Exit;
  if Pos('.', aClassname) > 0 then begin
    Result := False;
    FLastErrorMsg := Format(_('Wrong section: %s is a property name and not a class'), [aClassname]);
  end else begin
    aClassname := Trim(Lowercase(aClassname));
    FFormClasses.Add(aClassname);
  end;
end;

function TXExcludes.AddFormClassProperty(aPropertyname: string): boolean;
var
  p:integer;
begin
  Result := True;
  p := Pos('.', aPropertyname);
  if p = 0 then begin
    Result := False;
    FLastErrorMsg := Format(_('Wrong section: %s seems to be a class and not a property name'), [aPropertyname]);
  end else 
    FExcludeFormClassPropertyList.AddFormClassProperty(aPropertyname);
end;

function TXExcludes.AddFormFile(aFilename: string): boolean;
var
  wildcardfilecount: integer;
begin
  Result := True;
  aFilename := Trim(aFilename);
  if aFilename = '' then
    Exit;
  if BaseDirectory <> '' then begin
    wildcardfilecount := 0;
    // if a wildcard is used, add all possible Delphi- or Kylix-files to the list
    if RightStr(aFilename, 2) = '.*' then begin
      aFilename := Copy(aFilename, 1, Length(aFilename) -2);
      if AddFormFile(aFilename + '.dpr') then
        inc(wildcardfilecount);
      if AddFormFile(aFilename + '.pas') then
        inc(wildcardfilecount);
      if AddFormFile(aFilename + '.dfm') then
        inc(wildcardfilecount);
      if AddFormFile(aFilename + '.xfm') then
        inc(wildcardfilecount);
      if AddFormFile(aFilename + '.inc') then
        inc(wildcardfilecount);
      if AddFormFile(aFilename + '.rc') then
        inc(wildcardfilecount);
      if wildcardfilecount = 0 then begin
        Result := False;
        FLastErrorMsg := Format(_('No file found for "%s.*"'), [aFilename]);
      end;
      Exit;
    end;

    aFilename := GetFullInternalPath(aFilename);
    if FileExists(aFilename) then begin
      {$ifdef mswindows}
      FFiles.Add(AnsiLowerCase(aFilename));
      {$else}
      FFiles.Add(aFilename);
      {$endif}
    end else begin
      Result := False;
      FLastErrorMsg := Format(_('File %s doesn''t exist'), [aFilename]);
    end;
  end;
end;

function TXExcludes.AddFormInstance(aInstanceName: string): boolean;
var
  filenamepart,
  instancenamepart: string;
  i: integer;
  p: integer;
begin
  Result := True;
  aInstanceName := Trim(aInstanceName);
  if aInstanceName = '' then
    Exit;

  // Search from the end of the line
  // Take into account that filenames might be absolute, containing
  // ':' on Windows; and that a file-ext might be there.
  p := 0;
  for i := Length(aInstanceName) downto 1 do begin
    if aInstanceName[i] = ':' then begin
      p := i;
      break;
    end;
  end;

  if p = 0 then begin
    Result := False;
    FLastErrorMsg := Format(_('Wrong syntax: No ":" found in %s'), [aInstanceName]);
    exit;
  end;

  if p = Length(aInstanceName) then begin
    Result := False;
    FLastErrorMsg := Format(_('Wrong syntax: ":" is at the end of the line of %s'), [aInstanceName]);
    exit;
  end;

  filenamepart := GetFullInternalPath(LeftStr(aInstanceName, p-1));
  if not FileExists(filenamepart) then begin
    Result := False;
    FLastErrorMsg := Format(_('File "%s" doesn''t exist'), [filenamepart]);
    exit;
  end;
  {$ifdef mswindows}
  filenamepart := AnsiLowerCase(filenamepart);
  {$endif}
  instancenamepart := RightStr(aInstancename, Length(aInstancename)-p);
  FFormInstances.Append(Format('%s:%s', [filenamepart,instancenamepart]));
end;

procedure TXExcludes.Clear;
begin
  FFiles.Clear;
  FDirectories.Clear;
  FFormClasses.Clear;
  FFormInstances.Clear;
  FExcludeFormClassPropertyList.Clear;
  FLastErrorMsg := '';
  FBaseDirectory := '';
end;

constructor TXExcludes.Create;
begin
  FExcludeFormClassPropertyList := TXExcludeFormClassPropertyList.Create(TXExcludeFormClassProperties);
  FLastErrorMsg := '';

  FDirectories := TStringList.Create;
  FDirectories.Duplicates := dupIgnore;
  FDirectories.Sorted := True;
  FDirectories.CaseSensitive := True;

  FFiles := TStringList.Create;
  FFiles.Duplicates := dupIgnore;
  FFiles.Sorted := True;
  FFiles.CaseSensitive := True;

  FFormClasses := TStringList.Create;
  FFormClasses.Sorted := True;
  FFormClasses.Duplicates := dupIgnore;
  FFormClasses.CaseSensitive := False;

  FFormInstances := TStringList.Create;
end;

destructor TXExcludes.Destroy;
begin
  FreeAndNil(FExcludeFormClassPropertyList);
  FreeAndNil(FFormInstances);
  FreeAndNil(FFormClasses);
  FreeAndNil(FFiles);
  FreeAndNil(FDirectories);
  inherited;
end;

function TXExcludes.ExcludeDirectory(aDirectory: string): Boolean;
var
 i: Integer;
begin
  Result := False;
  if (Trim(aDirectory) = '') or (FDirectories.Count = 0) then
    Exit;
  if RightStr(aDirectory, 1) = PathDelim then
    aDirectory := Copy(aDirectory, 1, Length(aDirectory) -1);
  aDirectory := WindowsPathDelim2LinuxPathDelim(aDirectory);
  {$ifdef mswindows}
  aDirectory := AnsiLowerCase(aDirectory);
  {$endif}
  for i := 0 to FDirectories.Count-1 do begin
    // this checks for subfolders in FDirectories[i] as well:
    if Pos(FDirectories[i], aDirectory) = 1 then begin
      Result := True;
      Exit;
    end;
  end;
end;

function TXExcludes.ExcludeFormClass(aClassname: string): Boolean;
var
  i:integer;
  s:string;
begin
  Result := False;
  if (aClassname = '') or (FFormClasses.Count = 0) then
    Exit;
  aClassname := Trim(LowerCase(aClassname));
  for i := 0 to FFormClasses.Count-1 do begin
    s := FFormClasses[i];
    if RightStr(s, 1) = '*' then
      s := LeftStr(s, Length(s)-1);
    if s = aClassname then begin
      Result := true;
      exit;
    end;
  end;
end;

function TXExcludes.ExcludeFormClassProperty(aClassname,
  aPropertyname: string): Boolean;
begin
  Result := FExcludeFormClassPropertyList.ExcludeFormClassProperty(aClassname, aPropertyname)
end;

function TXExcludes.ExcludeFormClassProperty(aClassname: string): Boolean;
begin
  Result := Assigned(FExcludeFormClassPropertyList.FindItem(aClassname));
end;

function TXExcludes.ExcludeFormFile(aFilename: string): Boolean;
begin
  Result := False;
  if (aFilename = '') or (FFiles.Count = 0) then
    Exit;
  aFilename := WindowsPathDelim2LinuxPathDelim(aFilename);
  {$ifdef mswindows}
  aFilename := AnsiLowerCase(aFilename);
  {$endif}
  Result := FFiles.IndexOf(aFilename) > -1;
end;

function TXExcludes.ExcludeFormInstance(aFilename, aInstanceName: string): boolean;
var
  i,p: integer;
  filenamepart,
  instancenamepart: string;
begin
  Result := False;
  aFilename := WindowsPathDelim2LinuxPathDelim(aFilename);
  aInstanceName := Trim(aInstanceName);
  if (aInstanceName = '') or (aFilename = '') or (FFormInstances.Count = 0) then
    Exit;
  aInstanceName := LowerCase(aInstanceName);
  p := 0;
  for i := 0 to FFormInstances.Count-1 do begin
    if lowercase(RightStr(FFormInstances[i], Length(aInstancename))) = aInstancename then
      p := Length(aInstancename) -1
    else
      continue;
    if p > 0 then begin
      filenamepart := LeftStr(FFormInstances[i], Length(FFormInstances[i])-p-2);
      instancenamepart := lowercase(RightStr(FFormInstances[i], p+1));
      {$ifdef mswindows}
      if (AnsiLowercase(filenamepart) = AnsiLowercase(aFilename))
      {$else}
      if (filenamepart = aFilename)
      {$endif}
      and (instancenamepart = aInstanceName) then begin
        Result := true;
        exit;
      end;
    end;
  end;
end;

function TXExcludes.FormClassHasWildcard(aClassname: string): Boolean;
var
  i:integer;
begin
  Result := False;
  aClassname := Trim(LowerCase(aClassname));
  for i := 0 to FFormClasses.Count-1 do begin
    if RightStr(FFormClasses[i], 1) = '*' then begin
      if LeftStr(FFormClasses[i], Length(FFormClasses[i])-1) = aClassname then begin
        Result := true;
        exit;
      end;
    end;
  end;
end;

function TXExcludes.GetFullInternalPath(s:string): string;
begin
  Result := Trim(WindowsPathDelim2LinuxPathDelim(ExpandFilename(Basedirectory+s)));
end;

{ TXExcludeFormClassProperties }

procedure TXExcludeFormClassProperties.AddFormClassProperty(aPropertyname: string);
begin
  FProperties.Add(Trim(Lowercase(aPropertyname)));
end;

constructor TXExcludeFormClassProperties.Create(Collection: TCollection);
begin
  inherited;
  FProperties := TStringList.Create;
  FProperties.Duplicates := dupIgnore;
  FProperties.Sorted := True;
  FProperties.CaseSensitive := False;
end;

destructor TXExcludeFormClassProperties.Destroy;
begin
  FreeAndNil(FProperties);
  inherited;
end;

function TXExcludeFormClassProperties.ExcludeFormClassProperty(
  aPropertyname: string): boolean;
begin
  Result := FProperties.IndexOf(Trim(LowerCase(aPropertyname))) > -1;
end;

procedure TXExcludeFormClassProperties.SetNameOfClass(const Value: string);
begin
  FNameOfClass := Trim(Lowercase(Value));
end;

{ TXExcludeFormClassPropertyList }

function TXExcludeFormClassPropertyList.Add: TXExcludeFormClassProperties;
begin
  Result := TXExcludeFormClassProperties(inherited Add);
end;

function TXExcludeFormClassPropertyList.AddFormClass(aClassname: string): TXExcludeFormClassProperties;
begin
  Result := FindItem(aClassname);
  if not assigned(Result) then begin
    Result := Add;
    Result.NameOfClass := aClassname;
  end;
end;

function TXExcludeFormClassPropertyList.AddFormClassProperty(
  aClassPropertyname: string): TXExcludeFormClassProperties;
var
  p: integer;
  theclassname,
  thepropertyname: string;
  item: TXExcludeFormClassProperties;
begin
  Result := nil;
  if aClassPropertyname = '' then
    Exit;
  p := Pos('.', aClassPropertyname);
  aClassPropertyname := Trim(aClassPropertyname);
  theclassname := Trim(LeftStr(aClassPropertyname, p-1));
  thepropertyname := Trim(RightStr(aClassPropertyname, Length(aClassPropertyname) - p));
  item := AddFormClass(theclassname);
  assert(assigned(item), 'This should''t happen: item of a class was neither created nor found');
  item.AddFormClassProperty(thepropertyname);
end;

function TXExcludeFormClassPropertyList.ExcludeFormClassProperty(
  aClassname, aPropertyname: string): Boolean;
var
  item: TXExcludeFormClassProperties;
begin
  Result := False;
  if Count > 0 then begin
    item := FindItem(aClassname);
    if assigned(item) then
      Result := item.ExcludeFormClassProperty(aPropertyname);
  end;
end;

function TXExcludeFormClassPropertyList.FindItem(
  aClassname: string): TXExcludeFormClassProperties;
var
  i:integer;
begin
  Result := nil;
  if Count > 0 then begin
    aClassname := Trim(Lowercase(aClassname));
    for i := 0 to Count-1 do
      if Items[i].NameOfClass = aClassname then begin
        Result := Items[i];
        exit;
      end;
  end;
end;

function TXExcludeFormClassPropertyList.GetItems(
  Index: integer): TXExcludeFormClassProperties;
begin
  Result := TXExcludeFormClassProperties(inherited Items[Index]);
end;

procedure TXExcludeFormClassPropertyList.SetItem(Index: integer;
  const Value: TXExcludeFormClassProperties);
begin
  inherited SetItem(Index, Value);
end;


end.
